@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.route53.kotlin

import com.pulumi.aws.route53.ZoneArgs.builder
import com.pulumi.aws.route53.kotlin.inputs.ZoneVpcArgs
import com.pulumi.aws.route53.kotlin.inputs.ZoneVpcArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Manages a Route53 Hosted Zone. For managing Domain Name System Security Extensions (DNSSEC), see the `aws.route53.KeySigningKey` and `aws.route53.HostedZoneDnsSec` resources.
 * ## Example Usage
 * ### Public Zone
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const primary = new aws.route53.Zone("primary", {name: "example.com"});
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * primary = aws.route53.Zone("primary", name="example.com")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var primary = new Aws.Route53.Zone("primary", new()
 *     {
 *         Name = "example.com",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/route53"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := route53.NewZone(ctx, "primary", &route53.ZoneArgs{
 * 			Name: pulumi.String("example.com"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.route53.Zone;
 * import com.pulumi.aws.route53.ZoneArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var primary = new Zone("primary", ZoneArgs.builder()
 *             .name("example.com")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   primary:
 *     type: aws:route53:Zone
 *     properties:
 *       name: example.com
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Public Subdomain Zone
 * For use in subdomains, note that you need to create a
 * `aws.route53.Record` of type `NS` as well as the subdomain
 * zone.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const main = new aws.route53.Zone("main", {name: "example.com"});
 * const dev = new aws.route53.Zone("dev", {
 *     name: "dev.example.com",
 *     tags: {
 *         Environment: "dev",
 *     },
 * });
 * const dev_ns = new aws.route53.Record("dev-ns", {
 *     zoneId: main.zoneId,
 *     name: "dev.example.com",
 *     type: aws.route53.RecordType.NS,
 *     ttl: 30,
 *     records: dev.nameServers,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * main = aws.route53.Zone("main", name="example.com")
 * dev = aws.route53.Zone("dev",
 *     name="dev.example.com",
 *     tags={
 *         "Environment": "dev",
 *     })
 * dev_ns = aws.route53.Record("dev-ns",
 *     zone_id=main.zone_id,
 *     name="dev.example.com",
 *     type=aws.route53.RecordType.NS,
 *     ttl=30,
 *     records=dev.name_servers)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var main = new Aws.Route53.Zone("main", new()
 *     {
 *         Name = "example.com",
 *     });
 *     var dev = new Aws.Route53.Zone("dev", new()
 *     {
 *         Name = "dev.example.com",
 *         Tags =
 *         {
 *             { "Environment", "dev" },
 *         },
 *     });
 *     var dev_ns = new Aws.Route53.Record("dev-ns", new()
 *     {
 *         ZoneId = main.ZoneId,
 *         Name = "dev.example.com",
 *         Type = Aws.Route53.RecordType.NS,
 *         Ttl = 30,
 *         Records = dev.NameServers,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/route53"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		main, err := route53.NewZone(ctx, "main", &route53.ZoneArgs{
 * 			Name: pulumi.String("example.com"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		dev, err := route53.NewZone(ctx, "dev", &route53.ZoneArgs{
 * 			Name: pulumi.String("dev.example.com"),
 * 			Tags: pulumi.StringMap{
 * 				"Environment": pulumi.String("dev"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = route53.NewRecord(ctx, "dev-ns", &route53.RecordArgs{
 * 			ZoneId:  main.ZoneId,
 * 			Name:    pulumi.String("dev.example.com"),
 * 			Type:    pulumi.String(route53.RecordTypeNS),
 * 			Ttl:     pulumi.Int(30),
 * 			Records: dev.NameServers,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.route53.Zone;
 * import com.pulumi.aws.route53.ZoneArgs;
 * import com.pulumi.aws.route53.Record;
 * import com.pulumi.aws.route53.RecordArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var main = new Zone("main", ZoneArgs.builder()
 *             .name("example.com")
 *             .build());
 *         var dev = new Zone("dev", ZoneArgs.builder()
 *             .name("dev.example.com")
 *             .tags(Map.of("Environment", "dev"))
 *             .build());
 *         var dev_ns = new Record("dev-ns", RecordArgs.builder()
 *             .zoneId(main.zoneId())
 *             .name("dev.example.com")
 *             .type("NS")
 *             .ttl("30")
 *             .records(dev.nameServers())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   main:
 *     type: aws:route53:Zone
 *     properties:
 *       name: example.com
 *   dev:
 *     type: aws:route53:Zone
 *     properties:
 *       name: dev.example.com
 *       tags:
 *         Environment: dev
 *   dev-ns:
 *     type: aws:route53:Record
 *     properties:
 *       zoneId: ${main.zoneId}
 *       name: dev.example.com
 *       type: NS
 *       ttl: '30'
 *       records: ${dev.nameServers}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Private Zone
 * > **NOTE:** This provider provides both exclusive VPC associations defined in-line in this resource via `vpc` configuration blocks and a separate `Zone VPC Association resource. At this time, you cannot use in-line VPC associations in conjunction with any `aws.route53.ZoneAssociation` resources with the same zone ID otherwise it will cause a perpetual difference in plan output. You can optionally use [`ignoreChanges`](https://www.pulumi.com/docs/intro/concepts/programming-model/#ignorechanges) to manage additional associations via the `aws.route53.ZoneAssociation` resource.
 * > **NOTE:** Private zones require at least one VPC association at all times.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const _private = new aws.route53.Zone("private", {
 *     name: "example.com",
 *     vpcs: [{
 *         vpcId: example.id,
 *     }],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * private = aws.route53.Zone("private",
 *     name="example.com",
 *     vpcs=[{
 *         "vpc_id": example["id"],
 *     }])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var @private = new Aws.Route53.Zone("private", new()
 *     {
 *         Name = "example.com",
 *         Vpcs = new[]
 *         {
 *             new Aws.Route53.Inputs.ZoneVpcArgs
 *             {
 *                 VpcId = example.Id,
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/route53"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := route53.NewZone(ctx, "private", &route53.ZoneArgs{
 * 			Name: pulumi.String("example.com"),
 * 			Vpcs: route53.ZoneVpcArray{
 * 				&route53.ZoneVpcArgs{
 * 					VpcId: pulumi.Any(example.Id),
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.route53.Zone;
 * import com.pulumi.aws.route53.ZoneArgs;
 * import com.pulumi.aws.route53.inputs.ZoneVpcArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var private_ = new Zone("private", ZoneArgs.builder()
 *             .name("example.com")
 *             .vpcs(ZoneVpcArgs.builder()
 *                 .vpcId(example.id())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   private:
 *     type: aws:route53:Zone
 *     properties:
 *       name: example.com
 *       vpcs:
 *         - vpcId: ${example.id}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Route53 Zones using the zone `id`. For example:
 * ```sh
 * $ pulumi import aws:route53/zone:Zone myzone Z1D633PJN98FT9
 * ```
 * @property comment A comment for the hosted zone. Defaults to 'Managed by Pulumi'.
 * @property delegationSetId The ID of the reusable delegation set whose NS records you want to assign to the hosted zone. Conflicts with `vpc` as delegation sets can only be used for public zones.
 * @property forceDestroy Whether to destroy all records (possibly managed outside of this provider) in the zone when destroying the zone.
 * @property name This is the name of the hosted zone.
 * @property tags A mapping of tags to assign to the zone. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property vpcs Configuration block(s) specifying VPC(s) to associate with a private hosted zone. Conflicts with the `delegation_set_id` argument in this resource and any `aws.route53.ZoneAssociation` resource specifying the same zone ID. Detailed below.
 */
public data class ZoneArgs(
    public val comment: Output<String>? = null,
    public val delegationSetId: Output<String>? = null,
    public val forceDestroy: Output<Boolean>? = null,
    public val name: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val vpcs: Output<List<ZoneVpcArgs>>? = null,
) : ConvertibleToJava<com.pulumi.aws.route53.ZoneArgs> {
    override fun toJava(): com.pulumi.aws.route53.ZoneArgs = com.pulumi.aws.route53.ZoneArgs.builder()
        .comment(comment?.applyValue({ args0 -> args0 }))
        .delegationSetId(delegationSetId?.applyValue({ args0 -> args0 }))
        .forceDestroy(forceDestroy?.applyValue({ args0 -> args0 }))
        .name(name?.applyValue({ args0 -> args0 }))
        .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
        .vpcs(
            vpcs?.applyValue({ args0 ->
                args0.map({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                })
            }),
        ).build()
}

/**
 * Builder for [ZoneArgs].
 */
@PulumiTagMarker
public class ZoneArgsBuilder internal constructor() {
    private var comment: Output<String>? = null

    private var delegationSetId: Output<String>? = null

    private var forceDestroy: Output<Boolean>? = null

    private var name: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var vpcs: Output<List<ZoneVpcArgs>>? = null

    /**
     * @param value A comment for the hosted zone. Defaults to 'Managed by Pulumi'.
     */
    @JvmName("sbxpnpikklrpijqg")
    public suspend fun comment(`value`: Output<String>) {
        this.comment = value
    }

    /**
     * @param value The ID of the reusable delegation set whose NS records you want to assign to the hosted zone. Conflicts with `vpc` as delegation sets can only be used for public zones.
     */
    @JvmName("rbmualncqtongmky")
    public suspend fun delegationSetId(`value`: Output<String>) {
        this.delegationSetId = value
    }

    /**
     * @param value Whether to destroy all records (possibly managed outside of this provider) in the zone when destroying the zone.
     */
    @JvmName("iknsurushnpjxdji")
    public suspend fun forceDestroy(`value`: Output<Boolean>) {
        this.forceDestroy = value
    }

    /**
     * @param value This is the name of the hosted zone.
     */
    @JvmName("qyumkcetsnhutcbp")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value A mapping of tags to assign to the zone. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("jsuxwitmrvfnmofg")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Configuration block(s) specifying VPC(s) to associate with a private hosted zone. Conflicts with the `delegation_set_id` argument in this resource and any `aws.route53.ZoneAssociation` resource specifying the same zone ID. Detailed below.
     */
    @JvmName("oeferapxvfaujeup")
    public suspend fun vpcs(`value`: Output<List<ZoneVpcArgs>>) {
        this.vpcs = value
    }

    @JvmName("efaejogxvagduwoe")
    public suspend fun vpcs(vararg values: Output<ZoneVpcArgs>) {
        this.vpcs = Output.all(values.asList())
    }

    /**
     * @param values Configuration block(s) specifying VPC(s) to associate with a private hosted zone. Conflicts with the `delegation_set_id` argument in this resource and any `aws.route53.ZoneAssociation` resource specifying the same zone ID. Detailed below.
     */
    @JvmName("aoldbkvxouyhdimf")
    public suspend fun vpcs(values: List<Output<ZoneVpcArgs>>) {
        this.vpcs = Output.all(values)
    }

    /**
     * @param value A comment for the hosted zone. Defaults to 'Managed by Pulumi'.
     */
    @JvmName("tyjebmnsgjyrvueo")
    public suspend fun comment(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.comment = mapped
    }

    /**
     * @param value The ID of the reusable delegation set whose NS records you want to assign to the hosted zone. Conflicts with `vpc` as delegation sets can only be used for public zones.
     */
    @JvmName("pkhqxbirsaxvjoay")
    public suspend fun delegationSetId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.delegationSetId = mapped
    }

    /**
     * @param value Whether to destroy all records (possibly managed outside of this provider) in the zone when destroying the zone.
     */
    @JvmName("oeicddsvxtqqbljf")
    public suspend fun forceDestroy(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.forceDestroy = mapped
    }

    /**
     * @param value This is the name of the hosted zone.
     */
    @JvmName("axdxduibftnfrxsk")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value A mapping of tags to assign to the zone. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("ovkptownggodrrin")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A mapping of tags to assign to the zone. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("jydoyhpiapdbxnah")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Configuration block(s) specifying VPC(s) to associate with a private hosted zone. Conflicts with the `delegation_set_id` argument in this resource and any `aws.route53.ZoneAssociation` resource specifying the same zone ID. Detailed below.
     */
    @JvmName("ulgjunifrnwdvboa")
    public suspend fun vpcs(`value`: List<ZoneVpcArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.vpcs = mapped
    }

    /**
     * @param argument Configuration block(s) specifying VPC(s) to associate with a private hosted zone. Conflicts with the `delegation_set_id` argument in this resource and any `aws.route53.ZoneAssociation` resource specifying the same zone ID. Detailed below.
     */
    @JvmName("spxlciuvlxgixpwf")
    public suspend fun vpcs(argument: List<suspend ZoneVpcArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map { ZoneVpcArgsBuilder().applySuspend { it() }.build() }
        val mapped = of(toBeMapped)
        this.vpcs = mapped
    }

    /**
     * @param argument Configuration block(s) specifying VPC(s) to associate with a private hosted zone. Conflicts with the `delegation_set_id` argument in this resource and any `aws.route53.ZoneAssociation` resource specifying the same zone ID. Detailed below.
     */
    @JvmName("bcoxdtlbvthlqhra")
    public suspend fun vpcs(vararg argument: suspend ZoneVpcArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map { ZoneVpcArgsBuilder().applySuspend { it() }.build() }
        val mapped = of(toBeMapped)
        this.vpcs = mapped
    }

    /**
     * @param argument Configuration block(s) specifying VPC(s) to associate with a private hosted zone. Conflicts with the `delegation_set_id` argument in this resource and any `aws.route53.ZoneAssociation` resource specifying the same zone ID. Detailed below.
     */
    @JvmName("imdpmyxcbdxfxusk")
    public suspend fun vpcs(argument: suspend ZoneVpcArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(ZoneVpcArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.vpcs = mapped
    }

    /**
     * @param values Configuration block(s) specifying VPC(s) to associate with a private hosted zone. Conflicts with the `delegation_set_id` argument in this resource and any `aws.route53.ZoneAssociation` resource specifying the same zone ID. Detailed below.
     */
    @JvmName("plfnsbujkmpsceiq")
    public suspend fun vpcs(vararg values: ZoneVpcArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.vpcs = mapped
    }

    internal fun build(): ZoneArgs = ZoneArgs(
        comment = comment,
        delegationSetId = delegationSetId,
        forceDestroy = forceDestroy,
        name = name,
        tags = tags,
        vpcs = vpcs,
    )
}
