@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.s3.kotlin

import com.pulumi.asset.AssetOrArchive
import com.pulumi.aws.s3.kotlin.outputs.BucketObjectv2OverrideProvider
import com.pulumi.aws.s3.kotlin.outputs.BucketObjectv2OverrideProvider.Companion.toKotlin
import com.pulumi.core.Output
import com.pulumi.kotlin.KotlinCustomResource
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.ResourceMapper
import com.pulumi.kotlin.options.CustomResourceOptions
import com.pulumi.kotlin.options.CustomResourceOptionsBuilder
import com.pulumi.resources.Resource
import kotlin.Boolean
import kotlin.Deprecated
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.Map

/**
 * Builder for [BucketObjectv2].
 */
@PulumiTagMarker
public class BucketObjectv2ResourceBuilder internal constructor() {
    public var name: String? = null

    public var args: BucketObjectv2Args = BucketObjectv2Args()

    public var opts: CustomResourceOptions = CustomResourceOptions()

    /**
     * @param name The _unique_ name of the resulting resource.
     */
    public fun name(`value`: String) {
        this.name = value
    }

    /**
     * @param block The arguments to use to populate this resource's properties.
     */
    public suspend fun args(block: suspend BucketObjectv2ArgsBuilder.() -> Unit) {
        val builder = BucketObjectv2ArgsBuilder()
        block(builder)
        this.args = builder.build()
    }

    /**
     * @param block A bag of options that control this resource's behavior.
     */
    public suspend fun opts(block: suspend CustomResourceOptionsBuilder.() -> Unit) {
        this.opts = com.pulumi.kotlin.options.CustomResourceOptions.opts(block)
    }

    internal fun build(): BucketObjectv2 {
        val builtJavaResource = com.pulumi.aws.s3.BucketObjectv2(
            this.name,
            this.args.toJava(),
            this.opts.toJava(),
        )
        return BucketObjectv2(builtJavaResource)
    }
}

/**
 * Provides an S3 object resource.
 * ## Example Usage
 * ### Uploading a file to a bucket
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * import * as std from "@pulumi/std";
 * const object = new aws.s3.BucketObjectv2("object", {
 *     bucket: "your_bucket_name",
 *     key: "new_object_key",
 *     source: new pulumi.asset.FileAsset("path/to/file"),
 *     etag: std.filemd5({
 *         input: "path/to/file",
 *     }).then(invoke => invoke.result),
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * import pulumi_std as std
 * object = aws.s3.BucketObjectv2("object",
 *     bucket="your_bucket_name",
 *     key="new_object_key",
 *     source=pulumi.FileAsset("path/to/file"),
 *     etag=std.filemd5(input="path/to/file").result)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * using Std = Pulumi.Std;
 * return await Deployment.RunAsync(() =>
 * {
 *     var @object = new Aws.S3.BucketObjectv2("object", new()
 *     {
 *         Bucket = "your_bucket_name",
 *         Key = "new_object_key",
 *         Source = new FileAsset("path/to/file"),
 *         Etag = Std.Filemd5.Invoke(new()
 *         {
 *             Input = "path/to/file",
 *         }).Apply(invoke => invoke.Result),
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/s3"
 * 	"github.com/pulumi/pulumi-std/sdk/go/std"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		invokeFilemd5, err := std.Filemd5(ctx, &std.Filemd5Args{
 * 			Input: "path/to/file",
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = s3.NewBucketObjectv2(ctx, "object", &s3.BucketObjectv2Args{
 * 			Bucket: pulumi.Any("your_bucket_name"),
 * 			Key:    pulumi.String("new_object_key"),
 * 			Source: pulumi.NewFileAsset("path/to/file"),
 * 			Etag:   pulumi.String(invokeFilemd5.Result),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.BucketObjectv2;
 * import com.pulumi.aws.s3.BucketObjectv2Args;
 * import com.pulumi.asset.FileAsset;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var object = new BucketObjectv2("object", BucketObjectv2Args.builder()
 *             .bucket("your_bucket_name")
 *             .key("new_object_key")
 *             .source(new FileAsset("path/to/file"))
 *             .etag(StdFunctions.filemd5(Filemd5Args.builder()
 *                 .input("path/to/file")
 *                 .build()).result())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   object:
 *     type: aws:s3:BucketObjectv2
 *     properties:
 *       bucket: your_bucket_name
 *       key: new_object_key
 *       source:
 *         fn::FileAsset: path/to/file
 *       etag:
 *         fn::invoke:
 *           Function: std:filemd5
 *           Arguments:
 *             input: path/to/file
 *           Return: result
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Encrypting with KMS Key
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const examplekms = new aws.kms.Key("examplekms", {
 *     description: "KMS key 1",
 *     deletionWindowInDays: 7,
 * });
 * const examplebucket = new aws.s3.BucketV2("examplebucket", {bucket: "examplebuckettftest"});
 * const example = new aws.s3.BucketAclV2("example", {
 *     bucket: examplebucket.id,
 *     acl: "private",
 * });
 * const exampleBucketObjectv2 = new aws.s3.BucketObjectv2("example", {
 *     key: "someobject",
 *     bucket: examplebucket.id,
 *     source: new pulumi.asset.FileAsset("index.html"),
 *     kmsKeyId: examplekms.arn,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * examplekms = aws.kms.Key("examplekms",
 *     description="KMS key 1",
 *     deletion_window_in_days=7)
 * examplebucket = aws.s3.BucketV2("examplebucket", bucket="examplebuckettftest")
 * example = aws.s3.BucketAclV2("example",
 *     bucket=examplebucket.id,
 *     acl="private")
 * example_bucket_objectv2 = aws.s3.BucketObjectv2("example",
 *     key="someobject",
 *     bucket=examplebucket.id,
 *     source=pulumi.FileAsset("index.html"),
 *     kms_key_id=examplekms.arn)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var examplekms = new Aws.Kms.Key("examplekms", new()
 *     {
 *         Description = "KMS key 1",
 *         DeletionWindowInDays = 7,
 *     });
 *     var examplebucket = new Aws.S3.BucketV2("examplebucket", new()
 *     {
 *         Bucket = "examplebuckettftest",
 *     });
 *     var example = new Aws.S3.BucketAclV2("example", new()
 *     {
 *         Bucket = examplebucket.Id,
 *         Acl = "private",
 *     });
 *     var exampleBucketObjectv2 = new Aws.S3.BucketObjectv2("example", new()
 *     {
 *         Key = "someobject",
 *         Bucket = examplebucket.Id,
 *         Source = new FileAsset("index.html"),
 *         KmsKeyId = examplekms.Arn,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/kms"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/s3"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		examplekms, err := kms.NewKey(ctx, "examplekms", &kms.KeyArgs{
 * 			Description:          pulumi.String("KMS key 1"),
 * 			DeletionWindowInDays: pulumi.Int(7),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		examplebucket, err := s3.NewBucketV2(ctx, "examplebucket", &s3.BucketV2Args{
 * 			Bucket: pulumi.String("examplebuckettftest"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = s3.NewBucketAclV2(ctx, "example", &s3.BucketAclV2Args{
 * 			Bucket: examplebucket.ID(),
 * 			Acl:    pulumi.String("private"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = s3.NewBucketObjectv2(ctx, "example", &s3.BucketObjectv2Args{
 * 			Key:      pulumi.String("someobject"),
 * 			Bucket:   examplebucket.ID(),
 * 			Source:   pulumi.NewFileAsset("index.html"),
 * 			KmsKeyId: examplekms.Arn,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.kms.Key;
 * import com.pulumi.aws.kms.KeyArgs;
 * import com.pulumi.aws.s3.BucketV2;
 * import com.pulumi.aws.s3.BucketV2Args;
 * import com.pulumi.aws.s3.BucketAclV2;
 * import com.pulumi.aws.s3.BucketAclV2Args;
 * import com.pulumi.aws.s3.BucketObjectv2;
 * import com.pulumi.aws.s3.BucketObjectv2Args;
 * import com.pulumi.asset.FileAsset;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var examplekms = new Key("examplekms", KeyArgs.builder()
 *             .description("KMS key 1")
 *             .deletionWindowInDays(7)
 *             .build());
 *         var examplebucket = new BucketV2("examplebucket", BucketV2Args.builder()
 *             .bucket("examplebuckettftest")
 *             .build());
 *         var example = new BucketAclV2("example", BucketAclV2Args.builder()
 *             .bucket(examplebucket.id())
 *             .acl("private")
 *             .build());
 *         var exampleBucketObjectv2 = new BucketObjectv2("exampleBucketObjectv2", BucketObjectv2Args.builder()
 *             .key("someobject")
 *             .bucket(examplebucket.id())
 *             .source(new FileAsset("index.html"))
 *             .kmsKeyId(examplekms.arn())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   examplekms:
 *     type: aws:kms:Key
 *     properties:
 *       description: KMS key 1
 *       deletionWindowInDays: 7
 *   examplebucket:
 *     type: aws:s3:BucketV2
 *     properties:
 *       bucket: examplebuckettftest
 *   example:
 *     type: aws:s3:BucketAclV2
 *     properties:
 *       bucket: ${examplebucket.id}
 *       acl: private
 *   exampleBucketObjectv2:
 *     type: aws:s3:BucketObjectv2
 *     name: example
 *     properties:
 *       key: someobject
 *       bucket: ${examplebucket.id}
 *       source:
 *         fn::FileAsset: index.html
 *       kmsKeyId: ${examplekms.arn}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Server Side Encryption with S3 Default Master Key
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const examplebucket = new aws.s3.BucketV2("examplebucket", {bucket: "examplebuckettftest"});
 * const example = new aws.s3.BucketAclV2("example", {
 *     bucket: examplebucket.id,
 *     acl: "private",
 * });
 * const exampleBucketObjectv2 = new aws.s3.BucketObjectv2("example", {
 *     key: "someobject",
 *     bucket: examplebucket.id,
 *     source: new pulumi.asset.FileAsset("index.html"),
 *     serverSideEncryption: "aws:kms",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * examplebucket = aws.s3.BucketV2("examplebucket", bucket="examplebuckettftest")
 * example = aws.s3.BucketAclV2("example",
 *     bucket=examplebucket.id,
 *     acl="private")
 * example_bucket_objectv2 = aws.s3.BucketObjectv2("example",
 *     key="someobject",
 *     bucket=examplebucket.id,
 *     source=pulumi.FileAsset("index.html"),
 *     server_side_encryption="aws:kms")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var examplebucket = new Aws.S3.BucketV2("examplebucket", new()
 *     {
 *         Bucket = "examplebuckettftest",
 *     });
 *     var example = new Aws.S3.BucketAclV2("example", new()
 *     {
 *         Bucket = examplebucket.Id,
 *         Acl = "private",
 *     });
 *     var exampleBucketObjectv2 = new Aws.S3.BucketObjectv2("example", new()
 *     {
 *         Key = "someobject",
 *         Bucket = examplebucket.Id,
 *         Source = new FileAsset("index.html"),
 *         ServerSideEncryption = "aws:kms",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/s3"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		examplebucket, err := s3.NewBucketV2(ctx, "examplebucket", &s3.BucketV2Args{
 * 			Bucket: pulumi.String("examplebuckettftest"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = s3.NewBucketAclV2(ctx, "example", &s3.BucketAclV2Args{
 * 			Bucket: examplebucket.ID(),
 * 			Acl:    pulumi.String("private"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = s3.NewBucketObjectv2(ctx, "example", &s3.BucketObjectv2Args{
 * 			Key:                  pulumi.String("someobject"),
 * 			Bucket:               examplebucket.ID(),
 * 			Source:               pulumi.NewFileAsset("index.html"),
 * 			ServerSideEncryption: pulumi.String("aws:kms"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.BucketV2;
 * import com.pulumi.aws.s3.BucketV2Args;
 * import com.pulumi.aws.s3.BucketAclV2;
 * import com.pulumi.aws.s3.BucketAclV2Args;
 * import com.pulumi.aws.s3.BucketObjectv2;
 * import com.pulumi.aws.s3.BucketObjectv2Args;
 * import com.pulumi.asset.FileAsset;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var examplebucket = new BucketV2("examplebucket", BucketV2Args.builder()
 *             .bucket("examplebuckettftest")
 *             .build());
 *         var example = new BucketAclV2("example", BucketAclV2Args.builder()
 *             .bucket(examplebucket.id())
 *             .acl("private")
 *             .build());
 *         var exampleBucketObjectv2 = new BucketObjectv2("exampleBucketObjectv2", BucketObjectv2Args.builder()
 *             .key("someobject")
 *             .bucket(examplebucket.id())
 *             .source(new FileAsset("index.html"))
 *             .serverSideEncryption("aws:kms")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   examplebucket:
 *     type: aws:s3:BucketV2
 *     properties:
 *       bucket: examplebuckettftest
 *   example:
 *     type: aws:s3:BucketAclV2
 *     properties:
 *       bucket: ${examplebucket.id}
 *       acl: private
 *   exampleBucketObjectv2:
 *     type: aws:s3:BucketObjectv2
 *     name: example
 *     properties:
 *       key: someobject
 *       bucket: ${examplebucket.id}
 *       source:
 *         fn::FileAsset: index.html
 *       serverSideEncryption: aws:kms
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Server Side Encryption with AWS-Managed Key
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const examplebucket = new aws.s3.BucketV2("examplebucket", {bucket: "examplebuckettftest"});
 * const example = new aws.s3.BucketAclV2("example", {
 *     bucket: examplebucket.id,
 *     acl: "private",
 * });
 * const exampleBucketObjectv2 = new aws.s3.BucketObjectv2("example", {
 *     key: "someobject",
 *     bucket: examplebucket.id,
 *     source: new pulumi.asset.FileAsset("index.html"),
 *     serverSideEncryption: "AES256",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * examplebucket = aws.s3.BucketV2("examplebucket", bucket="examplebuckettftest")
 * example = aws.s3.BucketAclV2("example",
 *     bucket=examplebucket.id,
 *     acl="private")
 * example_bucket_objectv2 = aws.s3.BucketObjectv2("example",
 *     key="someobject",
 *     bucket=examplebucket.id,
 *     source=pulumi.FileAsset("index.html"),
 *     server_side_encryption="AES256")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var examplebucket = new Aws.S3.BucketV2("examplebucket", new()
 *     {
 *         Bucket = "examplebuckettftest",
 *     });
 *     var example = new Aws.S3.BucketAclV2("example", new()
 *     {
 *         Bucket = examplebucket.Id,
 *         Acl = "private",
 *     });
 *     var exampleBucketObjectv2 = new Aws.S3.BucketObjectv2("example", new()
 *     {
 *         Key = "someobject",
 *         Bucket = examplebucket.Id,
 *         Source = new FileAsset("index.html"),
 *         ServerSideEncryption = "AES256",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/s3"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		examplebucket, err := s3.NewBucketV2(ctx, "examplebucket", &s3.BucketV2Args{
 * 			Bucket: pulumi.String("examplebuckettftest"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = s3.NewBucketAclV2(ctx, "example", &s3.BucketAclV2Args{
 * 			Bucket: examplebucket.ID(),
 * 			Acl:    pulumi.String("private"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = s3.NewBucketObjectv2(ctx, "example", &s3.BucketObjectv2Args{
 * 			Key:                  pulumi.String("someobject"),
 * 			Bucket:               examplebucket.ID(),
 * 			Source:               pulumi.NewFileAsset("index.html"),
 * 			ServerSideEncryption: pulumi.String("AES256"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.BucketV2;
 * import com.pulumi.aws.s3.BucketV2Args;
 * import com.pulumi.aws.s3.BucketAclV2;
 * import com.pulumi.aws.s3.BucketAclV2Args;
 * import com.pulumi.aws.s3.BucketObjectv2;
 * import com.pulumi.aws.s3.BucketObjectv2Args;
 * import com.pulumi.asset.FileAsset;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var examplebucket = new BucketV2("examplebucket", BucketV2Args.builder()
 *             .bucket("examplebuckettftest")
 *             .build());
 *         var example = new BucketAclV2("example", BucketAclV2Args.builder()
 *             .bucket(examplebucket.id())
 *             .acl("private")
 *             .build());
 *         var exampleBucketObjectv2 = new BucketObjectv2("exampleBucketObjectv2", BucketObjectv2Args.builder()
 *             .key("someobject")
 *             .bucket(examplebucket.id())
 *             .source(new FileAsset("index.html"))
 *             .serverSideEncryption("AES256")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   examplebucket:
 *     type: aws:s3:BucketV2
 *     properties:
 *       bucket: examplebuckettftest
 *   example:
 *     type: aws:s3:BucketAclV2
 *     properties:
 *       bucket: ${examplebucket.id}
 *       acl: private
 *   exampleBucketObjectv2:
 *     type: aws:s3:BucketObjectv2
 *     name: example
 *     properties:
 *       key: someobject
 *       bucket: ${examplebucket.id}
 *       source:
 *         fn::FileAsset: index.html
 *       serverSideEncryption: AES256
 * ```
 * <!--End PulumiCodeChooser -->
 * ### S3 Object Lock
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const examplebucket = new aws.s3.BucketV2("examplebucket", {
 *     bucket: "examplebuckettftest",
 *     objectLockEnabled: true,
 * });
 * const example = new aws.s3.BucketAclV2("example", {
 *     bucket: examplebucket.id,
 *     acl: "private",
 * });
 * const exampleBucketVersioningV2 = new aws.s3.BucketVersioningV2("example", {
 *     bucket: examplebucket.id,
 *     versioningConfiguration: {
 *         status: "Enabled",
 *     },
 * });
 * const examplebucketObject = new aws.s3.BucketObjectv2("examplebucket_object", {
 *     key: "someobject",
 *     bucket: examplebucket.id,
 *     source: new pulumi.asset.FileAsset("important.txt"),
 *     objectLockLegalHoldStatus: "ON",
 *     objectLockMode: "GOVERNANCE",
 *     objectLockRetainUntilDate: "2021-12-31T23:59:60Z",
 *     forceDestroy: true,
 * }, {
 *     dependsOn: [exampleBucketVersioningV2],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * examplebucket = aws.s3.BucketV2("examplebucket",
 *     bucket="examplebuckettftest",
 *     object_lock_enabled=True)
 * example = aws.s3.BucketAclV2("example",
 *     bucket=examplebucket.id,
 *     acl="private")
 * example_bucket_versioning_v2 = aws.s3.BucketVersioningV2("example",
 *     bucket=examplebucket.id,
 *     versioning_configuration={
 *         "status": "Enabled",
 *     })
 * examplebucket_object = aws.s3.BucketObjectv2("examplebucket_object",
 *     key="someobject",
 *     bucket=examplebucket.id,
 *     source=pulumi.FileAsset("important.txt"),
 *     object_lock_legal_hold_status="ON",
 *     object_lock_mode="GOVERNANCE",
 *     object_lock_retain_until_date="2021-12-31T23:59:60Z",
 *     force_destroy=True,
 *     opts = pulumi.ResourceOptions(depends_on=[example_bucket_versioning_v2]))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var examplebucket = new Aws.S3.BucketV2("examplebucket", new()
 *     {
 *         Bucket = "examplebuckettftest",
 *         ObjectLockEnabled = true,
 *     });
 *     var example = new Aws.S3.BucketAclV2("example", new()
 *     {
 *         Bucket = examplebucket.Id,
 *         Acl = "private",
 *     });
 *     var exampleBucketVersioningV2 = new Aws.S3.BucketVersioningV2("example", new()
 *     {
 *         Bucket = examplebucket.Id,
 *         VersioningConfiguration = new Aws.S3.Inputs.BucketVersioningV2VersioningConfigurationArgs
 *         {
 *             Status = "Enabled",
 *         },
 *     });
 *     var examplebucketObject = new Aws.S3.BucketObjectv2("examplebucket_object", new()
 *     {
 *         Key = "someobject",
 *         Bucket = examplebucket.Id,
 *         Source = new FileAsset("important.txt"),
 *         ObjectLockLegalHoldStatus = "ON",
 *         ObjectLockMode = "GOVERNANCE",
 *         ObjectLockRetainUntilDate = "2021-12-31T23:59:60Z",
 *         ForceDestroy = true,
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             exampleBucketVersioningV2,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/s3"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		examplebucket, err := s3.NewBucketV2(ctx, "examplebucket", &s3.BucketV2Args{
 * 			Bucket:            pulumi.String("examplebuckettftest"),
 * 			ObjectLockEnabled: pulumi.Bool(true),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = s3.NewBucketAclV2(ctx, "example", &s3.BucketAclV2Args{
 * 			Bucket: examplebucket.ID(),
 * 			Acl:    pulumi.String("private"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleBucketVersioningV2, err := s3.NewBucketVersioningV2(ctx, "example", &s3.BucketVersioningV2Args{
 * 			Bucket: examplebucket.ID(),
 * 			VersioningConfiguration: &s3.BucketVersioningV2VersioningConfigurationArgs{
 * 				Status: pulumi.String("Enabled"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = s3.NewBucketObjectv2(ctx, "examplebucket_object", &s3.BucketObjectv2Args{
 * 			Key:                       pulumi.String("someobject"),
 * 			Bucket:                    examplebucket.ID(),
 * 			Source:                    pulumi.NewFileAsset("important.txt"),
 * 			ObjectLockLegalHoldStatus: pulumi.String("ON"),
 * 			ObjectLockMode:            pulumi.String("GOVERNANCE"),
 * 			ObjectLockRetainUntilDate: pulumi.String("2021-12-31T23:59:60Z"),
 * 			ForceDestroy:              pulumi.Bool(true),
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			exampleBucketVersioningV2,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.BucketV2;
 * import com.pulumi.aws.s3.BucketV2Args;
 * import com.pulumi.aws.s3.BucketAclV2;
 * import com.pulumi.aws.s3.BucketAclV2Args;
 * import com.pulumi.aws.s3.BucketVersioningV2;
 * import com.pulumi.aws.s3.BucketVersioningV2Args;
 * import com.pulumi.aws.s3.inputs.BucketVersioningV2VersioningConfigurationArgs;
 * import com.pulumi.aws.s3.BucketObjectv2;
 * import com.pulumi.aws.s3.BucketObjectv2Args;
 * import com.pulumi.resources.CustomResourceOptions;
 * import com.pulumi.asset.FileAsset;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var examplebucket = new BucketV2("examplebucket", BucketV2Args.builder()
 *             .bucket("examplebuckettftest")
 *             .objectLockEnabled(true)
 *             .build());
 *         var example = new BucketAclV2("example", BucketAclV2Args.builder()
 *             .bucket(examplebucket.id())
 *             .acl("private")
 *             .build());
 *         var exampleBucketVersioningV2 = new BucketVersioningV2("exampleBucketVersioningV2", BucketVersioningV2Args.builder()
 *             .bucket(examplebucket.id())
 *             .versioningConfiguration(BucketVersioningV2VersioningConfigurationArgs.builder()
 *                 .status("Enabled")
 *                 .build())
 *             .build());
 *         var examplebucketObject = new BucketObjectv2("examplebucketObject", BucketObjectv2Args.builder()
 *             .key("someobject")
 *             .bucket(examplebucket.id())
 *             .source(new FileAsset("important.txt"))
 *             .objectLockLegalHoldStatus("ON")
 *             .objectLockMode("GOVERNANCE")
 *             .objectLockRetainUntilDate("2021-12-31T23:59:60Z")
 *             .forceDestroy(true)
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(exampleBucketVersioningV2)
 *                 .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   examplebucket:
 *     type: aws:s3:BucketV2
 *     properties:
 *       bucket: examplebuckettftest
 *       objectLockEnabled: true
 *   example:
 *     type: aws:s3:BucketAclV2
 *     properties:
 *       bucket: ${examplebucket.id}
 *       acl: private
 *   exampleBucketVersioningV2:
 *     type: aws:s3:BucketVersioningV2
 *     name: example
 *     properties:
 *       bucket: ${examplebucket.id}
 *       versioningConfiguration:
 *         status: Enabled
 *   examplebucketObject:
 *     type: aws:s3:BucketObjectv2
 *     name: examplebucket_object
 *     properties:
 *       key: someobject
 *       bucket: ${examplebucket.id}
 *       source:
 *         fn::FileAsset: important.txt
 *       objectLockLegalHoldStatus: ON
 *       objectLockMode: GOVERNANCE
 *       objectLockRetainUntilDate: 2021-12-31T23:59:60Z
 *       forceDestroy: true
 *     options:
 *       dependson:
 *         - ${exampleBucketVersioningV2}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Ignoring Provider `default_tags`
 * S3 objects support a [maximum of 10 tags](https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-tagging.html).
 * If the resource's own `tags` and the provider-level `default_tags` would together lead to more than 10 tags on an S3 object, use the `override_provider` configuration block to suppress any provider-level `default_tags`.
 * > S3 objects stored in Amazon S3 Express directory buckets do not support tags, so any provider-level `default_tags` must be suppressed.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const examplebucket = new aws.s3.BucketV2("examplebucket", {bucket: "examplebuckettftest"});
 * const examplebucketObject = new aws.s3.BucketObjectv2("examplebucket_object", {
 *     key: "someobject",
 *     bucket: examplebucket.id,
 *     source: new pulumi.asset.FileAsset("important.txt"),
 *     tags: {
 *         Env: "test",
 *     },
 *     overrideProvider: {
 *         defaultTags: {
 *             tags: {},
 *         },
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * examplebucket = aws.s3.BucketV2("examplebucket", bucket="examplebuckettftest")
 * examplebucket_object = aws.s3.BucketObjectv2("examplebucket_object",
 *     key="someobject",
 *     bucket=examplebucket.id,
 *     source=pulumi.FileAsset("important.txt"),
 *     tags={
 *         "Env": "test",
 *     },
 *     override_provider={
 *         "default_tags": {
 *             "tags": {},
 *         },
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var examplebucket = new Aws.S3.BucketV2("examplebucket", new()
 *     {
 *         Bucket = "examplebuckettftest",
 *     });
 *     var examplebucketObject = new Aws.S3.BucketObjectv2("examplebucket_object", new()
 *     {
 *         Key = "someobject",
 *         Bucket = examplebucket.Id,
 *         Source = new FileAsset("important.txt"),
 *         Tags =
 *         {
 *             { "Env", "test" },
 *         },
 *         OverrideProvider = new Aws.S3.Inputs.BucketObjectv2OverrideProviderArgs
 *         {
 *             DefaultTags = new Aws.S3.Inputs.BucketObjectv2OverrideProviderDefaultTagsArgs
 *             {
 *                 Tags = null,
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/s3"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		examplebucket, err := s3.NewBucketV2(ctx, "examplebucket", &s3.BucketV2Args{
 * 			Bucket: pulumi.String("examplebuckettftest"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = s3.NewBucketObjectv2(ctx, "examplebucket_object", &s3.BucketObjectv2Args{
 * 			Key:    pulumi.String("someobject"),
 * 			Bucket: examplebucket.ID(),
 * 			Source: pulumi.NewFileAsset("important.txt"),
 * 			Tags: pulumi.StringMap{
 * 				"Env": pulumi.String("test"),
 * 			},
 * 			OverrideProvider: &s3.BucketObjectv2OverrideProviderArgs{
 * 				DefaultTags: &s3.BucketObjectv2OverrideProviderDefaultTagsArgs{
 * 					Tags: nil,
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.BucketV2;
 * import com.pulumi.aws.s3.BucketV2Args;
 * import com.pulumi.aws.s3.BucketObjectv2;
 * import com.pulumi.aws.s3.BucketObjectv2Args;
 * import com.pulumi.aws.s3.inputs.BucketObjectv2OverrideProviderArgs;
 * import com.pulumi.aws.s3.inputs.BucketObjectv2OverrideProviderDefaultTagsArgs;
 * import com.pulumi.asset.FileAsset;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var examplebucket = new BucketV2("examplebucket", BucketV2Args.builder()
 *             .bucket("examplebuckettftest")
 *             .build());
 *         var examplebucketObject = new BucketObjectv2("examplebucketObject", BucketObjectv2Args.builder()
 *             .key("someobject")
 *             .bucket(examplebucket.id())
 *             .source(new FileAsset("important.txt"))
 *             .tags(Map.of("Env", "test"))
 *             .overrideProvider(BucketObjectv2OverrideProviderArgs.builder()
 *                 .defaultTags(BucketObjectv2OverrideProviderDefaultTagsArgs.builder()
 *                     .tags()
 *                     .build())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   examplebucket:
 *     type: aws:s3:BucketV2
 *     properties:
 *       bucket: examplebuckettftest
 *   examplebucketObject:
 *     type: aws:s3:BucketObjectv2
 *     name: examplebucket_object
 *     properties:
 *       key: someobject
 *       bucket: ${examplebucket.id}
 *       source:
 *         fn::FileAsset: important.txt
 *       tags:
 *         Env: test
 *       overrideProvider:
 *         defaultTags:
 *           tags: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Import using S3 URL syntax:
 * __Using `pulumi import` to import__ objects using the `id` or S3 URL. For example:
 * Import using the `id`, which is the bucket name and the key together:
 * ```sh
 * $ pulumi import aws:s3/bucketObjectv2:BucketObjectv2 example some-bucket-name/some/key.txt
 * ```
 * Import using S3 URL syntax:
 * ```sh
 * $ pulumi import aws:s3/bucketObjectv2:BucketObjectv2 example s3://some-bucket-name/some/key.txt
 * ```
 */
public class BucketObjectv2 internal constructor(
    override val javaResource: com.pulumi.aws.s3.BucketObjectv2,
) : KotlinCustomResource(javaResource, BucketObjectv2Mapper) {
    /**
     * [Canned ACL](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl) to apply. Valid values are `private`, `public-read`, `public-read-write`, `aws-exec-read`, `authenticated-read`, `bucket-owner-read`, and `bucket-owner-full-control`.
     */
    public val acl: Output<String>
        get() = javaResource.acl().applyValue({ args0 -> args0 })

    /**
     * ARN of the object.
     */
    public val arn: Output<String>
        get() = javaResource.arn().applyValue({ args0 -> args0 })

    /**
     * Name of the bucket to put the file in. Alternatively, an [S3 access point](https://docs.aws.amazon.com/AmazonS3/latest/dev/using-access-points.html) ARN can be specified.
     */
    public val bucket: Output<String>
        get() = javaResource.bucket().applyValue({ args0 -> args0 })

    /**
     * Whether or not to use [Amazon S3 Bucket Keys](https://docs.aws.amazon.com/AmazonS3/latest/dev/bucket-key.html) for SSE-KMS.
     */
    public val bucketKeyEnabled: Output<Boolean>
        get() = javaResource.bucketKeyEnabled().applyValue({ args0 -> args0 })

    /**
     * Caching behavior along the request/reply chain Read [w3c cache_control](http://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.9) for further details.
     */
    public val cacheControl: Output<String>?
        get() = javaResource.cacheControl().applyValue({ args0 ->
            args0.map({ args0 ->
                args0
            }).orElse(null)
        })

    /**
     * Indicates the algorithm used to create the checksum for the object. If a value is specified and the object is encrypted with KMS, you must have permission to use the `kms:Decrypt` action. Valid values: `CRC32`, `CRC32C`, `SHA1`, `SHA256`.
     */
    public val checksumAlgorithm: Output<String>?
        get() = javaResource.checksumAlgorithm().applyValue({ args0 ->
            args0.map({ args0 ->
                args0
            }).orElse(null)
        })

    /**
     * The base64-encoded, 32-bit CRC32 checksum of the object.
     */
    public val checksumCrc32: Output<String>
        get() = javaResource.checksumCrc32().applyValue({ args0 -> args0 })

    /**
     * The base64-encoded, 32-bit CRC32C checksum of the object.
     */
    public val checksumCrc32c: Output<String>
        get() = javaResource.checksumCrc32c().applyValue({ args0 -> args0 })

    /**
     * The base64-encoded, 160-bit SHA-1 digest of the object.
     */
    public val checksumSha1: Output<String>
        get() = javaResource.checksumSha1().applyValue({ args0 -> args0 })

    /**
     * The base64-encoded, 256-bit SHA-256 digest of the object.
     */
    public val checksumSha256: Output<String>
        get() = javaResource.checksumSha256().applyValue({ args0 -> args0 })

    /**
     * Literal string value to use as the object content, which will be uploaded as UTF-8-encoded text.
     */
    public val content: Output<String>?
        get() = javaResource.content().applyValue({ args0 -> args0.map({ args0 -> args0 }).orElse(null) })

    /**
     * Base64-encoded data that will be decoded and uploaded as raw bytes for the object content. This allows safely uploading non-UTF8 binary data, but is recommended only for small content such as the result of the `gzipbase64` function with small text strings. For larger objects, use `source` to stream the content from a disk file.
     */
    public val contentBase64: Output<String>?
        get() = javaResource.contentBase64().applyValue({ args0 ->
            args0.map({ args0 ->
                args0
            }).orElse(null)
        })

    /**
     * Presentational information for the object. Read [w3c content_disposition](http://www.w3.org/Protocols/rfc2616/rfc2616-sec19.html#sec19.5.1) for further information.
     */
    public val contentDisposition: Output<String>?
        get() = javaResource.contentDisposition().applyValue({ args0 ->
            args0.map({ args0 ->
                args0
            }).orElse(null)
        })

    /**
     * Content encodings that have been applied to the object and thus what decoding mechanisms must be applied to obtain the media-type referenced by the Content-Type header field. Read [w3c content encoding](http://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.11) for further information.
     */
    public val contentEncoding: Output<String>?
        get() = javaResource.contentEncoding().applyValue({ args0 ->
            args0.map({ args0 ->
                args0
            }).orElse(null)
        })

    /**
     * Language the content is in e.g., en-US or en-GB.
     */
    public val contentLanguage: Output<String>?
        get() = javaResource.contentLanguage().applyValue({ args0 ->
            args0.map({ args0 ->
                args0
            }).orElse(null)
        })

    /**
     * Standard MIME type describing the format of the object data, e.g., application/octet-stream. All Valid MIME Types are valid for this input.
     */
    public val contentType: Output<String>
        get() = javaResource.contentType().applyValue({ args0 -> args0 })

    /**
     * Triggers updates when the value changes. This attribute is not compatible with KMS encryption, `kms_key_id` or `server_side_encryption = "aws:kms"`, also if an object is larger than 16 MB, the AWS Management Console will upload or copy that object as a Multipart Upload, and therefore the ETag will not be an MD5 digest (see `source_hash` instead).
     */
    public val etag: Output<String>
        get() = javaResource.etag().applyValue({ args0 -> args0 })

    /**
     * Whether to allow the object to be deleted by removing any legal hold on any object version. Default is `false`. This value should be set to `true` only if the bucket has S3 object lock enabled.
     */
    public val forceDestroy: Output<Boolean>?
        get() = javaResource.forceDestroy().applyValue({ args0 ->
            args0.map({ args0 ->
                args0
            }).orElse(null)
        })

    /**
     * Name of the object once it is in the bucket.
     * The following arguments are optional:
     */
    public val key: Output<String>
        get() = javaResource.key().applyValue({ args0 -> args0 })

    /**
     * ARN of the KMS Key to use for object encryption. If the S3 Bucket has server-side encryption enabled, that value will automatically be used. If referencing the `aws.kms.Key` resource, use the `arn` attribute. If referencing the `aws.kms.Alias` data source or resource, use the `target_key_arn` attribute. The provider will only perform drift detection if a configuration value is provided.
     */
    public val kmsKeyId: Output<String>
        get() = javaResource.kmsKeyId().applyValue({ args0 -> args0 })

    /**
     * Map of keys/values to provision metadata (will be automatically prefixed by `x-amz-meta-`, note that only lowercase label are currently supported by the AWS Go API).
     */
    public val metadata: Output<Map<String, String>>?
        get() = javaResource.metadata().applyValue({ args0 ->
            args0.map({ args0 ->
                args0.map({ args0 ->
                    args0.key.to(args0.value)
                }).toMap()
            }).orElse(null)
        })

    /**
     * [Legal hold](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-legal-holds) status that you want to apply to the specified object. Valid values are `ON` and `OFF`.
     */
    public val objectLockLegalHoldStatus: Output<String>?
        get() = javaResource.objectLockLegalHoldStatus().applyValue({ args0 ->
            args0.map({ args0 ->
                args0
            }).orElse(null)
        })

    /**
     * Object lock [retention mode](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-retention-modes) that you want to apply to this object. Valid values are `GOVERNANCE` and `COMPLIANCE`.
     */
    public val objectLockMode: Output<String>?
        get() = javaResource.objectLockMode().applyValue({ args0 ->
            args0.map({ args0 ->
                args0
            }).orElse(null)
        })

    /**
     * Date and time, in [RFC3339 format](https://tools&#46;ietf&#46;org/html/rfc3339#section-5&#46;8), when this object's object lock will [expire](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-retention-periods).
     */
    public val objectLockRetainUntilDate: Output<String>?
        get() = javaResource.objectLockRetainUntilDate().applyValue({ args0 ->
            args0.map({ args0 ->
                args0
            }).orElse(null)
        })

    /**
     * Override provider-level configuration options. See Override Provider below for more details.
     */
    public val overrideProvider: Output<BucketObjectv2OverrideProvider>?
        get() = javaResource.overrideProvider().applyValue({ args0 ->
            args0.map({ args0 ->
                args0.let({ args0 -> toKotlin(args0) })
            }).orElse(null)
        })

    /**
     * Server-side encryption of the object in S3. Valid values are "`AES256`" and "`aws:kms`".
     */
    public val serverSideEncryption: Output<String>
        get() = javaResource.serverSideEncryption().applyValue({ args0 -> args0 })

    /**
     * Path to a file that will be read and uploaded as raw bytes for the object content.
     */
    public val source: Output<AssetOrArchive>?
        get() = javaResource.source().applyValue({ args0 -> args0.map({ args0 -> args0 }).orElse(null) })

    /**
     * Triggers updates like `etag` but useful to address `etag` encryption limitations. (The value is only stored in state and not saved by AWS.)
     */
    public val sourceHash: Output<String>?
        get() = javaResource.sourceHash().applyValue({ args0 ->
            args0.map({ args0 ->
                args0
            }).orElse(null)
        })

    /**
     * [Storage Class](https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObject.html#AmazonS3-PutObject-request-header-StorageClass) for the object. Defaults to "`STANDARD`".
     */
    public val storageClass: Output<String>
        get() = javaResource.storageClass().applyValue({ args0 -> args0 })

    /**
     * Map of tags to assign to the object. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    public val tags: Output<Map<String, String>>?
        get() = javaResource.tags().applyValue({ args0 ->
            args0.map({ args0 ->
                args0.map({ args0 ->
                    args0.key.to(args0.value)
                }).toMap()
            }).orElse(null)
        })

    /**
     * Map of tags assigned to the resource, including those inherited from the provider `default_tags` configuration block.
     */
    @Deprecated(
        message = """
  Please use `tags` instead.
  """,
    )
    public val tagsAll: Output<Map<String, String>>
        get() = javaResource.tagsAll().applyValue({ args0 ->
            args0.map({ args0 ->
                args0.key.to(args0.value)
            }).toMap()
        })

    /**
     * Unique version ID value for the object, if bucket versioning is enabled.
     */
    public val versionId: Output<String>
        get() = javaResource.versionId().applyValue({ args0 -> args0 })

    /**
     * Target URL for [website redirect](http://docs.aws.amazon.com/AmazonS3/latest/dev/how-to-page-redirect.html).
     * If no content is provided through `source`, `content` or `content_base64`, then the object will be empty.
     * > **Note:** The provider ignores all leading `/`s in the object's `key` and treats multiple `/`s in the rest of the object's `key` as a single `/`, so values of `/index.html` and `index.html` correspond to the same S3 object as do `first//second///third//` and `first/second/third/`.
     */
    public val websiteRedirect: Output<String>?
        get() = javaResource.websiteRedirect().applyValue({ args0 ->
            args0.map({ args0 ->
                args0
            }).orElse(null)
        })
}

public object BucketObjectv2Mapper : ResourceMapper<BucketObjectv2> {
    override fun supportsMappingOfType(javaResource: Resource): Boolean =
        com.pulumi.aws.s3.BucketObjectv2::class == javaResource::class

    override fun map(javaResource: Resource): BucketObjectv2 = BucketObjectv2(
        javaResource as
            com.pulumi.aws.s3.BucketObjectv2,
    )
}

/**
 * @see [BucketObjectv2].
 * @param name The _unique_ name of the resulting resource.
 * @param block Builder for [BucketObjectv2].
 */
public suspend fun bucketObjectv2(
    name: String,
    block: suspend BucketObjectv2ResourceBuilder.() -> Unit,
): BucketObjectv2 {
    val builder = BucketObjectv2ResourceBuilder()
    builder.name(name)
    block(builder)
    return builder.build()
}

/**
 * @see [BucketObjectv2].
 * @param name The _unique_ name of the resulting resource.
 */
public fun bucketObjectv2(name: String): BucketObjectv2 {
    val builder = BucketObjectv2ResourceBuilder()
    builder.name(name)
    return builder.build()
}
