@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.s3.kotlin

import com.pulumi.core.Output
import com.pulumi.kotlin.KotlinCustomResource
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.ResourceMapper
import com.pulumi.kotlin.options.CustomResourceOptions
import com.pulumi.kotlin.options.CustomResourceOptionsBuilder
import com.pulumi.resources.Resource
import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.Unit

/**
 * Builder for [BucketPolicy].
 */
@PulumiTagMarker
public class BucketPolicyResourceBuilder internal constructor() {
    public var name: String? = null

    public var args: BucketPolicyArgs = BucketPolicyArgs()

    public var opts: CustomResourceOptions = CustomResourceOptions()

    /**
     * @param name The _unique_ name of the resulting resource.
     */
    public fun name(`value`: String) {
        this.name = value
    }

    /**
     * @param block The arguments to use to populate this resource's properties.
     */
    public suspend fun args(block: suspend BucketPolicyArgsBuilder.() -> Unit) {
        val builder = BucketPolicyArgsBuilder()
        block(builder)
        this.args = builder.build()
    }

    /**
     * @param block A bag of options that control this resource's behavior.
     */
    public suspend fun opts(block: suspend CustomResourceOptionsBuilder.() -> Unit) {
        this.opts = com.pulumi.kotlin.options.CustomResourceOptions.opts(block)
    }

    internal fun build(): BucketPolicy {
        val builtJavaResource = com.pulumi.aws.s3.BucketPolicy(
            this.name,
            this.args.toJava(),
            this.opts.toJava(),
        )
        return BucketPolicy(builtJavaResource)
    }
}

/**
 * Attaches a policy to an S3 bucket resource.
 * > Policies can be attached to both S3 general purpose buckets and S3 directory buckets.
 * ## Example Usage
 * ### Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.s3.BucketV2("example", {bucket: "my-tf-test-bucket"});
 * const allowAccessFromAnotherAccount = aws.iam.getPolicyDocumentOutput({
 *     statements: [{
 *         principals: [{
 *             type: "AWS",
 *             identifiers: ["123456789012"],
 *         }],
 *         actions: [
 *             "s3:GetObject",
 *             "s3:ListBucket",
 *         ],
 *         resources: [
 *             example.arn,
 *             pulumi.interpolate`${example.arn}/*`,
 *         ],
 *     }],
 * });
 * const allowAccessFromAnotherAccountBucketPolicy = new aws.s3.BucketPolicy("allow_access_from_another_account", {
 *     bucket: example.id,
 *     policy: allowAccessFromAnotherAccount.apply(allowAccessFromAnotherAccount => allowAccessFromAnotherAccount.json),
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.s3.BucketV2("example", bucket="my-tf-test-bucket")
 * allow_access_from_another_account = aws.iam.get_policy_document_output(statements=[{
 *     "principals": [{
 *         "type": "AWS",
 *         "identifiers": ["123456789012"],
 *     }],
 *     "actions": [
 *         "s3:GetObject",
 *         "s3:ListBucket",
 *     ],
 *     "resources": [
 *         example.arn,
 *         example.arn.apply(lambda arn: f"{arn}/*"),
 *     ],
 * }])
 * allow_access_from_another_account_bucket_policy = aws.s3.BucketPolicy("allow_access_from_another_account",
 *     bucket=example.id,
 *     policy=allow_access_from_another_account.json)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.S3.BucketV2("example", new()
 *     {
 *         Bucket = "my-tf-test-bucket",
 *     });
 *     var allowAccessFromAnotherAccount = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Type = "AWS",
 *                         Identifiers = new[]
 *                         {
 *                             "123456789012",
 *                         },
 *                     },
 *                 },
 *                 Actions = new[]
 *                 {
 *                     "s3:GetObject",
 *                     "s3:ListBucket",
 *                 },
 *                 Resources = new[]
 *                 {
 *                     example.Arn,
 *                     $"{example.Arn}/*",
 *                 },
 *             },
 *         },
 *     });
 *     var allowAccessFromAnotherAccountBucketPolicy = new Aws.S3.BucketPolicy("allow_access_from_another_account", new()
 *     {
 *         Bucket = example.Id,
 *         Policy = allowAccessFromAnotherAccount.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/s3"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := s3.NewBucketV2(ctx, "example", &s3.BucketV2Args{
 * 			Bucket: pulumi.String("my-tf-test-bucket"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		allowAccessFromAnotherAccount := iam.GetPolicyDocumentOutput(ctx, iam.GetPolicyDocumentOutputArgs{
 * 			Statements: iam.GetPolicyDocumentStatementArray{
 * 				&iam.GetPolicyDocumentStatementArgs{
 * 					Principals: iam.GetPolicyDocumentStatementPrincipalArray{
 * 						&iam.GetPolicyDocumentStatementPrincipalArgs{
 * 							Type: pulumi.String("AWS"),
 * 							Identifiers: pulumi.StringArray{
 * 								pulumi.String("123456789012"),
 * 							},
 * 						},
 * 					},
 * 					Actions: pulumi.StringArray{
 * 						pulumi.String("s3:GetObject"),
 * 						pulumi.String("s3:ListBucket"),
 * 					},
 * 					Resources: pulumi.StringArray{
 * 						example.Arn,
 * 						example.Arn.ApplyT(func(arn string) (string, error) {
 * 							return fmt.Sprintf("%v/*", arn), nil
 * 						}).(pulumi.StringOutput),
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		_, err = s3.NewBucketPolicy(ctx, "allow_access_from_another_account", &s3.BucketPolicyArgs{
 * 			Bucket: example.ID(),
 * 			Policy: pulumi.String(allowAccessFromAnotherAccount.ApplyT(func(allowAccessFromAnotherAccount iam.GetPolicyDocumentResult) (*string, error) {
 * 				return &allowAccessFromAnotherAccount.Json, nil
 * 			}).(pulumi.StringPtrOutput)),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.BucketV2;
 * import com.pulumi.aws.s3.BucketV2Args;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.s3.BucketPolicy;
 * import com.pulumi.aws.s3.BucketPolicyArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new BucketV2("example", BucketV2Args.builder()
 *             .bucket("my-tf-test-bucket")
 *             .build());
 *         final var allowAccessFromAnotherAccount = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("AWS")
 *                     .identifiers("123456789012")
 *                     .build())
 *                 .actions(
 *                     "s3:GetObject",
 *                     "s3:ListBucket")
 *                 .resources(
 *                     example.arn(),
 *                     example.arn().applyValue(arn -> String.format("%s/*", arn)))
 *                 .build())
 *             .build());
 *         var allowAccessFromAnotherAccountBucketPolicy = new BucketPolicy("allowAccessFromAnotherAccountBucketPolicy", BucketPolicyArgs.builder()
 *             .bucket(example.id())
 *             .policy(allowAccessFromAnotherAccount.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult).applyValue(allowAccessFromAnotherAccount -> allowAccessFromAnotherAccount.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json())))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:s3:BucketV2
 *     properties:
 *       bucket: my-tf-test-bucket
 *   allowAccessFromAnotherAccountBucketPolicy:
 *     type: aws:s3:BucketPolicy
 *     name: allow_access_from_another_account
 *     properties:
 *       bucket: ${example.id}
 *       policy: ${allowAccessFromAnotherAccount.json}
 * variables:
 *   allowAccessFromAnotherAccount:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - principals:
 *               - type: AWS
 *                 identifiers:
 *                   - '123456789012'
 *             actions:
 *               - s3:GetObject
 *               - s3:ListBucket
 *             resources:
 *               - ${example.arn}
 *               - ${example.arn}/*
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import S3 bucket policies using the bucket name. For example:
 * ```sh
 * $ pulumi import aws:s3/bucketPolicy:BucketPolicy allow_access_from_another_account my-tf-test-bucket
 * ```
 * */*/*/*/*/*/
 */
public class BucketPolicy internal constructor(
    override val javaResource: com.pulumi.aws.s3.BucketPolicy,
) : KotlinCustomResource(javaResource, BucketPolicyMapper) {
    /**
     * Name of the bucket to which to apply the policy.
     */
    public val bucket: Output<String>
        get() = javaResource.bucket().applyValue({ args0 -> args0 })

    /**
     * Text of the policy. Although this is a bucket policy rather than an IAM policy, the `aws.iam.getPolicyDocument` data source may be used, so long as it specifies a principal. For more information about building AWS IAM policy documents, see the AWS IAM Policy Document Guide. Note: Bucket policies are limited to 20 KB in size.
     */
    public val policy: Output<String>
        get() = javaResource.policy().applyValue({ args0 -> args0 })
}

public object BucketPolicyMapper : ResourceMapper<BucketPolicy> {
    override fun supportsMappingOfType(javaResource: Resource): Boolean =
        com.pulumi.aws.s3.BucketPolicy::class == javaResource::class

    override fun map(javaResource: Resource): BucketPolicy = BucketPolicy(
        javaResource as
            com.pulumi.aws.s3.BucketPolicy,
    )
}

/**
 * @see [BucketPolicy].
 * @param name The _unique_ name of the resulting resource.
 * @param block Builder for [BucketPolicy].
 */
public suspend fun bucketPolicy(
    name: String,
    block: suspend BucketPolicyResourceBuilder.() -> Unit,
): BucketPolicy {
    val builder = BucketPolicyResourceBuilder()
    builder.name(name)
    block(builder)
    return builder.build()
}

/**
 * @see [BucketPolicy].
 * @param name The _unique_ name of the resulting resource.
 */
public fun bucketPolicy(name: String): BucketPolicy {
    val builder = BucketPolicyResourceBuilder()
    builder.name(name)
    return builder.build()
}
