@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.s3.kotlin

import com.pulumi.aws.s3.ObjectCopyArgs.builder
import com.pulumi.aws.s3.kotlin.inputs.ObjectCopyGrantArgs
import com.pulumi.aws.s3.kotlin.inputs.ObjectCopyGrantArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a resource for copying an S3 object.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const test = new aws.s3.ObjectCopy("test", {
 *     bucket: "destination_bucket",
 *     key: "destination_key",
 *     source: "source_bucket/source_key",
 *     grants: [{
 *         uri: "http://acs.amazonaws.com/groups/global/AllUsers",
 *         type: "Group",
 *         permissions: ["READ"],
 *     }],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * test = aws.s3.ObjectCopy("test",
 *     bucket="destination_bucket",
 *     key="destination_key",
 *     source="source_bucket/source_key",
 *     grants=[{
 *         "uri": "http://acs.amazonaws.com/groups/global/AllUsers",
 *         "type": "Group",
 *         "permissions": ["READ"],
 *     }])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var test = new Aws.S3.ObjectCopy("test", new()
 *     {
 *         Bucket = "destination_bucket",
 *         Key = "destination_key",
 *         Source = "source_bucket/source_key",
 *         Grants = new[]
 *         {
 *             new Aws.S3.Inputs.ObjectCopyGrantArgs
 *             {
 *                 Uri = "http://acs.amazonaws.com/groups/global/AllUsers",
 *                 Type = "Group",
 *                 Permissions = new[]
 *                 {
 *                     "READ",
 *                 },
 *             },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/s3"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := s3.NewObjectCopy(ctx, "test", &s3.ObjectCopyArgs{
 * 			Bucket: pulumi.String("destination_bucket"),
 * 			Key:    pulumi.String("destination_key"),
 * 			Source: pulumi.String("source_bucket/source_key"),
 * 			Grants: s3.ObjectCopyGrantArray{
 * 				&s3.ObjectCopyGrantArgs{
 * 					Uri:  pulumi.String("http://acs.amazonaws.com/groups/global/AllUsers"),
 * 					Type: pulumi.String("Group"),
 * 					Permissions: pulumi.StringArray{
 * 						pulumi.String("READ"),
 * 					},
 * 				},
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.ObjectCopy;
 * import com.pulumi.aws.s3.ObjectCopyArgs;
 * import com.pulumi.aws.s3.inputs.ObjectCopyGrantArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var test = new ObjectCopy("test", ObjectCopyArgs.builder()
 *             .bucket("destination_bucket")
 *             .key("destination_key")
 *             .source("source_bucket/source_key")
 *             .grants(ObjectCopyGrantArgs.builder()
 *                 .uri("http://acs.amazonaws.com/groups/global/AllUsers")
 *                 .type("Group")
 *                 .permissions("READ")
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   test:
 *     type: aws:s3:ObjectCopy
 *     properties:
 *       bucket: destination_bucket
 *       key: destination_key
 *       source: source_bucket/source_key
 *       grants:
 *         - uri: http://acs.amazonaws.com/groups/global/AllUsers
 *           type: Group
 *           permissions:
 *             - READ
 * ```
 * <!--End PulumiCodeChooser -->
 * @property acl [Canned ACL](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl) to apply. Valid values are `private`, `public-read`, `public-read-write`, `authenticated-read`, `aws-exec-read`, `bucket-owner-read`, and `bucket-owner-full-control`. Conflicts with `grant`.
 * @property bucket Name of the bucket to put the file in.
 * @property bucketKeyEnabled
 * @property cacheControl Specifies caching behavior along the request/reply chain Read [w3c cache_control](http://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.9) for further details.
 * @property checksumAlgorithm Indicates the algorithm used to create the checksum for the object. If a value is specified and the object is encrypted with KMS, you must have permission to use the `kms:Decrypt` action. Valid values: `CRC32`, `CRC32C`, `SHA1`, `SHA256`.
 * @property contentDisposition Specifies presentational information for the object. Read [w3c content_disposition](http://www.w3.org/Protocols/rfc2616/rfc2616-sec19.html#sec19.5.1) for further information.
 * @property contentEncoding Specifies what content encodings have been applied to the object and thus what decoding mechanisms must be applied to obtain the media-type referenced by the Content-Type header field. Read [w3c content encoding](http://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.11) for further information.
 * @property contentLanguage Language the content is in e.g., en-US or en-GB.
 * @property contentType Standard MIME type describing the format of the object data, e.g., `application/octet-stream`. All Valid MIME Types are valid for this input.
 * @property copyIfMatch Copies the object if its entity tag (ETag) matches the specified tag.
 * @property copyIfModifiedSince Copies the object if it has been modified since the specified time, in [RFC3339 format](https://tools.ietf.org/html/rfc3339#section-5.8).
 * @property copyIfNoneMatch Copies the object if its entity tag (ETag) is different than the specified ETag.
 * @property copyIfUnmodifiedSince Copies the object if it hasn't been modified since the specified time, in [RFC3339 format](https://tools.ietf.org/html/rfc3339#section-5.8).
 * @property customerAlgorithm Specifies the algorithm to use to when encrypting the object (for example, AES256).
 * @property customerKey Specifies the customer-provided encryption key for Amazon S3 to use in encrypting data. This value is used to store the object and then it is discarded; Amazon S3 does not store the encryption key. The key must be appropriate for use with the algorithm specified in the x-amz-server-side-encryption-customer-algorithm header.
 * @property customerKeyMd5 Specifies the 128-bit MD5 digest of the encryption key according to RFC 1321. Amazon S3 uses this header for a message integrity check to ensure that the encryption key was transmitted without error.
 * @property expectedBucketOwner Account id of the expected destination bucket owner. If the destination bucket is owned by a different account, the request will fail with an HTTP 403 (Access Denied) error.
 * @property expectedSourceBucketOwner Account id of the expected source bucket owner. If the source bucket is owned by a different account, the request will fail with an HTTP 403 (Access Denied) error.
 * @property expires Date and time at which the object is no longer cacheable, in [RFC3339 format](https://tools.ietf.org/html/rfc3339#section-5.8).
 * @property forceDestroy Allow the object to be deleted by removing any legal hold on any object version. Default is `false`. This value should be set to `true` only if the bucket has S3 object lock enabled.
 * @property grants Configuration block for header grants. Documented below. Conflicts with `acl`.
 * @property key Name of the object once it is in the bucket.
 * @property kmsEncryptionContext Specifies the AWS KMS Encryption Context to use for object encryption. The value is a base64-encoded UTF-8 string holding JSON with the encryption context key-value pairs.
 * @property kmsKeyId Specifies the AWS KMS Key ARN to use for object encryption. This value is a fully qualified **ARN** of the KMS Key. If using `aws.kms.Key`, use the exported `arn` attribute: `kms_key_id = aws_kms_key.foo.arn`
 * @property metadata Map of keys/values to provision metadata (will be automatically prefixed by `x-amz-meta-`, note that only lowercase label are currently supported by the AWS Go API).
 * @property metadataDirective Specifies whether the metadata is copied from the source object or replaced with metadata provided in the request. Valid values are `COPY` and `REPLACE`.
 * @property objectLockLegalHoldStatus The [legal hold](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-legal-holds) status that you want to apply to the specified object. Valid values are `ON` and `OFF`.
 * @property objectLockMode Object lock [retention mode](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-retention-modes) that you want to apply to this object. Valid values are `GOVERNANCE` and `COMPLIANCE`.
 * @property objectLockRetainUntilDate Date and time, in [RFC3339 format](https://tools&#46;ietf&#46;org/html/rfc3339#section-5&#46;8), when this object's object lock will [expire](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-retention-periods).
 * @property requestPayer Confirms that the requester knows that they will be charged for the request. Bucket owners need not specify this parameter in their requests. For information about downloading objects from requester pays buckets, see Downloading Objects in Requestor Pays Buckets (https://docs.aws.amazon.com/AmazonS3/latest/dev/ObjectsinRequesterPaysBuckets.html) in the Amazon S3 Developer Guide. If included, the only valid value is `requester`.
 * @property serverSideEncryption Specifies server-side encryption of the object in S3. Valid values are `AES256` and `aws:kms`.
 * @property source Specifies the source object for the copy operation. You specify the value in one of two formats. For objects not accessed through an access point, specify the name of the source bucket and the key of the source object, separated by a slash (`/`). For example, `testbucket/test1.json`. For objects accessed through access points, specify the ARN of the object as accessed through the access point, in the format `arn:aws:s3:<Region>:<account-id>:accesspoint/<access-point-name>/object/<key>`. For example, `arn:aws:s3:us-west-2:9999912999:accesspoint/my-access-point/object/testbucket/test1.json`.
 * The following arguments are optional:
 * @property sourceCustomerAlgorithm Specifies the algorithm to use when decrypting the source object (for example, AES256).
 * @property sourceCustomerKey Specifies the customer-provided encryption key for Amazon S3 to use to decrypt the source object. The encryption key provided in this header must be one that was used when the source object was created.
 * @property sourceCustomerKeyMd5 Specifies the 128-bit MD5 digest of the encryption key according to RFC 1321. Amazon S3 uses this header for a message integrity check to ensure that the encryption key was transmitted without error.
 * @property storageClass Specifies the desired [storage class](https://docs.aws.amazon.com/AmazonS3/latest/API/API_CopyObject.html#AmazonS3-CopyObject-request-header-StorageClass) for the object. Defaults to `STANDARD`.
 * @property taggingDirective Specifies whether the object tag-set are copied from the source object or replaced with tag-set provided in the request. Valid values are `COPY` and `REPLACE`.
 * @property tags Map of tags to assign to the object. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property websiteRedirect Specifies a target URL for [website redirect](http://docs.aws.amazon.com/AmazonS3/latest/dev/how-to-page-redirect.html).
 */
public data class ObjectCopyArgs(
    public val acl: Output<String>? = null,
    public val bucket: Output<String>? = null,
    public val bucketKeyEnabled: Output<Boolean>? = null,
    public val cacheControl: Output<String>? = null,
    public val checksumAlgorithm: Output<String>? = null,
    public val contentDisposition: Output<String>? = null,
    public val contentEncoding: Output<String>? = null,
    public val contentLanguage: Output<String>? = null,
    public val contentType: Output<String>? = null,
    public val copyIfMatch: Output<String>? = null,
    public val copyIfModifiedSince: Output<String>? = null,
    public val copyIfNoneMatch: Output<String>? = null,
    public val copyIfUnmodifiedSince: Output<String>? = null,
    public val customerAlgorithm: Output<String>? = null,
    public val customerKey: Output<String>? = null,
    public val customerKeyMd5: Output<String>? = null,
    public val expectedBucketOwner: Output<String>? = null,
    public val expectedSourceBucketOwner: Output<String>? = null,
    public val expires: Output<String>? = null,
    public val forceDestroy: Output<Boolean>? = null,
    public val grants: Output<List<ObjectCopyGrantArgs>>? = null,
    public val key: Output<String>? = null,
    public val kmsEncryptionContext: Output<String>? = null,
    public val kmsKeyId: Output<String>? = null,
    public val metadata: Output<Map<String, String>>? = null,
    public val metadataDirective: Output<String>? = null,
    public val objectLockLegalHoldStatus: Output<String>? = null,
    public val objectLockMode: Output<String>? = null,
    public val objectLockRetainUntilDate: Output<String>? = null,
    public val requestPayer: Output<String>? = null,
    public val serverSideEncryption: Output<String>? = null,
    public val source: Output<String>? = null,
    public val sourceCustomerAlgorithm: Output<String>? = null,
    public val sourceCustomerKey: Output<String>? = null,
    public val sourceCustomerKeyMd5: Output<String>? = null,
    public val storageClass: Output<String>? = null,
    public val taggingDirective: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val websiteRedirect: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.s3.ObjectCopyArgs> {
    override fun toJava(): com.pulumi.aws.s3.ObjectCopyArgs =
        com.pulumi.aws.s3.ObjectCopyArgs.builder()
            .acl(acl?.applyValue({ args0 -> args0 }))
            .bucket(bucket?.applyValue({ args0 -> args0 }))
            .bucketKeyEnabled(bucketKeyEnabled?.applyValue({ args0 -> args0 }))
            .cacheControl(cacheControl?.applyValue({ args0 -> args0 }))
            .checksumAlgorithm(checksumAlgorithm?.applyValue({ args0 -> args0 }))
            .contentDisposition(contentDisposition?.applyValue({ args0 -> args0 }))
            .contentEncoding(contentEncoding?.applyValue({ args0 -> args0 }))
            .contentLanguage(contentLanguage?.applyValue({ args0 -> args0 }))
            .contentType(contentType?.applyValue({ args0 -> args0 }))
            .copyIfMatch(copyIfMatch?.applyValue({ args0 -> args0 }))
            .copyIfModifiedSince(copyIfModifiedSince?.applyValue({ args0 -> args0 }))
            .copyIfNoneMatch(copyIfNoneMatch?.applyValue({ args0 -> args0 }))
            .copyIfUnmodifiedSince(copyIfUnmodifiedSince?.applyValue({ args0 -> args0 }))
            .customerAlgorithm(customerAlgorithm?.applyValue({ args0 -> args0 }))
            .customerKey(customerKey?.applyValue({ args0 -> args0 }))
            .customerKeyMd5(customerKeyMd5?.applyValue({ args0 -> args0 }))
            .expectedBucketOwner(expectedBucketOwner?.applyValue({ args0 -> args0 }))
            .expectedSourceBucketOwner(expectedSourceBucketOwner?.applyValue({ args0 -> args0 }))
            .expires(expires?.applyValue({ args0 -> args0 }))
            .forceDestroy(forceDestroy?.applyValue({ args0 -> args0 }))
            .grants(grants?.applyValue({ args0 -> args0.map({ args0 -> args0.let({ args0 -> args0.toJava() }) }) }))
            .key(key?.applyValue({ args0 -> args0 }))
            .kmsEncryptionContext(kmsEncryptionContext?.applyValue({ args0 -> args0 }))
            .kmsKeyId(kmsKeyId?.applyValue({ args0 -> args0 }))
            .metadata(
                metadata?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            )
            .metadataDirective(metadataDirective?.applyValue({ args0 -> args0 }))
            .objectLockLegalHoldStatus(objectLockLegalHoldStatus?.applyValue({ args0 -> args0 }))
            .objectLockMode(objectLockMode?.applyValue({ args0 -> args0 }))
            .objectLockRetainUntilDate(objectLockRetainUntilDate?.applyValue({ args0 -> args0 }))
            .requestPayer(requestPayer?.applyValue({ args0 -> args0 }))
            .serverSideEncryption(serverSideEncryption?.applyValue({ args0 -> args0 }))
            .source(source?.applyValue({ args0 -> args0 }))
            .sourceCustomerAlgorithm(sourceCustomerAlgorithm?.applyValue({ args0 -> args0 }))
            .sourceCustomerKey(sourceCustomerKey?.applyValue({ args0 -> args0 }))
            .sourceCustomerKeyMd5(sourceCustomerKeyMd5?.applyValue({ args0 -> args0 }))
            .storageClass(storageClass?.applyValue({ args0 -> args0 }))
            .taggingDirective(taggingDirective?.applyValue({ args0 -> args0 }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .websiteRedirect(websiteRedirect?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [ObjectCopyArgs].
 */
@PulumiTagMarker
public class ObjectCopyArgsBuilder internal constructor() {
    private var acl: Output<String>? = null

    private var bucket: Output<String>? = null

    private var bucketKeyEnabled: Output<Boolean>? = null

    private var cacheControl: Output<String>? = null

    private var checksumAlgorithm: Output<String>? = null

    private var contentDisposition: Output<String>? = null

    private var contentEncoding: Output<String>? = null

    private var contentLanguage: Output<String>? = null

    private var contentType: Output<String>? = null

    private var copyIfMatch: Output<String>? = null

    private var copyIfModifiedSince: Output<String>? = null

    private var copyIfNoneMatch: Output<String>? = null

    private var copyIfUnmodifiedSince: Output<String>? = null

    private var customerAlgorithm: Output<String>? = null

    private var customerKey: Output<String>? = null

    private var customerKeyMd5: Output<String>? = null

    private var expectedBucketOwner: Output<String>? = null

    private var expectedSourceBucketOwner: Output<String>? = null

    private var expires: Output<String>? = null

    private var forceDestroy: Output<Boolean>? = null

    private var grants: Output<List<ObjectCopyGrantArgs>>? = null

    private var key: Output<String>? = null

    private var kmsEncryptionContext: Output<String>? = null

    private var kmsKeyId: Output<String>? = null

    private var metadata: Output<Map<String, String>>? = null

    private var metadataDirective: Output<String>? = null

    private var objectLockLegalHoldStatus: Output<String>? = null

    private var objectLockMode: Output<String>? = null

    private var objectLockRetainUntilDate: Output<String>? = null

    private var requestPayer: Output<String>? = null

    private var serverSideEncryption: Output<String>? = null

    private var source: Output<String>? = null

    private var sourceCustomerAlgorithm: Output<String>? = null

    private var sourceCustomerKey: Output<String>? = null

    private var sourceCustomerKeyMd5: Output<String>? = null

    private var storageClass: Output<String>? = null

    private var taggingDirective: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var websiteRedirect: Output<String>? = null

    /**
     * @param value [Canned ACL](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl) to apply. Valid values are `private`, `public-read`, `public-read-write`, `authenticated-read`, `aws-exec-read`, `bucket-owner-read`, and `bucket-owner-full-control`. Conflicts with `grant`.
     */
    @JvmName("tcqnksdvndpesbei")
    public suspend fun acl(`value`: Output<String>) {
        this.acl = value
    }

    /**
     * @param value Name of the bucket to put the file in.
     */
    @JvmName("yhlsjvnbatruwyta")
    public suspend fun bucket(`value`: Output<String>) {
        this.bucket = value
    }

    /**
     * @param value
     */
    @JvmName("gbjjvmurplniyvjd")
    public suspend fun bucketKeyEnabled(`value`: Output<Boolean>) {
        this.bucketKeyEnabled = value
    }

    /**
     * @param value Specifies caching behavior along the request/reply chain Read [w3c cache_control](http://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.9) for further details.
     */
    @JvmName("kuvempsmjamxvnma")
    public suspend fun cacheControl(`value`: Output<String>) {
        this.cacheControl = value
    }

    /**
     * @param value Indicates the algorithm used to create the checksum for the object. If a value is specified and the object is encrypted with KMS, you must have permission to use the `kms:Decrypt` action. Valid values: `CRC32`, `CRC32C`, `SHA1`, `SHA256`.
     */
    @JvmName("llmyvwxiwldeuhhk")
    public suspend fun checksumAlgorithm(`value`: Output<String>) {
        this.checksumAlgorithm = value
    }

    /**
     * @param value Specifies presentational information for the object. Read [w3c content_disposition](http://www.w3.org/Protocols/rfc2616/rfc2616-sec19.html#sec19.5.1) for further information.
     */
    @JvmName("pntdmcbqcokrpspd")
    public suspend fun contentDisposition(`value`: Output<String>) {
        this.contentDisposition = value
    }

    /**
     * @param value Specifies what content encodings have been applied to the object and thus what decoding mechanisms must be applied to obtain the media-type referenced by the Content-Type header field. Read [w3c content encoding](http://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.11) for further information.
     */
    @JvmName("wclvkidfgpuvmtgb")
    public suspend fun contentEncoding(`value`: Output<String>) {
        this.contentEncoding = value
    }

    /**
     * @param value Language the content is in e.g., en-US or en-GB.
     */
    @JvmName("htsvsevvcoysawnn")
    public suspend fun contentLanguage(`value`: Output<String>) {
        this.contentLanguage = value
    }

    /**
     * @param value Standard MIME type describing the format of the object data, e.g., `application/octet-stream`. All Valid MIME Types are valid for this input.
     */
    @JvmName("vtadbbowsgmkeolu")
    public suspend fun contentType(`value`: Output<String>) {
        this.contentType = value
    }

    /**
     * @param value Copies the object if its entity tag (ETag) matches the specified tag.
     */
    @JvmName("putanrwmdgbnaxpd")
    public suspend fun copyIfMatch(`value`: Output<String>) {
        this.copyIfMatch = value
    }

    /**
     * @param value Copies the object if it has been modified since the specified time, in [RFC3339 format](https://tools.ietf.org/html/rfc3339#section-5.8).
     */
    @JvmName("erpkegixvyukkwpa")
    public suspend fun copyIfModifiedSince(`value`: Output<String>) {
        this.copyIfModifiedSince = value
    }

    /**
     * @param value Copies the object if its entity tag (ETag) is different than the specified ETag.
     */
    @JvmName("vxstywnjpkhaitdr")
    public suspend fun copyIfNoneMatch(`value`: Output<String>) {
        this.copyIfNoneMatch = value
    }

    /**
     * @param value Copies the object if it hasn't been modified since the specified time, in [RFC3339 format](https://tools.ietf.org/html/rfc3339#section-5.8).
     */
    @JvmName("gjqmdqnpewtnfoxd")
    public suspend fun copyIfUnmodifiedSince(`value`: Output<String>) {
        this.copyIfUnmodifiedSince = value
    }

    /**
     * @param value Specifies the algorithm to use to when encrypting the object (for example, AES256).
     */
    @JvmName("mywtnxkuybvoonij")
    public suspend fun customerAlgorithm(`value`: Output<String>) {
        this.customerAlgorithm = value
    }

    /**
     * @param value Specifies the customer-provided encryption key for Amazon S3 to use in encrypting data. This value is used to store the object and then it is discarded; Amazon S3 does not store the encryption key. The key must be appropriate for use with the algorithm specified in the x-amz-server-side-encryption-customer-algorithm header.
     */
    @JvmName("bnedghaxxyeuucpp")
    public suspend fun customerKey(`value`: Output<String>) {
        this.customerKey = value
    }

    /**
     * @param value Specifies the 128-bit MD5 digest of the encryption key according to RFC 1321. Amazon S3 uses this header for a message integrity check to ensure that the encryption key was transmitted without error.
     */
    @JvmName("wvrgrovbdofrukuw")
    public suspend fun customerKeyMd5(`value`: Output<String>) {
        this.customerKeyMd5 = value
    }

    /**
     * @param value Account id of the expected destination bucket owner. If the destination bucket is owned by a different account, the request will fail with an HTTP 403 (Access Denied) error.
     */
    @JvmName("ykmounrbsxufpuoy")
    public suspend fun expectedBucketOwner(`value`: Output<String>) {
        this.expectedBucketOwner = value
    }

    /**
     * @param value Account id of the expected source bucket owner. If the source bucket is owned by a different account, the request will fail with an HTTP 403 (Access Denied) error.
     */
    @JvmName("hpempjtnqnlbapqk")
    public suspend fun expectedSourceBucketOwner(`value`: Output<String>) {
        this.expectedSourceBucketOwner = value
    }

    /**
     * @param value Date and time at which the object is no longer cacheable, in [RFC3339 format](https://tools.ietf.org/html/rfc3339#section-5.8).
     */
    @JvmName("fxjavcbifgcriglw")
    public suspend fun expires(`value`: Output<String>) {
        this.expires = value
    }

    /**
     * @param value Allow the object to be deleted by removing any legal hold on any object version. Default is `false`. This value should be set to `true` only if the bucket has S3 object lock enabled.
     */
    @JvmName("cthotiueqfrfffqt")
    public suspend fun forceDestroy(`value`: Output<Boolean>) {
        this.forceDestroy = value
    }

    /**
     * @param value Configuration block for header grants. Documented below. Conflicts with `acl`.
     */
    @JvmName("cyetintxdxuhprfc")
    public suspend fun grants(`value`: Output<List<ObjectCopyGrantArgs>>) {
        this.grants = value
    }

    @JvmName("hvdcrtwqehmjcupv")
    public suspend fun grants(vararg values: Output<ObjectCopyGrantArgs>) {
        this.grants = Output.all(values.asList())
    }

    /**
     * @param values Configuration block for header grants. Documented below. Conflicts with `acl`.
     */
    @JvmName("sqeicobmhbmxtsew")
    public suspend fun grants(values: List<Output<ObjectCopyGrantArgs>>) {
        this.grants = Output.all(values)
    }

    /**
     * @param value Name of the object once it is in the bucket.
     */
    @JvmName("msxydkrvcxvmonfj")
    public suspend fun key(`value`: Output<String>) {
        this.key = value
    }

    /**
     * @param value Specifies the AWS KMS Encryption Context to use for object encryption. The value is a base64-encoded UTF-8 string holding JSON with the encryption context key-value pairs.
     */
    @JvmName("wbpwhfmmlnrilsjo")
    public suspend fun kmsEncryptionContext(`value`: Output<String>) {
        this.kmsEncryptionContext = value
    }

    /**
     * @param value Specifies the AWS KMS Key ARN to use for object encryption. This value is a fully qualified **ARN** of the KMS Key. If using `aws.kms.Key`, use the exported `arn` attribute: `kms_key_id = aws_kms_key.foo.arn`
     */
    @JvmName("ucriudfgywlxnnjh")
    public suspend fun kmsKeyId(`value`: Output<String>) {
        this.kmsKeyId = value
    }

    /**
     * @param value Map of keys/values to provision metadata (will be automatically prefixed by `x-amz-meta-`, note that only lowercase label are currently supported by the AWS Go API).
     */
    @JvmName("jbfwygatlcpywhur")
    public suspend fun metadata(`value`: Output<Map<String, String>>) {
        this.metadata = value
    }

    /**
     * @param value Specifies whether the metadata is copied from the source object or replaced with metadata provided in the request. Valid values are `COPY` and `REPLACE`.
     */
    @JvmName("omuokojglrpovnyc")
    public suspend fun metadataDirective(`value`: Output<String>) {
        this.metadataDirective = value
    }

    /**
     * @param value The [legal hold](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-legal-holds) status that you want to apply to the specified object. Valid values are `ON` and `OFF`.
     */
    @JvmName("xltpfixpqmtfqjfj")
    public suspend fun objectLockLegalHoldStatus(`value`: Output<String>) {
        this.objectLockLegalHoldStatus = value
    }

    /**
     * @param value Object lock [retention mode](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-retention-modes) that you want to apply to this object. Valid values are `GOVERNANCE` and `COMPLIANCE`.
     */
    @JvmName("jaxuhqrjxdqwddhs")
    public suspend fun objectLockMode(`value`: Output<String>) {
        this.objectLockMode = value
    }

    /**
     * @param value Date and time, in [RFC3339 format](https://tools&#46;ietf&#46;org/html/rfc3339#section-5&#46;8), when this object's object lock will [expire](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-retention-periods).
     */
    @JvmName("rwwdyvhfgushrxhh")
    public suspend fun objectLockRetainUntilDate(`value`: Output<String>) {
        this.objectLockRetainUntilDate = value
    }

    /**
     * @param value Confirms that the requester knows that they will be charged for the request. Bucket owners need not specify this parameter in their requests. For information about downloading objects from requester pays buckets, see Downloading Objects in Requestor Pays Buckets (https://docs.aws.amazon.com/AmazonS3/latest/dev/ObjectsinRequesterPaysBuckets.html) in the Amazon S3 Developer Guide. If included, the only valid value is `requester`.
     */
    @JvmName("dityfqdtsalivckr")
    public suspend fun requestPayer(`value`: Output<String>) {
        this.requestPayer = value
    }

    /**
     * @param value Specifies server-side encryption of the object in S3. Valid values are `AES256` and `aws:kms`.
     */
    @JvmName("geywcbnvcdkafcni")
    public suspend fun serverSideEncryption(`value`: Output<String>) {
        this.serverSideEncryption = value
    }

    /**
     * @param value Specifies the source object for the copy operation. You specify the value in one of two formats. For objects not accessed through an access point, specify the name of the source bucket and the key of the source object, separated by a slash (`/`). For example, `testbucket/test1.json`. For objects accessed through access points, specify the ARN of the object as accessed through the access point, in the format `arn:aws:s3:<Region>:<account-id>:accesspoint/<access-point-name>/object/<key>`. For example, `arn:aws:s3:us-west-2:9999912999:accesspoint/my-access-point/object/testbucket/test1.json`.
     * The following arguments are optional:
     */
    @JvmName("sabueevrbdvbkvln")
    public suspend fun source(`value`: Output<String>) {
        this.source = value
    }

    /**
     * @param value Specifies the algorithm to use when decrypting the source object (for example, AES256).
     */
    @JvmName("bidhnbmjrrloqtnr")
    public suspend fun sourceCustomerAlgorithm(`value`: Output<String>) {
        this.sourceCustomerAlgorithm = value
    }

    /**
     * @param value Specifies the customer-provided encryption key for Amazon S3 to use to decrypt the source object. The encryption key provided in this header must be one that was used when the source object was created.
     */
    @JvmName("midtkncaqmxmcian")
    public suspend fun sourceCustomerKey(`value`: Output<String>) {
        this.sourceCustomerKey = value
    }

    /**
     * @param value Specifies the 128-bit MD5 digest of the encryption key according to RFC 1321. Amazon S3 uses this header for a message integrity check to ensure that the encryption key was transmitted without error.
     */
    @JvmName("mxiiqefeqrditber")
    public suspend fun sourceCustomerKeyMd5(`value`: Output<String>) {
        this.sourceCustomerKeyMd5 = value
    }

    /**
     * @param value Specifies the desired [storage class](https://docs.aws.amazon.com/AmazonS3/latest/API/API_CopyObject.html#AmazonS3-CopyObject-request-header-StorageClass) for the object. Defaults to `STANDARD`.
     */
    @JvmName("jmockqklkrdhesvb")
    public suspend fun storageClass(`value`: Output<String>) {
        this.storageClass = value
    }

    /**
     * @param value Specifies whether the object tag-set are copied from the source object or replaced with tag-set provided in the request. Valid values are `COPY` and `REPLACE`.
     */
    @JvmName("poycadydvpmatygq")
    public suspend fun taggingDirective(`value`: Output<String>) {
        this.taggingDirective = value
    }

    /**
     * @param value Map of tags to assign to the object. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("cipkybvxgbhtkbov")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Specifies a target URL for [website redirect](http://docs.aws.amazon.com/AmazonS3/latest/dev/how-to-page-redirect.html).
     */
    @JvmName("kowlmeyhobnefhjx")
    public suspend fun websiteRedirect(`value`: Output<String>) {
        this.websiteRedirect = value
    }

    /**
     * @param value [Canned ACL](https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl) to apply. Valid values are `private`, `public-read`, `public-read-write`, `authenticated-read`, `aws-exec-read`, `bucket-owner-read`, and `bucket-owner-full-control`. Conflicts with `grant`.
     */
    @JvmName("ncbrdcnraaombkli")
    public suspend fun acl(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.acl = mapped
    }

    /**
     * @param value Name of the bucket to put the file in.
     */
    @JvmName("ihjwsqugwuhdtrrd")
    public suspend fun bucket(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.bucket = mapped
    }

    /**
     * @param value
     */
    @JvmName("khfpuvatgfuavqxs")
    public suspend fun bucketKeyEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.bucketKeyEnabled = mapped
    }

    /**
     * @param value Specifies caching behavior along the request/reply chain Read [w3c cache_control](http://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.9) for further details.
     */
    @JvmName("csysgcongrpstgml")
    public suspend fun cacheControl(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.cacheControl = mapped
    }

    /**
     * @param value Indicates the algorithm used to create the checksum for the object. If a value is specified and the object is encrypted with KMS, you must have permission to use the `kms:Decrypt` action. Valid values: `CRC32`, `CRC32C`, `SHA1`, `SHA256`.
     */
    @JvmName("cogglinihgaqrean")
    public suspend fun checksumAlgorithm(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.checksumAlgorithm = mapped
    }

    /**
     * @param value Specifies presentational information for the object. Read [w3c content_disposition](http://www.w3.org/Protocols/rfc2616/rfc2616-sec19.html#sec19.5.1) for further information.
     */
    @JvmName("qxbyqeotwetafuhx")
    public suspend fun contentDisposition(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.contentDisposition = mapped
    }

    /**
     * @param value Specifies what content encodings have been applied to the object and thus what decoding mechanisms must be applied to obtain the media-type referenced by the Content-Type header field. Read [w3c content encoding](http://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.11) for further information.
     */
    @JvmName("ulgubiceodmbuyyv")
    public suspend fun contentEncoding(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.contentEncoding = mapped
    }

    /**
     * @param value Language the content is in e.g., en-US or en-GB.
     */
    @JvmName("nsllkivrfyufmwsp")
    public suspend fun contentLanguage(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.contentLanguage = mapped
    }

    /**
     * @param value Standard MIME type describing the format of the object data, e.g., `application/octet-stream`. All Valid MIME Types are valid for this input.
     */
    @JvmName("xysfubioseywonbc")
    public suspend fun contentType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.contentType = mapped
    }

    /**
     * @param value Copies the object if its entity tag (ETag) matches the specified tag.
     */
    @JvmName("cmxisrimwvcddhgo")
    public suspend fun copyIfMatch(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.copyIfMatch = mapped
    }

    /**
     * @param value Copies the object if it has been modified since the specified time, in [RFC3339 format](https://tools.ietf.org/html/rfc3339#section-5.8).
     */
    @JvmName("yyjivjgsqulxxoat")
    public suspend fun copyIfModifiedSince(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.copyIfModifiedSince = mapped
    }

    /**
     * @param value Copies the object if its entity tag (ETag) is different than the specified ETag.
     */
    @JvmName("enceulbtfmvcxyuk")
    public suspend fun copyIfNoneMatch(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.copyIfNoneMatch = mapped
    }

    /**
     * @param value Copies the object if it hasn't been modified since the specified time, in [RFC3339 format](https://tools.ietf.org/html/rfc3339#section-5.8).
     */
    @JvmName("ipwgxsuiywtpissg")
    public suspend fun copyIfUnmodifiedSince(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.copyIfUnmodifiedSince = mapped
    }

    /**
     * @param value Specifies the algorithm to use to when encrypting the object (for example, AES256).
     */
    @JvmName("kowkosnksvcgevxg")
    public suspend fun customerAlgorithm(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.customerAlgorithm = mapped
    }

    /**
     * @param value Specifies the customer-provided encryption key for Amazon S3 to use in encrypting data. This value is used to store the object and then it is discarded; Amazon S3 does not store the encryption key. The key must be appropriate for use with the algorithm specified in the x-amz-server-side-encryption-customer-algorithm header.
     */
    @JvmName("roqoswrjdfiyyhpg")
    public suspend fun customerKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.customerKey = mapped
    }

    /**
     * @param value Specifies the 128-bit MD5 digest of the encryption key according to RFC 1321. Amazon S3 uses this header for a message integrity check to ensure that the encryption key was transmitted without error.
     */
    @JvmName("liwgnvdvnrfgmlvl")
    public suspend fun customerKeyMd5(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.customerKeyMd5 = mapped
    }

    /**
     * @param value Account id of the expected destination bucket owner. If the destination bucket is owned by a different account, the request will fail with an HTTP 403 (Access Denied) error.
     */
    @JvmName("tsgqbipaatrcttnb")
    public suspend fun expectedBucketOwner(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.expectedBucketOwner = mapped
    }

    /**
     * @param value Account id of the expected source bucket owner. If the source bucket is owned by a different account, the request will fail with an HTTP 403 (Access Denied) error.
     */
    @JvmName("yoeihgcsxdmoovut")
    public suspend fun expectedSourceBucketOwner(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.expectedSourceBucketOwner = mapped
    }

    /**
     * @param value Date and time at which the object is no longer cacheable, in [RFC3339 format](https://tools.ietf.org/html/rfc3339#section-5.8).
     */
    @JvmName("cjxkibqjjnvwvvim")
    public suspend fun expires(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.expires = mapped
    }

    /**
     * @param value Allow the object to be deleted by removing any legal hold on any object version. Default is `false`. This value should be set to `true` only if the bucket has S3 object lock enabled.
     */
    @JvmName("afbcaugofjtstmmi")
    public suspend fun forceDestroy(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.forceDestroy = mapped
    }

    /**
     * @param value Configuration block for header grants. Documented below. Conflicts with `acl`.
     */
    @JvmName("wvajknyncmotulop")
    public suspend fun grants(`value`: List<ObjectCopyGrantArgs>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.grants = mapped
    }

    /**
     * @param argument Configuration block for header grants. Documented below. Conflicts with `acl`.
     */
    @JvmName("hdybdrbjxnjpmdwy")
    public suspend fun grants(argument: List<suspend ObjectCopyGrantArgsBuilder.() -> Unit>) {
        val toBeMapped = argument.toList().map {
            ObjectCopyGrantArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.grants = mapped
    }

    /**
     * @param argument Configuration block for header grants. Documented below. Conflicts with `acl`.
     */
    @JvmName("hcwefvgmhdfrbkfb")
    public suspend fun grants(vararg argument: suspend ObjectCopyGrantArgsBuilder.() -> Unit) {
        val toBeMapped = argument.toList().map {
            ObjectCopyGrantArgsBuilder().applySuspend {
                it()
            }.build()
        }
        val mapped = of(toBeMapped)
        this.grants = mapped
    }

    /**
     * @param argument Configuration block for header grants. Documented below. Conflicts with `acl`.
     */
    @JvmName("resadtrcnocvdvue")
    public suspend fun grants(argument: suspend ObjectCopyGrantArgsBuilder.() -> Unit) {
        val toBeMapped = listOf(ObjectCopyGrantArgsBuilder().applySuspend { argument() }.build())
        val mapped = of(toBeMapped)
        this.grants = mapped
    }

    /**
     * @param values Configuration block for header grants. Documented below. Conflicts with `acl`.
     */
    @JvmName("wcecpnxtqpsopehd")
    public suspend fun grants(vararg values: ObjectCopyGrantArgs) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.grants = mapped
    }

    /**
     * @param value Name of the object once it is in the bucket.
     */
    @JvmName("renpcffwtyjgeauh")
    public suspend fun key(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.key = mapped
    }

    /**
     * @param value Specifies the AWS KMS Encryption Context to use for object encryption. The value is a base64-encoded UTF-8 string holding JSON with the encryption context key-value pairs.
     */
    @JvmName("nlhnapybrtqtcpxb")
    public suspend fun kmsEncryptionContext(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.kmsEncryptionContext = mapped
    }

    /**
     * @param value Specifies the AWS KMS Key ARN to use for object encryption. This value is a fully qualified **ARN** of the KMS Key. If using `aws.kms.Key`, use the exported `arn` attribute: `kms_key_id = aws_kms_key.foo.arn`
     */
    @JvmName("cedbdqoutnvssrds")
    public suspend fun kmsKeyId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.kmsKeyId = mapped
    }

    /**
     * @param value Map of keys/values to provision metadata (will be automatically prefixed by `x-amz-meta-`, note that only lowercase label are currently supported by the AWS Go API).
     */
    @JvmName("rafdmtxhkyccimmc")
    public suspend fun metadata(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.metadata = mapped
    }

    /**
     * @param values Map of keys/values to provision metadata (will be automatically prefixed by `x-amz-meta-`, note that only lowercase label are currently supported by the AWS Go API).
     */
    @JvmName("xoehmutiysklovcf")
    public fun metadata(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.metadata = mapped
    }

    /**
     * @param value Specifies whether the metadata is copied from the source object or replaced with metadata provided in the request. Valid values are `COPY` and `REPLACE`.
     */
    @JvmName("mqyltwpsvjcawvec")
    public suspend fun metadataDirective(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.metadataDirective = mapped
    }

    /**
     * @param value The [legal hold](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-legal-holds) status that you want to apply to the specified object. Valid values are `ON` and `OFF`.
     */
    @JvmName("qygisvebbldvcgbt")
    public suspend fun objectLockLegalHoldStatus(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.objectLockLegalHoldStatus = mapped
    }

    /**
     * @param value Object lock [retention mode](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-retention-modes) that you want to apply to this object. Valid values are `GOVERNANCE` and `COMPLIANCE`.
     */
    @JvmName("tsyepgqvealvlken")
    public suspend fun objectLockMode(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.objectLockMode = mapped
    }

    /**
     * @param value Date and time, in [RFC3339 format](https://tools&#46;ietf&#46;org/html/rfc3339#section-5&#46;8), when this object's object lock will [expire](https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-retention-periods).
     */
    @JvmName("qgorndqstotwrgah")
    public suspend fun objectLockRetainUntilDate(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.objectLockRetainUntilDate = mapped
    }

    /**
     * @param value Confirms that the requester knows that they will be charged for the request. Bucket owners need not specify this parameter in their requests. For information about downloading objects from requester pays buckets, see Downloading Objects in Requestor Pays Buckets (https://docs.aws.amazon.com/AmazonS3/latest/dev/ObjectsinRequesterPaysBuckets.html) in the Amazon S3 Developer Guide. If included, the only valid value is `requester`.
     */
    @JvmName("nfexmwkxxwulblog")
    public suspend fun requestPayer(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.requestPayer = mapped
    }

    /**
     * @param value Specifies server-side encryption of the object in S3. Valid values are `AES256` and `aws:kms`.
     */
    @JvmName("jmcrkkokymkvrnev")
    public suspend fun serverSideEncryption(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.serverSideEncryption = mapped
    }

    /**
     * @param value Specifies the source object for the copy operation. You specify the value in one of two formats. For objects not accessed through an access point, specify the name of the source bucket and the key of the source object, separated by a slash (`/`). For example, `testbucket/test1.json`. For objects accessed through access points, specify the ARN of the object as accessed through the access point, in the format `arn:aws:s3:<Region>:<account-id>:accesspoint/<access-point-name>/object/<key>`. For example, `arn:aws:s3:us-west-2:9999912999:accesspoint/my-access-point/object/testbucket/test1.json`.
     * The following arguments are optional:
     */
    @JvmName("ynymyhujaacyhamb")
    public suspend fun source(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.source = mapped
    }

    /**
     * @param value Specifies the algorithm to use when decrypting the source object (for example, AES256).
     */
    @JvmName("kiocofmrhxetdbsx")
    public suspend fun sourceCustomerAlgorithm(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceCustomerAlgorithm = mapped
    }

    /**
     * @param value Specifies the customer-provided encryption key for Amazon S3 to use to decrypt the source object. The encryption key provided in this header must be one that was used when the source object was created.
     */
    @JvmName("jxekjnyyicjjianr")
    public suspend fun sourceCustomerKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceCustomerKey = mapped
    }

    /**
     * @param value Specifies the 128-bit MD5 digest of the encryption key according to RFC 1321. Amazon S3 uses this header for a message integrity check to ensure that the encryption key was transmitted without error.
     */
    @JvmName("edrlgceajltxtrav")
    public suspend fun sourceCustomerKeyMd5(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceCustomerKeyMd5 = mapped
    }

    /**
     * @param value Specifies the desired [storage class](https://docs.aws.amazon.com/AmazonS3/latest/API/API_CopyObject.html#AmazonS3-CopyObject-request-header-StorageClass) for the object. Defaults to `STANDARD`.
     */
    @JvmName("pynbogomrftqlrpy")
    public suspend fun storageClass(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.storageClass = mapped
    }

    /**
     * @param value Specifies whether the object tag-set are copied from the source object or replaced with tag-set provided in the request. Valid values are `COPY` and `REPLACE`.
     */
    @JvmName("cywbmkdifydnwynd")
    public suspend fun taggingDirective(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.taggingDirective = mapped
    }

    /**
     * @param value Map of tags to assign to the object. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("fdxdhkbsojmxlhwk")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Map of tags to assign to the object. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("kononhestrknxtxq")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Specifies a target URL for [website redirect](http://docs.aws.amazon.com/AmazonS3/latest/dev/how-to-page-redirect.html).
     */
    @JvmName("yjodlrbxxqduikop")
    public suspend fun websiteRedirect(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.websiteRedirect = mapped
    }

    internal fun build(): ObjectCopyArgs = ObjectCopyArgs(
        acl = acl,
        bucket = bucket,
        bucketKeyEnabled = bucketKeyEnabled,
        cacheControl = cacheControl,
        checksumAlgorithm = checksumAlgorithm,
        contentDisposition = contentDisposition,
        contentEncoding = contentEncoding,
        contentLanguage = contentLanguage,
        contentType = contentType,
        copyIfMatch = copyIfMatch,
        copyIfModifiedSince = copyIfModifiedSince,
        copyIfNoneMatch = copyIfNoneMatch,
        copyIfUnmodifiedSince = copyIfUnmodifiedSince,
        customerAlgorithm = customerAlgorithm,
        customerKey = customerKey,
        customerKeyMd5 = customerKeyMd5,
        expectedBucketOwner = expectedBucketOwner,
        expectedSourceBucketOwner = expectedSourceBucketOwner,
        expires = expires,
        forceDestroy = forceDestroy,
        grants = grants,
        key = key,
        kmsEncryptionContext = kmsEncryptionContext,
        kmsKeyId = kmsKeyId,
        metadata = metadata,
        metadataDirective = metadataDirective,
        objectLockLegalHoldStatus = objectLockLegalHoldStatus,
        objectLockMode = objectLockMode,
        objectLockRetainUntilDate = objectLockRetainUntilDate,
        requestPayer = requestPayer,
        serverSideEncryption = serverSideEncryption,
        source = source,
        sourceCustomerAlgorithm = sourceCustomerAlgorithm,
        sourceCustomerKey = sourceCustomerKey,
        sourceCustomerKeyMd5 = sourceCustomerKeyMd5,
        storageClass = storageClass,
        taggingDirective = taggingDirective,
        tags = tags,
        websiteRedirect = websiteRedirect,
    )
}
