@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.s3control.kotlin

import com.pulumi.aws.s3control.AccessPointPolicyArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Provides a resource to manage an S3 Access Point resource policy.
 * > **NOTE on Access Points and Access Point Policies:** The provider provides both a standalone Access Point Policy resource and an Access Point resource with a resource policy defined in-line. You cannot use an Access Point with in-line resource policy in conjunction with an Access Point Policy resource. Doing so will cause a conflict of policies and will overwrite the access point's resource policy.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.s3.BucketV2("example", {bucket: "example"});
 * const exampleAccessPoint = new aws.s3.AccessPoint("example", {
 *     bucket: example.id,
 *     name: "example",
 *     publicAccessBlockConfiguration: {
 *         blockPublicAcls: true,
 *         blockPublicPolicy: false,
 *         ignorePublicAcls: true,
 *         restrictPublicBuckets: false,
 *     },
 * });
 * const exampleAccessPointPolicy = new aws.s3control.AccessPointPolicy("example", {
 *     accessPointArn: exampleAccessPoint.arn,
 *     policy: pulumi.jsonStringify({
 *         Version: "2008-10-17",
 *         Statement: [{
 *             Effect: "Allow",
 *             Action: "s3:GetObjectTagging",
 *             Principal: {
 *                 AWS: "*",
 *             },
 *             Resource: pulumi.interpolate`${exampleAccessPoint.arn}/object/*`,
 *         }],
 *     }),
 * });
 * ```
 * ```python
 * import pulumi
 * import json
 * import pulumi_aws as aws
 * example = aws.s3.BucketV2("example", bucket="example")
 * example_access_point = aws.s3.AccessPoint("example",
 *     bucket=example.id,
 *     name="example",
 *     public_access_block_configuration={
 *         "block_public_acls": True,
 *         "block_public_policy": False,
 *         "ignore_public_acls": True,
 *         "restrict_public_buckets": False,
 *     })
 * example_access_point_policy = aws.s3control.AccessPointPolicy("example",
 *     access_point_arn=example_access_point.arn,
 *     policy=pulumi.Output.json_dumps({
 *         "Version": "2008-10-17",
 *         "Statement": [{
 *             "Effect": "Allow",
 *             "Action": "s3:GetObjectTagging",
 *             "Principal": {
 *                 "AWS": "*",
 *             },
 *             "Resource": example_access_point.arn.apply(lambda arn: f"{arn}/object/*"),
 *         }],
 *     }))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using System.Text.Json;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.S3.BucketV2("example", new()
 *     {
 *         Bucket = "example",
 *     });
 *     var exampleAccessPoint = new Aws.S3.AccessPoint("example", new()
 *     {
 *         Bucket = example.Id,
 *         Name = "example",
 *         PublicAccessBlockConfiguration = new Aws.S3.Inputs.AccessPointPublicAccessBlockConfigurationArgs
 *         {
 *             BlockPublicAcls = true,
 *             BlockPublicPolicy = false,
 *             IgnorePublicAcls = true,
 *             RestrictPublicBuckets = false,
 *         },
 *     });
 *     var exampleAccessPointPolicy = new Aws.S3Control.AccessPointPolicy("example", new()
 *     {
 *         AccessPointArn = exampleAccessPoint.Arn,
 *         Policy = Output.JsonSerialize(Output.Create(new Dictionary<string, object?>
 *         {
 *             ["Version"] = "2008-10-17",
 *             ["Statement"] = new[]
 *             {
 *                 new Dictionary<string, object?>
 *                 {
 *                     ["Effect"] = "Allow",
 *                     ["Action"] = "s3:GetObjectTagging",
 *                     ["Principal"] = new Dictionary<string, object?>
 *                     {
 *                         ["AWS"] = "*",
 *                     },
 *                     ["Resource"] = exampleAccessPoint.Arn.Apply(arn => $"{arn}/object/*"),
 *                 },
 *             },
 *         })),
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"encoding/json"
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/s3"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/s3control"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := s3.NewBucketV2(ctx, "example", &s3.BucketV2Args{
 * 			Bucket: pulumi.String("example"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleAccessPoint, err := s3.NewAccessPoint(ctx, "example", &s3.AccessPointArgs{
 * 			Bucket: example.ID(),
 * 			Name:   pulumi.String("example"),
 * 			PublicAccessBlockConfiguration: &s3.AccessPointPublicAccessBlockConfigurationArgs{
 * 				BlockPublicAcls:       pulumi.Bool(true),
 * 				BlockPublicPolicy:     pulumi.Bool(false),
 * 				IgnorePublicAcls:      pulumi.Bool(true),
 * 				RestrictPublicBuckets: pulumi.Bool(false),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = s3control.NewAccessPointPolicy(ctx, "example", &s3control.AccessPointPolicyArgs{
 * 			AccessPointArn: exampleAccessPoint.Arn,
 * 			Policy: exampleAccessPoint.Arn.ApplyT(func(arn string) (pulumi.String, error) {
 * 				var _zero pulumi.String
 * 				tmpJSON0, err := json.Marshal(map[string]interface{}{
 * 					"Version": "2008-10-17",
 * 					"Statement": []map[string]interface{}{
 * 						map[string]interface{}{
 * 							"Effect": "Allow",
 * 							"Action": "s3:GetObjectTagging",
 * 							"Principal": map[string]interface{}{
 * 								"AWS": "*",
 * 							},
 * 							"Resource": fmt.Sprintf("%v/object/*", arn),
 * 						},
 * 					},
 * 				})
 * 				if err != nil {
 * 					return _zero, err
 * 				}
 * 				json0 := string(tmpJSON0)
 * 				return pulumi.String(json0), nil
 * 			}).(pulumi.StringOutput),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.s3.BucketV2;
 * import com.pulumi.aws.s3.BucketV2Args;
 * import com.pulumi.aws.s3.AccessPoint;
 * import com.pulumi.aws.s3.AccessPointArgs;
 * import com.pulumi.aws.s3.inputs.AccessPointPublicAccessBlockConfigurationArgs;
 * import com.pulumi.aws.s3control.AccessPointPolicy;
 * import com.pulumi.aws.s3control.AccessPointPolicyArgs;
 * import static com.pulumi.codegen.internal.Serialization.*;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new BucketV2("example", BucketV2Args.builder()
 *             .bucket("example")
 *             .build());
 *         var exampleAccessPoint = new AccessPoint("exampleAccessPoint", AccessPointArgs.builder()
 *             .bucket(example.id())
 *             .name("example")
 *             .publicAccessBlockConfiguration(AccessPointPublicAccessBlockConfigurationArgs.builder()
 *                 .blockPublicAcls(true)
 *                 .blockPublicPolicy(false)
 *                 .ignorePublicAcls(true)
 *                 .restrictPublicBuckets(false)
 *                 .build())
 *             .build());
 *         var exampleAccessPointPolicy = new AccessPointPolicy("exampleAccessPointPolicy", AccessPointPolicyArgs.builder()
 *             .accessPointArn(exampleAccessPoint.arn())
 *             .policy(exampleAccessPoint.arn().applyValue(arn -> serializeJson(
 *                 jsonObject(
 *                     jsonProperty("Version", "2008-10-17"),
 *                     jsonProperty("Statement", jsonArray(jsonObject(
 *                         jsonProperty("Effect", "Allow"),
 *                         jsonProperty("Action", "s3:GetObjectTagging"),
 *                         jsonProperty("Principal", jsonObject(
 *                             jsonProperty("AWS", "*")
 *                         )),
 *                         jsonProperty("Resource", String.format("%s/object/*", arn))
 *                     )))
 *                 ))))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:s3:BucketV2
 *     properties:
 *       bucket: example
 *   exampleAccessPoint:
 *     type: aws:s3:AccessPoint
 *     name: example
 *     properties:
 *       bucket: ${example.id}
 *       name: example
 *       publicAccessBlockConfiguration:
 *         blockPublicAcls: true
 *         blockPublicPolicy: false
 *         ignorePublicAcls: true
 *         restrictPublicBuckets: false
 *   exampleAccessPointPolicy:
 *     type: aws:s3control:AccessPointPolicy
 *     name: example
 *     properties:
 *       accessPointArn: ${exampleAccessPoint.arn}
 *       policy:
 *         fn::toJSON:
 *           Version: 2008-10-17
 *           Statement:
 *             - Effect: Allow
 *               Action: s3:GetObjectTagging
 *               Principal:
 *                 AWS: '*'
 *               Resource: ${exampleAccessPoint.arn}/object/*
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Access Point policies using the `access_point_arn`. For example:
 * ```sh
 * $ pulumi import aws:s3control/accessPointPolicy:AccessPointPolicy example arn:aws:s3:us-west-2:123456789012:accesspoint/example
 * ```
 * @property accessPointArn The ARN of the access point that you want to associate with the specified policy.
 * @property policy The policy that you want to apply to the specified access point.
 * */*/*/*/*/*/
 */
public data class AccessPointPolicyArgs(
    public val accessPointArn: Output<String>? = null,
    public val policy: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.s3control.AccessPointPolicyArgs> {
    override fun toJava(): com.pulumi.aws.s3control.AccessPointPolicyArgs =
        com.pulumi.aws.s3control.AccessPointPolicyArgs.builder()
            .accessPointArn(accessPointArn?.applyValue({ args0 -> args0 }))
            .policy(policy?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [AccessPointPolicyArgs].
 */
@PulumiTagMarker
public class AccessPointPolicyArgsBuilder internal constructor() {
    private var accessPointArn: Output<String>? = null

    private var policy: Output<String>? = null

    /**
     * @param value The ARN of the access point that you want to associate with the specified policy.
     */
    @JvmName("njlgjsidvxoqcvcb")
    public suspend fun accessPointArn(`value`: Output<String>) {
        this.accessPointArn = value
    }

    /**
     * @param value The policy that you want to apply to the specified access point.
     */
    @JvmName("fskwdrpirpsriylm")
    public suspend fun policy(`value`: Output<String>) {
        this.policy = value
    }

    /**
     * @param value The ARN of the access point that you want to associate with the specified policy.
     */
    @JvmName("jykgkqpvnsebvcnf")
    public suspend fun accessPointArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.accessPointArn = mapped
    }

    /**
     * @param value The policy that you want to apply to the specified access point.
     */
    @JvmName("ljyxubchtdhdoogi")
    public suspend fun policy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.policy = mapped
    }

    internal fun build(): AccessPointPolicyArgs = AccessPointPolicyArgs(
        accessPointArn = accessPointArn,
        policy = policy,
    )
}
