@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.ses.kotlin

import com.pulumi.aws.ses.IdentityPolicyArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Manages a SES Identity Policy. More information about SES Sending Authorization Policies can be found in the [SES Developer Guide](https://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-policies.html).
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const exampleDomainIdentity = new aws.ses.DomainIdentity("example", {domain: "example.com"});
 * const example = aws.iam.getPolicyDocumentOutput({
 *     statements: [{
 *         actions: [
 *             "SES:SendEmail",
 *             "SES:SendRawEmail",
 *         ],
 *         resources: [exampleDomainIdentity&#46;arn],
 *         principals: [{
 *             identifiers: ["*"],
 *             type: "AWS",
 *         }],
 *     }],
 * });
 * const exampleIdentityPolicy = new aws.ses.IdentityPolicy("example", {
 *     identity: exampleDomainIdentity.arn,
 *     name: "example",
 *     policy: example.apply(example => example.json),
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example_domain_identity = aws.ses.DomainIdentity("example", domain="example.com")
 * example = aws.iam.get_policy_document_output(statements=[{
 *     "actions": [
 *         "SES:SendEmail",
 *         "SES:SendRawEmail",
 *     ],
 *     "resources": [example_domain_identity&#46;arn],
 *     "principals": [{
 *         "identifiers": ["*"],
 *         "type": "AWS",
 *     }],
 * }])
 * example_identity_policy = aws.ses.IdentityPolicy("example",
 *     identity=example_domain_identity.arn,
 *     name="example",
 *     policy=example.json)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var exampleDomainIdentity = new Aws.Ses.DomainIdentity("example", new()
 *     {
 *         Domain = "example.com",
 *     });
 *     var example = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Actions = new[]
 *                 {
 *                     "SES:SendEmail",
 *                     "SES:SendRawEmail",
 *                 },
 *                 Resources = new[]
 *                 {
 *                     exampleDomainIdentity.Arn,
 *                 },
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Identifiers = new[]
 *                         {
 *                             "*",
 *                         },
 *                         Type = "AWS",
 *                     },
 *                 },
 *             },
 *         },
 *     });
 *     var exampleIdentityPolicy = new Aws.Ses.IdentityPolicy("example", new()
 *     {
 *         Identity = exampleDomainIdentity.Arn,
 *         Name = "example",
 *         Policy = example.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ses"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		exampleDomainIdentity, err := ses.NewDomainIdentity(ctx, "example", &ses.DomainIdentityArgs{
 * 			Domain: pulumi.String("example.com"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example := iam.GetPolicyDocumentOutput(ctx, iam.GetPolicyDocumentOutputArgs{
 * 			Statements: iam.GetPolicyDocumentStatementArray{
 * 				&iam.GetPolicyDocumentStatementArgs{
 * 					Actions: pulumi.StringArray{
 * 						pulumi.String("SES:SendEmail"),
 * 						pulumi.String("SES:SendRawEmail"),
 * 					},
 * 					Resources: pulumi.StringArray{
 * 						exampleDomainIdentity.Arn,
 * 					},
 * 					Principals: iam.GetPolicyDocumentStatementPrincipalArray{
 * 						&iam.GetPolicyDocumentStatementPrincipalArgs{
 * 							Identifiers: pulumi.StringArray{
 * 								pulumi.String("*"),
 * 							},
 * 							Type: pulumi.String("AWS"),
 * 						},
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		_, err = ses.NewIdentityPolicy(ctx, "example", &ses.IdentityPolicyArgs{
 * 			Identity: exampleDomainIdentity.Arn,
 * 			Name:     pulumi.String("example"),
 * 			Policy: pulumi.String(example.ApplyT(func(example iam.GetPolicyDocumentResult) (*string, error) {
 * 				return &example.Json, nil
 * 			}).(pulumi.StringPtrOutput)),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ses.DomainIdentity;
 * import com.pulumi.aws.ses.DomainIdentityArgs;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.ses.IdentityPolicy;
 * import com.pulumi.aws.ses.IdentityPolicyArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var exampleDomainIdentity = new DomainIdentity("exampleDomainIdentity", DomainIdentityArgs.builder()
 *             .domain("example.com")
 *             .build());
 *         final var example = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .actions(
 *                     "SES:SendEmail",
 *                     "SES:SendRawEmail")
 *                 .resources(exampleDomainIdentity.arn())
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .identifiers("*")
 *                     .type("AWS")
 *                     .build())
 *                 .build())
 *             .build());
 *         var exampleIdentityPolicy = new IdentityPolicy("exampleIdentityPolicy", IdentityPolicyArgs.builder()
 *             .identity(exampleDomainIdentity.arn())
 *             .name("example")
 *             .policy(example.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult).applyValue(example -> example.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json())))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   exampleDomainIdentity:
 *     type: aws:ses:DomainIdentity
 *     name: example
 *     properties:
 *       domain: example.com
 *   exampleIdentityPolicy:
 *     type: aws:ses:IdentityPolicy
 *     name: example
 *     properties:
 *       identity: ${exampleDomainIdentity.arn}
 *       name: example
 *       policy: ${example.json}
 * variables:
 *   example:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - actions:
 *               - SES:SendEmail
 *               - SES:SendRawEmail
 *             resources:
 *               - ${exampleDomainIdentity.arn}
 *             principals:
 *               - identifiers:
 *                   - '*'
 *                 type: AWS
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import SES Identity Policies using the identity and policy name, separated by a pipe character (`|`). For example:
 * ```sh
 * $ pulumi import aws:ses/identityPolicy:IdentityPolicy example 'example.com|example'
 * ```
 * @property identity Name or Amazon Resource Name (ARN) of the SES Identity.
 * @property name Name of the policy.
 * @property policy JSON string of the policy.
 */
public data class IdentityPolicyArgs(
    public val identity: Output<String>? = null,
    public val name: Output<String>? = null,
    public val policy: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.ses.IdentityPolicyArgs> {
    override fun toJava(): com.pulumi.aws.ses.IdentityPolicyArgs =
        com.pulumi.aws.ses.IdentityPolicyArgs.builder()
            .identity(identity?.applyValue({ args0 -> args0 }))
            .name(name?.applyValue({ args0 -> args0 }))
            .policy(policy?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [IdentityPolicyArgs].
 */
@PulumiTagMarker
public class IdentityPolicyArgsBuilder internal constructor() {
    private var identity: Output<String>? = null

    private var name: Output<String>? = null

    private var policy: Output<String>? = null

    /**
     * @param value Name or Amazon Resource Name (ARN) of the SES Identity.
     */
    @JvmName("keqbrjcfnglernby")
    public suspend fun identity(`value`: Output<String>) {
        this.identity = value
    }

    /**
     * @param value Name of the policy.
     */
    @JvmName("vawcogrsgvsgxosj")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value JSON string of the policy.
     */
    @JvmName("fcfjwmbibwbnddrf")
    public suspend fun policy(`value`: Output<String>) {
        this.policy = value
    }

    /**
     * @param value Name or Amazon Resource Name (ARN) of the SES Identity.
     */
    @JvmName("ywrvlyyvshnmenap")
    public suspend fun identity(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.identity = mapped
    }

    /**
     * @param value Name of the policy.
     */
    @JvmName("cfsuyifiaxltxuoo")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value JSON string of the policy.
     */
    @JvmName("ryoqyjsfkkdenruc")
    public suspend fun policy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.policy = mapped
    }

    internal fun build(): IdentityPolicyArgs = IdentityPolicyArgs(
        identity = identity,
        name = name,
        policy = policy,
    )
}
