@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.shield.kotlin

import com.pulumi.aws.shield.ProtectionArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Enables AWS Shield Advanced for a specific AWS resource.
 * The resource can be an Amazon CloudFront distribution, Elastic Load Balancing load balancer, AWS Global Accelerator accelerator, Elastic IP Address, or an Amazon Route 53 hosted zone.
 * ## Example Usage
 * ### Create protection
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const available = aws.getAvailabilityZones({});
 * const current = aws.getRegion({});
 * const currentGetCallerIdentity = aws.getCallerIdentity({});
 * const example = new aws.ec2.Eip("example", {domain: "vpc"});
 * const exampleProtection = new aws.shield.Protection("example", {
 *     name: "example",
 *     resourceArn: pulumi.all([current, currentGetCallerIdentity, example&#46;id])&#46;apply(([current, currentGetCallerIdentity, id]) => `arn:aws:ec2:${current.name}:${currentGetCallerIdentity.accountId}:eip-allocation/${id}`),
 *     tags: {
 *         Environment: "Dev",
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * available = aws.get_availability_zones()
 * current = aws.get_region()
 * current_get_caller_identity = aws.get_caller_identity()
 * example = aws.ec2.Eip("example", domain="vpc")
 * example_protection = aws.shield.Protection("example",
 *     name="example",
 *     resource_arn=example.id.apply(lambda id: f"arn:aws:ec2:{current.name}:{current_get_caller_identity.account_id}:eip-allocation/{id}"),
 *     tags={
 *         "Environment": "Dev",
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var available = Aws.GetAvailabilityZones.Invoke();
 *     var current = Aws.GetRegion.Invoke();
 *     var currentGetCallerIdentity = Aws.GetCallerIdentity.Invoke();
 *     var example = new Aws.Ec2.Eip("example", new()
 *     {
 *         Domain = "vpc",
 *     });
 *     var exampleProtection = new Aws.Shield.Protection("example", new()
 *     {
 *         Name = "example",
 *         ResourceArn = Output.Tuple(current, currentGetCallerIdentity, example.Id).Apply(values =>
 *         {
 *             var current = values.Item1;
 *             var currentGetCallerIdentity = values.Item2;
 *             var id = values.Item3;
 *             return $"arn:aws:ec2:{current.Apply(getRegionResult => getRegionResult.Name)}:{currentGetCallerIdentity.Apply(getCallerIdentityResult => getCallerIdentityResult.AccountId)}:eip-allocation/{id}";
 *         }),
 *         Tags =
 *         {
 *             { "Environment", "Dev" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ec2"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/shield"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := aws.GetAvailabilityZones(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		current, err := aws.GetRegion(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		currentGetCallerIdentity, err := aws.GetCallerIdentity(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		example, err := ec2.NewEip(ctx, "example", &ec2.EipArgs{
 * 			Domain: pulumi.String("vpc"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = shield.NewProtection(ctx, "example", &shield.ProtectionArgs{
 * 			Name: pulumi.String("example"),
 * 			ResourceArn: example.ID().ApplyT(func(id string) (string, error) {
 * 				return fmt.Sprintf("arn:aws:ec2:%v:%v:eip-allocation/%v", current.Name, currentGetCallerIdentity.AccountId, id), nil
 * 			}).(pulumi.StringOutput),
 * 			Tags: pulumi.StringMap{
 * 				"Environment": pulumi.String("Dev"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.AwsFunctions;
 * import com.pulumi.aws.inputs.GetAvailabilityZonesArgs;
 * import com.pulumi.aws.inputs.GetRegionArgs;
 * import com.pulumi.aws.inputs.GetCallerIdentityArgs;
 * import com.pulumi.aws.ec2.Eip;
 * import com.pulumi.aws.ec2.EipArgs;
 * import com.pulumi.aws.shield.Protection;
 * import com.pulumi.aws.shield.ProtectionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var available = AwsFunctions.getAvailabilityZones();
 *         final var current = AwsFunctions.getRegion();
 *         final var currentGetCallerIdentity = AwsFunctions.getCallerIdentity();
 *         var example = new Eip("example", EipArgs.builder()
 *             .domain("vpc")
 *             .build());
 *         var exampleProtection = new Protection("exampleProtection", ProtectionArgs.builder()
 *             .name("example")
 *             .resourceArn(example.id().applyValue(id -> String.format("arn:aws:ec2:%s:%s:eip-allocation/%s", current.applyValue(getRegionResult -> getRegionResult.name()),currentGetCallerIdentity.applyValue(getCallerIdentityResult -> getCallerIdentityResult.accountId()),id)))
 *             .tags(Map.of("Environment", "Dev"))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:ec2:Eip
 *     properties:
 *       domain: vpc
 *   exampleProtection:
 *     type: aws:shield:Protection
 *     name: example
 *     properties:
 *       name: example
 *       resourceArn: arn:aws:ec2:${current.name}:${currentGetCallerIdentity.accountId}:eip-allocation/${example.id}
 *       tags:
 *         Environment: Dev
 * variables:
 *   available:
 *     fn::invoke:
 *       Function: aws:getAvailabilityZones
 *       Arguments: {}
 *   current:
 *     fn::invoke:
 *       Function: aws:getRegion
 *       Arguments: {}
 *   currentGetCallerIdentity:
 *     fn::invoke:
 *       Function: aws:getCallerIdentity
 *       Arguments: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import Shield protection resources using specifying their ID. For example:
 * ```sh
 * $ pulumi import aws:shield/protection:Protection example ff9592dc-22f3-4e88-afa1-7b29fde9669a
 * ```
 * @property name A friendly name for the Protection you are creating.
 * @property resourceArn The ARN (Amazon Resource Name) of the resource to be protected.
 * @property tags Key-value map of resource tags. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 */
public data class ProtectionArgs(
    public val name: Output<String>? = null,
    public val resourceArn: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
) : ConvertibleToJava<com.pulumi.aws.shield.ProtectionArgs> {
    override fun toJava(): com.pulumi.aws.shield.ProtectionArgs =
        com.pulumi.aws.shield.ProtectionArgs.builder()
            .name(name?.applyValue({ args0 -> args0 }))
            .resourceArn(resourceArn?.applyValue({ args0 -> args0 }))
            .tags(
                tags?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0.key.to(args0.value)
                    }).toMap()
                }),
            ).build()
}

/**
 * Builder for [ProtectionArgs].
 */
@PulumiTagMarker
public class ProtectionArgsBuilder internal constructor() {
    private var name: Output<String>? = null

    private var resourceArn: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    /**
     * @param value A friendly name for the Protection you are creating.
     */
    @JvmName("mfgrhxvofswgoapo")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value The ARN (Amazon Resource Name) of the resource to be protected.
     */
    @JvmName("tcvsiyqnlmkdhost")
    public suspend fun resourceArn(`value`: Output<String>) {
        this.resourceArn = value
    }

    /**
     * @param value Key-value map of resource tags. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("clkisyiqyxjtfbpl")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value A friendly name for the Protection you are creating.
     */
    @JvmName("xmqlrskegthegcpd")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value The ARN (Amazon Resource Name) of the resource to be protected.
     */
    @JvmName("fwggsgbqkgmvykxw")
    public suspend fun resourceArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.resourceArn = mapped
    }

    /**
     * @param value Key-value map of resource tags. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("ccmpcrxsdsblqoxm")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Key-value map of resource tags. .If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("jcneaiwgmnethbku")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    internal fun build(): ProtectionArgs = ProtectionArgs(
        name = name,
        resourceArn = resourceArn,
        tags = tags,
    )
}
