@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.sns.kotlin

import com.pulumi.aws.sns.TopicSubscriptionArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Provides a resource for subscribing to SNS topics. Requires that an SNS topic exist for the subscription to attach to. This resource allows you to automatically place messages sent to SNS topics in SQS queues, send them as HTTP(S) POST requests to a given endpoint, send SMS messages, or notify devices / applications. The most likely use case for provider users will probably be SQS queues.
 * > **NOTE:** If the SNS topic and SQS queue are in different AWS regions, the `aws.sns.TopicSubscription` must use an AWS provider that is in the same region as the SNS topic. If the `aws.sns.TopicSubscription` uses a provider with a different region than the SNS topic, this provider will fail to create the subscription.
 * > **NOTE:** Setup of cross-account subscriptions from SNS topics to SQS queues requires the provider to have access to BOTH accounts.
 * > **NOTE:** If an SNS topic and SQS queue are in different AWS accounts but the same region, the `aws.sns.TopicSubscription` must use the AWS provider for the account with the SQS queue. If `aws.sns.TopicSubscription` uses a Provider with a different account than the SQS queue, this provider creates the subscription but does not keep state and tries to re-create the subscription at every `apply`.
 * > **NOTE:** If an SNS topic and SQS queue are in different AWS accounts and different AWS regions, the subscription needs to be initiated from the account with the SQS queue but in the region of the SNS topic.
 * > **NOTE:** You cannot unsubscribe to a subscription that is pending confirmation. If you use `email`, `email-json`, or `http`/`https` (without auto-confirmation enabled), until the subscription is confirmed (e.g., outside of this provider), AWS does not allow this provider to delete / unsubscribe the subscription. If you `destroy` an unconfirmed subscription, this provider will remove the subscription from its state but the subscription will still exist in AWS. However, if you delete an SNS topic, SNS [deletes all the subscriptions](https://docs.aws.amazon.com/sns/latest/dg/sns-delete-subscription-topic.html) associated with the topic. Also, you can import a subscription after confirmation and then have the capability to delete it.
 * ## Example Usage
 * You can directly supply a topic and ARN by hand in the `topic_arn` property along with the queue ARN:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const userUpdatesSqsTarget = new aws.sns.TopicSubscription("user_updates_sqs_target", {
 *     topic: "arn:aws:sns:us-west-2:432981146916:user-updates-topic",
 *     protocol: "sqs",
 *     endpoint: "arn:aws:sqs:us-west-2:432981146916:queue-too",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * user_updates_sqs_target = aws.sns.TopicSubscription("user_updates_sqs_target",
 *     topic="arn:aws:sns:us-west-2:432981146916:user-updates-topic",
 *     protocol="sqs",
 *     endpoint="arn:aws:sqs:us-west-2:432981146916:queue-too")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var userUpdatesSqsTarget = new Aws.Sns.TopicSubscription("user_updates_sqs_target", new()
 *     {
 *         Topic = "arn:aws:sns:us-west-2:432981146916:user-updates-topic",
 *         Protocol = "sqs",
 *         Endpoint = "arn:aws:sqs:us-west-2:432981146916:queue-too",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sns"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sns.NewTopicSubscription(ctx, "user_updates_sqs_target", &sns.TopicSubscriptionArgs{
 * 			Topic:    pulumi.Any("arn:aws:sns:us-west-2:432981146916:user-updates-topic"),
 * 			Protocol: pulumi.String("sqs"),
 * 			Endpoint: pulumi.String("arn:aws:sqs:us-west-2:432981146916:queue-too"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.sns.TopicSubscription;
 * import com.pulumi.aws.sns.TopicSubscriptionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var userUpdatesSqsTarget = new TopicSubscription("userUpdatesSqsTarget", TopicSubscriptionArgs.builder()
 *             .topic("arn:aws:sns:us-west-2:432981146916:user-updates-topic")
 *             .protocol("sqs")
 *             .endpoint("arn:aws:sqs:us-west-2:432981146916:queue-too")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   userUpdatesSqsTarget:
 *     type: aws:sns:TopicSubscription
 *     name: user_updates_sqs_target
 *     properties:
 *       topic: arn:aws:sns:us-west-2:432981146916:user-updates-topic
 *       protocol: sqs
 *       endpoint: arn:aws:sqs:us-west-2:432981146916:queue-too
 * ```
 * <!--End PulumiCodeChooser -->
 * Alternatively you can use the ARN properties of a managed SNS topic and SQS queue:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const userUpdates = new aws.sns.Topic("user_updates", {name: "user-updates-topic"});
 * const userUpdatesQueue = new aws.sqs.Queue("user_updates_queue", {name: "user-updates-queue"});
 * const userUpdatesSqsTarget = new aws.sns.TopicSubscription("user_updates_sqs_target", {
 *     topic: userUpdates.arn,
 *     protocol: "sqs",
 *     endpoint: userUpdatesQueue.arn,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * user_updates = aws.sns.Topic("user_updates", name="user-updates-topic")
 * user_updates_queue = aws.sqs.Queue("user_updates_queue", name="user-updates-queue")
 * user_updates_sqs_target = aws.sns.TopicSubscription("user_updates_sqs_target",
 *     topic=user_updates.arn,
 *     protocol="sqs",
 *     endpoint=user_updates_queue.arn)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var userUpdates = new Aws.Sns.Topic("user_updates", new()
 *     {
 *         Name = "user-updates-topic",
 *     });
 *     var userUpdatesQueue = new Aws.Sqs.Queue("user_updates_queue", new()
 *     {
 *         Name = "user-updates-queue",
 *     });
 *     var userUpdatesSqsTarget = new Aws.Sns.TopicSubscription("user_updates_sqs_target", new()
 *     {
 *         Topic = userUpdates.Arn,
 *         Protocol = "sqs",
 *         Endpoint = userUpdatesQueue.Arn,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sns"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sqs"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		userUpdates, err := sns.NewTopic(ctx, "user_updates", &sns.TopicArgs{
 * 			Name: pulumi.String("user-updates-topic"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		userUpdatesQueue, err := sqs.NewQueue(ctx, "user_updates_queue", &sqs.QueueArgs{
 * 			Name: pulumi.String("user-updates-queue"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = sns.NewTopicSubscription(ctx, "user_updates_sqs_target", &sns.TopicSubscriptionArgs{
 * 			Topic:    userUpdates.Arn,
 * 			Protocol: pulumi.String("sqs"),
 * 			Endpoint: userUpdatesQueue.Arn,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.sns.Topic;
 * import com.pulumi.aws.sns.TopicArgs;
 * import com.pulumi.aws.sqs.Queue;
 * import com.pulumi.aws.sqs.QueueArgs;
 * import com.pulumi.aws.sns.TopicSubscription;
 * import com.pulumi.aws.sns.TopicSubscriptionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var userUpdates = new Topic("userUpdates", TopicArgs.builder()
 *             .name("user-updates-topic")
 *             .build());
 *         var userUpdatesQueue = new Queue("userUpdatesQueue", QueueArgs.builder()
 *             .name("user-updates-queue")
 *             .build());
 *         var userUpdatesSqsTarget = new TopicSubscription("userUpdatesSqsTarget", TopicSubscriptionArgs.builder()
 *             .topic(userUpdates.arn())
 *             .protocol("sqs")
 *             .endpoint(userUpdatesQueue.arn())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   userUpdates:
 *     type: aws:sns:Topic
 *     name: user_updates
 *     properties:
 *       name: user-updates-topic
 *   userUpdatesQueue:
 *     type: aws:sqs:Queue
 *     name: user_updates_queue
 *     properties:
 *       name: user-updates-queue
 *   userUpdatesSqsTarget:
 *     type: aws:sns:TopicSubscription
 *     name: user_updates_sqs_target
 *     properties:
 *       topic: ${userUpdates.arn}
 *       protocol: sqs
 *       endpoint: ${userUpdatesQueue.arn}
 * ```
 * <!--End PulumiCodeChooser -->
 * You can subscribe SNS topics to SQS queues in different Amazon accounts and regions:
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const config = new pulumi.Config();
 * const sns = config.getObject("sns") || {
 *     "account-id": "111111111111",
 *     displayName: "example",
 *     name: "example-sns-topic",
 *     region: "us-west-1",
 *     "role-name": "service/service",
 * };
 * const sqs = config.getObject("sqs") || {
 *     "account-id": "222222222222",
 *     name: "example-sqs-queue",
 *     region: "us-east-1",
 *     "role-name": "service/service",
 * };
 * const sns-topic-policy = aws.iam.getPolicyDocument({
 *     policyId: "__default_policy_ID",
 *     statements: [
 *         {
 *             actions: [
 *                 "SNS:Subscribe",
 *                 "SNS:SetTopicAttributes",
 *                 "SNS:RemovePermission",
 *                 "SNS:Publish",
 *                 "SNS:ListSubscriptionsByTopic",
 *                 "SNS:GetTopicAttributes",
 *                 "SNS:DeleteTopic",
 *                 "SNS:AddPermission",
 *             ],
 *             conditions: [{
 *                 test: "StringEquals",
 *                 variable: "AWS:SourceOwner",
 *                 values: [sns["account-id"]],
 *             }],
 *             effect: "Allow",
 *             principals: [{
 *                 type: "AWS",
 *                 identifiers: ["*"],
 *             }],
 *             resources: [`arn:aws:sns:${sns&#46;region}:${sns["account-id"]}:${sns&#46;name}`],
 *             sid: "__default_statement_ID",
 *         },
 *         {
 *             actions: [
 *                 "SNS:Subscribe",
 *                 "SNS:Receive",
 *             ],
 *             conditions: [{
 *                 test: "StringLike",
 *                 variable: "SNS:Endpoint",
 *                 values: [`arn:aws:sqs:${sqs&#46;region}:${sqs["account-id"]}:${sqs&#46;name}`],
 *             }],
 *             effect: "Allow",
 *             principals: [{
 *                 type: "AWS",
 *                 identifiers: ["*"],
 *             }],
 *             resources: [`arn:aws:sns:${sns&#46;region}:${sns["account-id"]}:${sns&#46;name}`],
 *             sid: "__console_sub_0",
 *         },
 *     ],
 * });
 * const sqs-queue-policy = aws.iam.getPolicyDocument({
 *     policyId: `arn:aws:sqs:${sqs.region}:${sqs["account-id"]}:${sqs.name}/SQSDefaultPolicy`,
 *     statements: [{
 *         sid: "example-sns-topic",
 *         effect: "Allow",
 *         principals: [{
 *             type: "AWS",
 *             identifiers: ["*"],
 *         }],
 *         actions: ["SQS:SendMessage"],
 *         resources: [`arn:aws:sqs:${sqs&#46;region}:${sqs["account-id"]}:${sqs&#46;name}`],
 *         conditions: [{
 *             test: "ArnEquals",
 *             variable: "aws:SourceArn",
 *             values: [`arn:aws:sns:${sns&#46;region}:${sns["account-id"]}:${sns&#46;name}`],
 *         }],
 *     }],
 * });
 * const sns_topic = new aws.sns.Topic("sns-topic", {
 *     name: sns.name,
 *     displayName: sns.display_name,
 *     policy: sns_topic_policy.then(sns_topic_policy => sns_topic_policy.json),
 * });
 * const sqs_queue = new aws.sqs.Queue("sqs-queue", {
 *     name: sqs.name,
 *     policy: sqs_queue_policy.then(sqs_queue_policy => sqs_queue_policy.json),
 * });
 * const sns_topicTopicSubscription = new aws.sns.TopicSubscription("sns-topic", {
 *     topic: sns_topic.arn,
 *     protocol: "sqs",
 *     endpoint: sqs_queue.arn,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * config = pulumi.Config()
 * sns = config.get_object("sns")
 * if sns is None:
 *     sns = {
 *         "account-id": "111111111111",
 *         "displayName": "example",
 *         "name": "example-sns-topic",
 *         "region": "us-west-1",
 *         "role-name": "service/service",
 *     }
 * sqs = config.get_object("sqs")
 * if sqs is None:
 *     sqs = {
 *         "account-id": "222222222222",
 *         "name": "example-sqs-queue",
 *         "region": "us-east-1",
 *         "role-name": "service/service",
 *     }
 * sns_topic_policy = aws.iam.get_policy_document(policy_id="__default_policy_ID",
 *     statements=[
 *         {
 *             "actions": [
 *                 "SNS:Subscribe",
 *                 "SNS:SetTopicAttributes",
 *                 "SNS:RemovePermission",
 *                 "SNS:Publish",
 *                 "SNS:ListSubscriptionsByTopic",
 *                 "SNS:GetTopicAttributes",
 *                 "SNS:DeleteTopic",
 *                 "SNS:AddPermission",
 *             ],
 *             "conditions": [{
 *                 "test": "StringEquals",
 *                 "variable": "AWS:SourceOwner",
 *                 "values": [sns["account-id"]],
 *             }],
 *             "effect": "Allow",
 *             "principals": [{
 *                 "type": "AWS",
 *                 "identifiers": ["*"],
 *             }],
 *             "resources": [f"arn:aws:sns:{sns['region']}:{sns['account-id']}:{sns['name']}"],
 *             "sid": "__default_statement_ID",
 *         },
 *         {
 *             "actions": [
 *                 "SNS:Subscribe",
 *                 "SNS:Receive",
 *             ],
 *             "conditions": [{
 *                 "test": "StringLike",
 *                 "variable": "SNS:Endpoint",
 *                 "values": [f"arn:aws:sqs:{sqs['region']}:{sqs['account-id']}:{sqs['name']}"],
 *             }],
 *             "effect": "Allow",
 *             "principals": [{
 *                 "type": "AWS",
 *                 "identifiers": ["*"],
 *             }],
 *             "resources": [f"arn:aws:sns:{sns['region']}:{sns['account-id']}:{sns['name']}"],
 *             "sid": "__console_sub_0",
 *         },
 *     ])
 * sqs_queue_policy = aws.iam.get_policy_document(policy_id=f"arn:aws:sqs:{sqs['region']}:{sqs['account-id']}:{sqs['name']}/SQSDefaultPolicy",
 *     statements=[{
 *         "sid": "example-sns-topic",
 *         "effect": "Allow",
 *         "principals": [{
 *             "type": "AWS",
 *             "identifiers": ["*"],
 *         }],
 *         "actions": ["SQS:SendMessage"],
 *         "resources": [f"arn:aws:sqs:{sqs['region']}:{sqs['account-id']}:{sqs['name']}"],
 *         "conditions": [{
 *             "test": "ArnEquals",
 *             "variable": "aws:SourceArn",
 *             "values": [f"arn:aws:sns:{sns['region']}:{sns['account-id']}:{sns['name']}"],
 *         }],
 *     }])
 * sns_topic = aws.sns.Topic("sns-topic",
 *     name=sns["name"],
 *     display_name=sns["display_name"],
 *     policy=sns_topic_policy.json)
 * sqs_queue = aws.sqs.Queue("sqs-queue",
 *     name=sqs["name"],
 *     policy=sqs_queue_policy.json)
 * sns_topic_topic_subscription = aws.sns.TopicSubscription("sns-topic",
 *     topic=sns_topic.arn,
 *     protocol="sqs",
 *     endpoint=sqs_queue.arn)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var config = new Config();
 *     var sns = config.GetObject<dynamic>("sns") ??
 *     {
 *         { "account-id", "111111111111" },
 *         { "displayName", "example" },
 *         { "name", "example-sns-topic" },
 *         { "region", "us-west-1" },
 *         { "role-name", "service/service" },
 *     };
 *     var sqs = config.GetObject<dynamic>("sqs") ??
 *     {
 *         { "account-id", "222222222222" },
 *         { "name", "example-sqs-queue" },
 *         { "region", "us-east-1" },
 *         { "role-name", "service/service" },
 *     };
 *     var sns_topic_policy = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         PolicyId = "__default_policy_ID",
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Actions = new[]
 *                 {
 *                     "SNS:Subscribe",
 *                     "SNS:SetTopicAttributes",
 *                     "SNS:RemovePermission",
 *                     "SNS:Publish",
 *                     "SNS:ListSubscriptionsByTopic",
 *                     "SNS:GetTopicAttributes",
 *                     "SNS:DeleteTopic",
 *                     "SNS:AddPermission",
 *                 },
 *                 Conditions = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementConditionInputArgs
 *                     {
 *                         Test = "StringEquals",
 *                         Variable = "AWS:SourceOwner",
 *                         Values = new[]
 *                         {
 *                             sns.Account_id,
 *                         },
 *                     },
 *                 },
 *                 Effect = "Allow",
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Type = "AWS",
 *                         Identifiers = new[]
 *                         {
 *                             "*",
 *                         },
 *                     },
 *                 },
 *                 Resources = new[]
 *                 {
 *                     $"arn:aws:sns:{sns.Region}:{sns.Account_id}:{sns.Name}",
 *                 },
 *                 Sid = "__default_statement_ID",
 *             },
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Actions = new[]
 *                 {
 *                     "SNS:Subscribe",
 *                     "SNS:Receive",
 *                 },
 *                 Conditions = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementConditionInputArgs
 *                     {
 *                         Test = "StringLike",
 *                         Variable = "SNS:Endpoint",
 *                         Values = new[]
 *                         {
 *                             $"arn:aws:sqs:{sqs.Region}:{sqs.Account_id}:{sqs.Name}",
 *                         },
 *                     },
 *                 },
 *                 Effect = "Allow",
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Type = "AWS",
 *                         Identifiers = new[]
 *                         {
 *                             "*",
 *                         },
 *                     },
 *                 },
 *                 Resources = new[]
 *                 {
 *                     $"arn:aws:sns:{sns.Region}:{sns.Account_id}:{sns.Name}",
 *                 },
 *                 Sid = "__console_sub_0",
 *             },
 *         },
 *     });
 *     var sqs_queue_policy = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         PolicyId = $"arn:aws:sqs:{sqs.Region}:{sqs.Account_id}:{sqs.Name}/SQSDefaultPolicy",
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Sid = "example-sns-topic",
 *                 Effect = "Allow",
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Type = "AWS",
 *                         Identifiers = new[]
 *                         {
 *                             "*",
 *                         },
 *                     },
 *                 },
 *                 Actions = new[]
 *                 {
 *                     "SQS:SendMessage",
 *                 },
 *                 Resources = new[]
 *                 {
 *                     $"arn:aws:sqs:{sqs.Region}:{sqs.Account_id}:{sqs.Name}",
 *                 },
 *                 Conditions = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementConditionInputArgs
 *                     {
 *                         Test = "ArnEquals",
 *                         Variable = "aws:SourceArn",
 *                         Values = new[]
 *                         {
 *                             $"arn:aws:sns:{sns.Region}:{sns.Account_id}:{sns.Name}",
 *                         },
 *                     },
 *                 },
 *             },
 *         },
 *     });
 *     var sns_topic = new Aws.Sns.Topic("sns-topic", new()
 *     {
 *         Name = sns.Name,
 *         DisplayName = sns.Display_name,
 *         Policy = sns_topic_policy.Apply(sns_topic_policy => sns_topic_policy.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json)),
 *     });
 *     var sqs_queue = new Aws.Sqs.Queue("sqs-queue", new()
 *     {
 *         Name = sqs.Name,
 *         Policy = sqs_queue_policy.Apply(sqs_queue_policy => sqs_queue_policy.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json)),
 *     });
 *     var sns_topicTopicSubscription = new Aws.Sns.TopicSubscription("sns-topic", new()
 *     {
 *         Topic = sns_topic.Arn,
 *         Protocol = "sqs",
 *         Endpoint = sqs_queue.Arn,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sns"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sqs"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi/config"
 * )
 * func main() {
 * pulumi.Run(func(ctx *pulumi.Context) error {
 * cfg := config.New(ctx, "")
 * sns := map[string]interface{}{
 * "account-id": "111111111111",
 * "displayName": "example",
 * "name": "example-sns-topic",
 * "region": "us-west-1",
 * "role-name": "service/service",
 * };
 * if param := cfg.GetObject("sns"); param != nil {
 * sns = param
 * }
 * sqs := map[string]interface{}{
 * "account-id": "222222222222",
 * "name": "example-sqs-queue",
 * "region": "us-east-1",
 * "role-name": "service/service",
 * };
 * if param := cfg.GetObject("sqs"); param != nil {
 * sqs = param
 * }
 * sns_topic_policy, err := iam.GetPolicyDocument(ctx, &iam.GetPolicyDocumentArgs{
 * PolicyId: pulumi.StringRef("__default_policy_ID"),
 * Statements: []iam.GetPolicyDocumentStatement{
 * {
 * Actions: []string{
 * "SNS:Subscribe",
 * "SNS:SetTopicAttributes",
 * "SNS:RemovePermission",
 * "SNS:Publish",
 * "SNS:ListSubscriptionsByTopic",
 * "SNS:GetTopicAttributes",
 * "SNS:DeleteTopic",
 * "SNS:AddPermission",
 * },
 * Conditions: []iam.GetPolicyDocumentStatementCondition{
 * {
 * Test: "StringEquals",
 * Variable: "AWS:SourceOwner",
 * Values: interface{}{
 * sns.AccountId,
 * },
 * },
 * },
 * Effect: pulumi.StringRef("Allow"),
 * Principals: []iam.GetPolicyDocumentStatementPrincipal{
 * {
 * Type: "AWS",
 * Identifiers: []string{
 * "*",
 * },
 * },
 * },
 * Resources: []string{
 * fmt.Sprintf("arn:aws:sns:%v:%v:%v", sns.Region, sns.AccountId, sns.Name),
 * },
 * Sid: pulumi.StringRef("__default_statement_ID"),
 * },
 * {
 * Actions: []string{
 * "SNS:Subscribe",
 * "SNS:Receive",
 * },
 * Conditions: []iam.GetPolicyDocumentStatementCondition{
 * {
 * Test: "StringLike",
 * Variable: "SNS:Endpoint",
 * Values: []string{
 * fmt.Sprintf("arn:aws:sqs:%v:%v:%v", sqs.Region, sqs.AccountId, sqs.Name),
 * },
 * },
 * },
 * Effect: pulumi.StringRef("Allow"),
 * Principals: []iam.GetPolicyDocumentStatementPrincipal{
 * {
 * Type: "AWS",
 * Identifiers: []string{
 * "*",
 * },
 * },
 * },
 * Resources: []string{
 * fmt.Sprintf("arn:aws:sns:%v:%v:%v", sns.Region, sns.AccountId, sns.Name),
 * },
 * Sid: pulumi.StringRef("__console_sub_0"),
 * },
 * },
 * }, nil);
 * if err != nil {
 * return err
 * }
 * sqs_queue_policy, err := iam.GetPolicyDocument(ctx, &iam.GetPolicyDocumentArgs{
 * PolicyId: pulumi.StringRef(fmt.Sprintf("arn:aws:sqs:%v:%v:%v/SQSDefaultPolicy", sqs.Region, sqs.AccountId, sqs.Name)),
 * Statements: []iam.GetPolicyDocumentStatement{
 * {
 * Sid: pulumi.StringRef("example-sns-topic"),
 * Effect: pulumi.StringRef("Allow"),
 * Principals: []iam.GetPolicyDocumentStatementPrincipal{
 * {
 * Type: "AWS",
 * Identifiers: []string{
 * "*",
 * },
 * },
 * },
 * Actions: []string{
 * "SQS:SendMessage",
 * },
 * Resources: []string{
 * fmt.Sprintf("arn:aws:sqs:%v:%v:%v", sqs.Region, sqs.AccountId, sqs.Name),
 * },
 * Conditions: []iam.GetPolicyDocumentStatementCondition{
 * {
 * Test: "ArnEquals",
 * Variable: "aws:SourceArn",
 * Values: []string{
 * fmt.Sprintf("arn:aws:sns:%v:%v:%v", sns.Region, sns.AccountId, sns.Name),
 * },
 * },
 * },
 * },
 * },
 * }, nil);
 * if err != nil {
 * return err
 * }
 * _, err = sns.NewTopic(ctx, "sns-topic", &sns.TopicArgs{
 * Name: pulumi.Any(sns.Name),
 * DisplayName: pulumi.Any(sns.Display_name),
 * Policy: pulumi.String(sns_topic_policy.Json),
 * })
 * if err != nil {
 * return err
 * }
 * _, err = sqs.NewQueue(ctx, "sqs-queue", &sqs.QueueArgs{
 * Name: pulumi.Any(sqs.Name),
 * Policy: pulumi.String(sqs_queue_policy.Json),
 * })
 * if err != nil {
 * return err
 * }
 * _, err = sns.NewTopicSubscription(ctx, "sns-topic", &sns.TopicSubscriptionArgs{
 * Topic: sns_topic.Arn,
 * Protocol: pulumi.String("sqs"),
 * Endpoint: sqs_queue.Arn,
 * })
 * if err != nil {
 * return err
 * }
 * return nil
 * })
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.sns.Topic;
 * import com.pulumi.aws.sns.TopicArgs;
 * import com.pulumi.aws.sqs.Queue;
 * import com.pulumi.aws.sqs.QueueArgs;
 * import com.pulumi.aws.sns.TopicSubscription;
 * import com.pulumi.aws.sns.TopicSubscriptionArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var config = ctx.config();
 *         final var sns = config.get("sns").orElse(%!v(PANIC=Format method: runtime error: invalid memory address or nil pointer dereference));
 *         final var sqs = config.get("sqs").orElse(%!v(PANIC=Format method: runtime error: invalid memory address or nil pointer dereference));
 *         final var sns-topic-policy = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .policyId("__default_policy_ID")
 *             .statements(
 *                 GetPolicyDocumentStatementArgs.builder()
 *                     .actions(
 *                         "SNS:Subscribe",
 *                         "SNS:SetTopicAttributes",
 *                         "SNS:RemovePermission",
 *                         "SNS:Publish",
 *                         "SNS:ListSubscriptionsByTopic",
 *                         "SNS:GetTopicAttributes",
 *                         "SNS:DeleteTopic",
 *                         "SNS:AddPermission")
 *                     .conditions(GetPolicyDocumentStatementConditionArgs.builder()
 *                         .test("StringEquals")
 *                         .variable("AWS:SourceOwner")
 *                         .values(sns.account-id())
 *                         .build())
 *                     .effect("Allow")
 *                     .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                         .type("AWS")
 *                         .identifiers("*")
 *                         .build())
 *                     .resources(String.format("arn:aws:sns:%s:%s:%s", sns.region(),sns.account-id(),sns.name()))
 *                     .sid("__default_statement_ID")
 *                     .build(),
 *                 GetPolicyDocumentStatementArgs.builder()
 *                     .actions(
 *                         "SNS:Subscribe",
 *                         "SNS:Receive")
 *                     .conditions(GetPolicyDocumentStatementConditionArgs.builder()
 *                         .test("StringLike")
 *                         .variable("SNS:Endpoint")
 *                         .values(String.format("arn:aws:sqs:%s:%s:%s", sqs.region(),sqs.account-id(),sqs.name()))
 *                         .build())
 *                     .effect("Allow")
 *                     .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                         .type("AWS")
 *                         .identifiers("*")
 *                         .build())
 *                     .resources(String.format("arn:aws:sns:%s:%s:%s", sns.region(),sns.account-id(),sns.name()))
 *                     .sid("__console_sub_0")
 *                     .build())
 *             .build());
 *         final var sqs-queue-policy = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .policyId(String.format("arn:aws:sqs:%s:%s:%s/SQSDefaultPolicy", sqs.region(),sqs.account-id(),sqs.name()))
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .sid("example-sns-topic")
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("AWS")
 *                     .identifiers("*")
 *                     .build())
 *                 .actions("SQS:SendMessage")
 *                 .resources(String.format("arn:aws:sqs:%s:%s:%s", sqs.region(),sqs.account-id(),sqs.name()))
 *                 .conditions(GetPolicyDocumentStatementConditionArgs.builder()
 *                     .test("ArnEquals")
 *                     .variable("aws:SourceArn")
 *                     .values(String.format("arn:aws:sns:%s:%s:%s", sns.region(),sns.account-id(),sns.name()))
 *                     .build())
 *                 .build())
 *             .build());
 *         var sns_topic = new Topic("sns-topic", TopicArgs.builder()
 *             .name(sns.name())
 *             .displayName(sns.display_name())
 *             .policy(sns_topic_policy.json())
 *             .build());
 *         var sqs_queue = new Queue("sqs-queue", QueueArgs.builder()
 *             .name(sqs.name())
 *             .policy(sqs_queue_policy.json())
 *             .build());
 *         var sns_topicTopicSubscription = new TopicSubscription("sns-topicTopicSubscription", TopicSubscriptionArgs.builder()
 *             .topic(sns_topic.arn())
 *             .protocol("sqs")
 *             .endpoint(sqs_queue.arn())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * configuration:
 *   sns:
 *     type: dynamic
 *     default:
 *       account-id: '111111111111'
 *       displayName: example
 *       name: example-sns-topic
 *       region: us-west-1
 *       role-name: service/service
 *   sqs:
 *     type: dynamic
 *     default:
 *       account-id: '222222222222'
 *       name: example-sqs-queue
 *       region: us-east-1
 *       role-name: service/service
 * resources:
 *   sns-topic:
 *     type: aws:sns:Topic
 *     properties:
 *       name: ${sns.name}
 *       displayName: ${sns.display_name}
 *       policy: ${["sns-topic-policy"].json}
 *   sqs-queue:
 *     type: aws:sqs:Queue
 *     properties:
 *       name: ${sqs.name}
 *       policy: ${["sqs-queue-policy"].json}
 *   sns-topicTopicSubscription:
 *     type: aws:sns:TopicSubscription
 *     name: sns-topic
 *     properties:
 *       topic: ${["sns-topic"].arn}
 *       protocol: sqs
 *       endpoint: ${["sqs-queue"].arn}
 * variables:
 *   sns-topic-policy:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         policyId: __default_policy_ID
 *         statements:
 *           - actions:
 *               - SNS:Subscribe
 *               - SNS:SetTopicAttributes
 *               - SNS:RemovePermission
 *               - SNS:Publish
 *               - SNS:ListSubscriptionsByTopic
 *               - SNS:GetTopicAttributes
 *               - SNS:DeleteTopic
 *               - SNS:AddPermission
 *             conditions:
 *               - test: StringEquals
 *                 variable: AWS:SourceOwner
 *                 values:
 *                   - ${sns"account-id"[%!s(MISSING)]}
 *             effect: Allow
 *             principals:
 *               - type: AWS
 *                 identifiers:
 *                   - '*'
 *             resources:
 *               - arn:aws:sns:${sns.region}:${sns"account-id"[%!s(MISSING)]}:${sns.name}
 *             sid: __default_statement_ID
 *           - actions:
 *               - SNS:Subscribe
 *               - SNS:Receive
 *             conditions:
 *               - test: StringLike
 *                 variable: SNS:Endpoint
 *                 values:
 *                   - arn:aws:sqs:${sqs.region}:${sqs"account-id"[%!s(MISSING)]}:${sqs.name}
 *             effect: Allow
 *             principals:
 *               - type: AWS
 *                 identifiers:
 *                   - '*'
 *             resources:
 *               - arn:aws:sns:${sns.region}:${sns"account-id"[%!s(MISSING)]}:${sns.name}
 *             sid: __console_sub_0
 *   sqs-queue-policy:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         policyId: arn:aws:sqs:${sqs.region}:${sqs"account-id"[%!s(MISSING)]}:${sqs.name}/SQSDefaultPolicy
 *         statements:
 *           - sid: example-sns-topic
 *             effect: Allow
 *             principals:
 *               - type: AWS
 *                 identifiers:
 *                   - '*'
 *             actions:
 *               - SQS:SendMessage
 *             resources:
 *               - arn:aws:sqs:${sqs.region}:${sqs"account-id"[%!s(MISSING)]}:${sqs.name}
 *             conditions:
 *               - test: ArnEquals
 *                 variable: aws:SourceArn
 *                 values:
 *                   - arn:aws:sns:${sns.region}:${sns"account-id"[%!s(MISSING)]}:${sns.name}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import SNS Topic Subscriptions using the subscription `arn`. For example:
 * ```sh
 * $ pulumi import aws:sns/topicSubscription:TopicSubscription user_updates_sqs_target arn:aws:sns:us-west-2:0123456789012:my-topic:8a21d249-4329-4871-acc6-7be709c6ea7f
 * ```
 * @property confirmationTimeoutInMinutes Integer indicating number of minutes to wait in retrying mode for fetching subscription arn before marking it as failure. Only applicable for http and https protocols. Default is `1`.
 * @property deliveryPolicy JSON String with the delivery policy (retries, backoff, etc.) that will be used in the subscription - this only applies to HTTP/S subscriptions. Refer to the [SNS docs](https://docs.aws.amazon.com/sns/latest/dg/DeliveryPolicies.html) for more details.
 * @property endpoint Endpoint to send data to. The contents vary with the protocol. See details below.
 * @property endpointAutoConfirms Whether the endpoint is capable of [auto confirming subscription](http://docs.aws.amazon.com/sns/latest/dg/SendMessageToHttp.html#SendMessageToHttp.prepare) (e.g., PagerDuty). Default is `false`.
 * @property filterPolicy JSON String with the filter policy that will be used in the subscription to filter messages seen by the target resource. Refer to the [SNS docs](https://docs.aws.amazon.com/sns/latest/dg/message-filtering.html) for more details.
 * @property filterPolicyScope Whether the `filter_policy` applies to `MessageAttributes` (default) or `MessageBody`.
 * @property protocol Protocol to use. Valid values are: `sqs`, `sms`, `lambda`, `firehose`, and `application`. Protocols `email`, `email-json`, `http` and `https` are also valid but partially supported. See details below.
 * @property rawMessageDelivery Whether to enable raw message delivery (the original message is directly passed, not wrapped in JSON with the original message in the message property). Default is `false`.
 * @property redrivePolicy JSON String with the redrive policy that will be used in the subscription. Refer to the [SNS docs](https://docs.aws.amazon.com/sns/latest/dg/sns-dead-letter-queues.html#how-messages-moved-into-dead-letter-queue) for more details.
 * @property replayPolicy JSON String with the archived message replay policy that will be used in the subscription. Refer to the [SNS docs](https://docs.aws.amazon.com/sns/latest/dg/message-archiving-and-replay-subscriber.html) for more details.
 * @property subscriptionRoleArn ARN of the IAM role to publish to Kinesis Data Firehose delivery stream. Refer to [SNS docs](https://docs.aws.amazon.com/sns/latest/dg/sns-firehose-as-subscriber.html).
 * @property topic ARN of the SNS topic to subscribe to.
 * The following arguments are optional:
 */
public data class TopicSubscriptionArgs(
    public val confirmationTimeoutInMinutes: Output<Int>? = null,
    public val deliveryPolicy: Output<String>? = null,
    public val endpoint: Output<String>? = null,
    public val endpointAutoConfirms: Output<Boolean>? = null,
    public val filterPolicy: Output<String>? = null,
    public val filterPolicyScope: Output<String>? = null,
    public val protocol: Output<String>? = null,
    public val rawMessageDelivery: Output<Boolean>? = null,
    public val redrivePolicy: Output<String>? = null,
    public val replayPolicy: Output<String>? = null,
    public val subscriptionRoleArn: Output<String>? = null,
    public val topic: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.sns.TopicSubscriptionArgs> {
    override fun toJava(): com.pulumi.aws.sns.TopicSubscriptionArgs =
        com.pulumi.aws.sns.TopicSubscriptionArgs.builder()
            .confirmationTimeoutInMinutes(confirmationTimeoutInMinutes?.applyValue({ args0 -> args0 }))
            .deliveryPolicy(deliveryPolicy?.applyValue({ args0 -> args0 }))
            .endpoint(endpoint?.applyValue({ args0 -> args0 }))
            .endpointAutoConfirms(endpointAutoConfirms?.applyValue({ args0 -> args0 }))
            .filterPolicy(filterPolicy?.applyValue({ args0 -> args0 }))
            .filterPolicyScope(filterPolicyScope?.applyValue({ args0 -> args0 }))
            .protocol(protocol?.applyValue({ args0 -> args0 }))
            .rawMessageDelivery(rawMessageDelivery?.applyValue({ args0 -> args0 }))
            .redrivePolicy(redrivePolicy?.applyValue({ args0 -> args0 }))
            .replayPolicy(replayPolicy?.applyValue({ args0 -> args0 }))
            .subscriptionRoleArn(subscriptionRoleArn?.applyValue({ args0 -> args0 }))
            .topic(topic?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [TopicSubscriptionArgs].
 */
@PulumiTagMarker
public class TopicSubscriptionArgsBuilder internal constructor() {
    private var confirmationTimeoutInMinutes: Output<Int>? = null

    private var deliveryPolicy: Output<String>? = null

    private var endpoint: Output<String>? = null

    private var endpointAutoConfirms: Output<Boolean>? = null

    private var filterPolicy: Output<String>? = null

    private var filterPolicyScope: Output<String>? = null

    private var protocol: Output<String>? = null

    private var rawMessageDelivery: Output<Boolean>? = null

    private var redrivePolicy: Output<String>? = null

    private var replayPolicy: Output<String>? = null

    private var subscriptionRoleArn: Output<String>? = null

    private var topic: Output<String>? = null

    /**
     * @param value Integer indicating number of minutes to wait in retrying mode for fetching subscription arn before marking it as failure. Only applicable for http and https protocols. Default is `1`.
     */
    @JvmName("lhlqlvtygltejufh")
    public suspend fun confirmationTimeoutInMinutes(`value`: Output<Int>) {
        this.confirmationTimeoutInMinutes = value
    }

    /**
     * @param value JSON String with the delivery policy (retries, backoff, etc.) that will be used in the subscription - this only applies to HTTP/S subscriptions. Refer to the [SNS docs](https://docs.aws.amazon.com/sns/latest/dg/DeliveryPolicies.html) for more details.
     */
    @JvmName("rbgqxfofrrasqxjc")
    public suspend fun deliveryPolicy(`value`: Output<String>) {
        this.deliveryPolicy = value
    }

    /**
     * @param value Endpoint to send data to. The contents vary with the protocol. See details below.
     */
    @JvmName("oesbqfanvevlhttx")
    public suspend fun endpoint(`value`: Output<String>) {
        this.endpoint = value
    }

    /**
     * @param value Whether the endpoint is capable of [auto confirming subscription](http://docs.aws.amazon.com/sns/latest/dg/SendMessageToHttp.html#SendMessageToHttp.prepare) (e.g., PagerDuty). Default is `false`.
     */
    @JvmName("uqvccwjvmgmtwxmc")
    public suspend fun endpointAutoConfirms(`value`: Output<Boolean>) {
        this.endpointAutoConfirms = value
    }

    /**
     * @param value JSON String with the filter policy that will be used in the subscription to filter messages seen by the target resource. Refer to the [SNS docs](https://docs.aws.amazon.com/sns/latest/dg/message-filtering.html) for more details.
     */
    @JvmName("ffapogunyuifbsmb")
    public suspend fun filterPolicy(`value`: Output<String>) {
        this.filterPolicy = value
    }

    /**
     * @param value Whether the `filter_policy` applies to `MessageAttributes` (default) or `MessageBody`.
     */
    @JvmName("kqqxsqdkbntpsnvo")
    public suspend fun filterPolicyScope(`value`: Output<String>) {
        this.filterPolicyScope = value
    }

    /**
     * @param value Protocol to use. Valid values are: `sqs`, `sms`, `lambda`, `firehose`, and `application`. Protocols `email`, `email-json`, `http` and `https` are also valid but partially supported. See details below.
     */
    @JvmName("lgjyucnsyrygkdbb")
    public suspend fun protocol(`value`: Output<String>) {
        this.protocol = value
    }

    /**
     * @param value Whether to enable raw message delivery (the original message is directly passed, not wrapped in JSON with the original message in the message property). Default is `false`.
     */
    @JvmName("lrcqbhlybnanebuo")
    public suspend fun rawMessageDelivery(`value`: Output<Boolean>) {
        this.rawMessageDelivery = value
    }

    /**
     * @param value JSON String with the redrive policy that will be used in the subscription. Refer to the [SNS docs](https://docs.aws.amazon.com/sns/latest/dg/sns-dead-letter-queues.html#how-messages-moved-into-dead-letter-queue) for more details.
     */
    @JvmName("nshsyqcxsvsbkahs")
    public suspend fun redrivePolicy(`value`: Output<String>) {
        this.redrivePolicy = value
    }

    /**
     * @param value JSON String with the archived message replay policy that will be used in the subscription. Refer to the [SNS docs](https://docs.aws.amazon.com/sns/latest/dg/message-archiving-and-replay-subscriber.html) for more details.
     */
    @JvmName("rascudonwmndfpnh")
    public suspend fun replayPolicy(`value`: Output<String>) {
        this.replayPolicy = value
    }

    /**
     * @param value ARN of the IAM role to publish to Kinesis Data Firehose delivery stream. Refer to [SNS docs](https://docs.aws.amazon.com/sns/latest/dg/sns-firehose-as-subscriber.html).
     */
    @JvmName("bjkwbpntabccoabr")
    public suspend fun subscriptionRoleArn(`value`: Output<String>) {
        this.subscriptionRoleArn = value
    }

    /**
     * @param value ARN of the SNS topic to subscribe to.
     * The following arguments are optional:
     */
    @JvmName("pbsdjgawgejhllpt")
    public suspend fun topic(`value`: Output<String>) {
        this.topic = value
    }

    /**
     * @param value Integer indicating number of minutes to wait in retrying mode for fetching subscription arn before marking it as failure. Only applicable for http and https protocols. Default is `1`.
     */
    @JvmName("olotkacjeuyidffj")
    public suspend fun confirmationTimeoutInMinutes(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.confirmationTimeoutInMinutes = mapped
    }

    /**
     * @param value JSON String with the delivery policy (retries, backoff, etc.) that will be used in the subscription - this only applies to HTTP/S subscriptions. Refer to the [SNS docs](https://docs.aws.amazon.com/sns/latest/dg/DeliveryPolicies.html) for more details.
     */
    @JvmName("ibygkhtkgbimbqpx")
    public suspend fun deliveryPolicy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.deliveryPolicy = mapped
    }

    /**
     * @param value Endpoint to send data to. The contents vary with the protocol. See details below.
     */
    @JvmName("bmsygomjapjymdsi")
    public suspend fun endpoint(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.endpoint = mapped
    }

    /**
     * @param value Whether the endpoint is capable of [auto confirming subscription](http://docs.aws.amazon.com/sns/latest/dg/SendMessageToHttp.html#SendMessageToHttp.prepare) (e.g., PagerDuty). Default is `false`.
     */
    @JvmName("ucrwvfaikmywpohf")
    public suspend fun endpointAutoConfirms(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.endpointAutoConfirms = mapped
    }

    /**
     * @param value JSON String with the filter policy that will be used in the subscription to filter messages seen by the target resource. Refer to the [SNS docs](https://docs.aws.amazon.com/sns/latest/dg/message-filtering.html) for more details.
     */
    @JvmName("rmfgybgdscqjehwl")
    public suspend fun filterPolicy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.filterPolicy = mapped
    }

    /**
     * @param value Whether the `filter_policy` applies to `MessageAttributes` (default) or `MessageBody`.
     */
    @JvmName("cnchwpwklmprtlqd")
    public suspend fun filterPolicyScope(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.filterPolicyScope = mapped
    }

    /**
     * @param value Protocol to use. Valid values are: `sqs`, `sms`, `lambda`, `firehose`, and `application`. Protocols `email`, `email-json`, `http` and `https` are also valid but partially supported. See details below.
     */
    @JvmName("ywgaphyvxlektiie")
    public suspend fun protocol(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.protocol = mapped
    }

    /**
     * @param value Whether to enable raw message delivery (the original message is directly passed, not wrapped in JSON with the original message in the message property). Default is `false`.
     */
    @JvmName("uyorhtxfqaxvbbrn")
    public suspend fun rawMessageDelivery(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.rawMessageDelivery = mapped
    }

    /**
     * @param value JSON String with the redrive policy that will be used in the subscription. Refer to the [SNS docs](https://docs.aws.amazon.com/sns/latest/dg/sns-dead-letter-queues.html#how-messages-moved-into-dead-letter-queue) for more details.
     */
    @JvmName("tdlorvohkriwwmrg")
    public suspend fun redrivePolicy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.redrivePolicy = mapped
    }

    /**
     * @param value JSON String with the archived message replay policy that will be used in the subscription. Refer to the [SNS docs](https://docs.aws.amazon.com/sns/latest/dg/message-archiving-and-replay-subscriber.html) for more details.
     */
    @JvmName("kljxywgacrjxrnok")
    public suspend fun replayPolicy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.replayPolicy = mapped
    }

    /**
     * @param value ARN of the IAM role to publish to Kinesis Data Firehose delivery stream. Refer to [SNS docs](https://docs.aws.amazon.com/sns/latest/dg/sns-firehose-as-subscriber.html).
     */
    @JvmName("wcatqkpkawrdgfwd")
    public suspend fun subscriptionRoleArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.subscriptionRoleArn = mapped
    }

    /**
     * @param value ARN of the SNS topic to subscribe to.
     * The following arguments are optional:
     */
    @JvmName("wgyfkmkdpgjdknwb")
    public suspend fun topic(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.topic = mapped
    }

    internal fun build(): TopicSubscriptionArgs = TopicSubscriptionArgs(
        confirmationTimeoutInMinutes = confirmationTimeoutInMinutes,
        deliveryPolicy = deliveryPolicy,
        endpoint = endpoint,
        endpointAutoConfirms = endpointAutoConfirms,
        filterPolicy = filterPolicy,
        filterPolicyScope = filterPolicyScope,
        protocol = protocol,
        rawMessageDelivery = rawMessageDelivery,
        redrivePolicy = redrivePolicy,
        replayPolicy = replayPolicy,
        subscriptionRoleArn = subscriptionRoleArn,
        topic = topic,
    )
}
