@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.sqs.kotlin

import com.pulumi.aws.sqs.QueuePolicyArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Allows you to set a policy of an SQS Queue
 * while referencing ARN of the queue within the policy.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const q = new aws.sqs.Queue("q", {name: "examplequeue"});
 * const test = q.arn.apply(arn => aws.iam.getPolicyDocumentOutput({
 *     statements: [{
 *         sid: "First",
 *         effect: "Allow",
 *         principals: [{
 *             type: "*",
 *             identifiers: ["*"],
 *         }],
 *         actions: ["sqs:SendMessage"],
 *         resources: [arn],
 *         conditions: [{
 *             test: "ArnEquals",
 *             variable: "aws:SourceArn",
 *             values: [example&#46;arn],
 *         }],
 *     }],
 * }));
 * const testQueuePolicy = new aws.sqs.QueuePolicy("test", {
 *     queueUrl: q.id,
 *     policy: test.apply(test => test.json),
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * q = aws.sqs.Queue("q", name="examplequeue")
 * test = q.arn.apply(lambda arn: aws.iam.get_policy_document_output(statements=[{
 *     "sid": "First",
 *     "effect": "Allow",
 *     "principals": [{
 *         "type": "*",
 *         "identifiers": ["*"],
 *     }],
 *     "actions": ["sqs:SendMessage"],
 *     "resources": [arn],
 *     "conditions": [{
 *         "test": "ArnEquals",
 *         "variable": "aws:SourceArn",
 *         "values": [example["arn"]],
 *     }],
 * }]))
 * test_queue_policy = aws.sqs.QueuePolicy("test",
 *     queue_url=q.id,
 *     policy=test.json)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var q = new Aws.Sqs.Queue("q", new()
 *     {
 *         Name = "examplequeue",
 *     });
 *     var test = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Sid = "First",
 *                 Effect = "Allow",
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Type = "*",
 *                         Identifiers = new[]
 *                         {
 *                             "*",
 *                         },
 *                     },
 *                 },
 *                 Actions = new[]
 *                 {
 *                     "sqs:SendMessage",
 *                 },
 *                 Resources = new[]
 *                 {
 *                     q.Arn,
 *                 },
 *                 Conditions = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementConditionInputArgs
 *                     {
 *                         Test = "ArnEquals",
 *                         Variable = "aws:SourceArn",
 *                         Values = new[]
 *                         {
 *                             example.Arn,
 *                         },
 *                     },
 *                 },
 *             },
 *         },
 *     });
 *     var testQueuePolicy = new Aws.Sqs.QueuePolicy("test", new()
 *     {
 *         QueueUrl = q.Id,
 *         Policy = test.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/sqs"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * pulumi.Run(func(ctx *pulumi.Context) error {
 * q, err := sqs.NewQueue(ctx, "q", &sqs.QueueArgs{
 * Name: pulumi.String("examplequeue"),
 * })
 * if err != nil {
 * return err
 * }
 * test := q.Arn.ApplyT(func(arn string) (iam.GetPolicyDocumentResult, error) {
 * return iam.GetPolicyDocumentResult(interface{}(iam.GetPolicyDocumentOutput(ctx, iam.GetPolicyDocumentOutputArgs{
 * Statements: []iam.GetPolicyDocumentStatement{
 * {
 * Sid: "First",
 * Effect: "Allow",
 * Principals: []iam.GetPolicyDocumentStatementPrincipal{
 * {
 * Type: "*",
 * Identifiers: []string{
 * "*",
 * },
 * },
 * },
 * Actions: []string{
 * "sqs:SendMessage",
 * },
 * Resources: []string{
 * arn,
 * },
 * Conditions: []iam.GetPolicyDocumentStatementCondition{
 * {
 * Test: "ArnEquals",
 * Variable: "aws:SourceArn",
 * Values: interface{}{
 * example.Arn,
 * },
 * },
 * },
 * },
 * },
 * }, nil))), nil
 * }).(iam.GetPolicyDocumentResultOutput)
 * _, err = sqs.NewQueuePolicy(ctx, "test", &sqs.QueuePolicyArgs{
 * QueueUrl: q.ID(),
 * Policy: pulumi.String(test.ApplyT(func(test iam.GetPolicyDocumentResult) (*string, error) {
 * return &test.Json, nil
 * }).(pulumi.StringPtrOutput)),
 * })
 * if err != nil {
 * return err
 * }
 * return nil
 * })
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.sqs.Queue;
 * import com.pulumi.aws.sqs.QueueArgs;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.sqs.QueuePolicy;
 * import com.pulumi.aws.sqs.QueuePolicyArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var q = new Queue("q", QueueArgs.builder()
 *             .name("examplequeue")
 *             .build());
 *         final var test = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .sid("First")
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("*")
 *                     .identifiers("*")
 *                     .build())
 *                 .actions("sqs:SendMessage")
 *                 .resources(q.arn())
 *                 .conditions(GetPolicyDocumentStatementConditionArgs.builder()
 *                     .test("ArnEquals")
 *                     .variable("aws:SourceArn")
 *                     .values(example.arn())
 *                     .build())
 *                 .build())
 *             .build());
 *         var testQueuePolicy = new QueuePolicy("testQueuePolicy", QueuePolicyArgs.builder()
 *             .queueUrl(q.id())
 *             .policy(test.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult).applyValue(test -> test.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json())))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   q:
 *     type: aws:sqs:Queue
 *     properties:
 *       name: examplequeue
 *   testQueuePolicy:
 *     type: aws:sqs:QueuePolicy
 *     name: test
 *     properties:
 *       queueUrl: ${q.id}
 *       policy: ${test.json}
 * variables:
 *   test:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - sid: First
 *             effect: Allow
 *             principals:
 *               - type: '*'
 *                 identifiers:
 *                   - '*'
 *             actions:
 *               - sqs:SendMessage
 *             resources:
 *               - ${q.arn}
 *             conditions:
 *               - test: ArnEquals
 *                 variable: aws:SourceArn
 *                 values:
 *                   - ${example.arn}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import SQS Queue Policies using the queue URL. For example:
 * ```sh
 * $ pulumi import aws:sqs/queuePolicy:QueuePolicy test https://queue.amazonaws.com/0123456789012/myqueue
 * ```
 * @property policy The JSON policy for the SQS queue.
 * @property queueUrl The URL of the SQS Queue to which to attach the policy
 */
public data class QueuePolicyArgs(
    public val policy: Output<String>? = null,
    public val queueUrl: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.sqs.QueuePolicyArgs> {
    override fun toJava(): com.pulumi.aws.sqs.QueuePolicyArgs =
        com.pulumi.aws.sqs.QueuePolicyArgs.builder()
            .policy(policy?.applyValue({ args0 -> args0 }))
            .queueUrl(queueUrl?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [QueuePolicyArgs].
 */
@PulumiTagMarker
public class QueuePolicyArgsBuilder internal constructor() {
    private var policy: Output<String>? = null

    private var queueUrl: Output<String>? = null

    /**
     * @param value The JSON policy for the SQS queue.
     */
    @JvmName("floydxymavgnmnyw")
    public suspend fun policy(`value`: Output<String>) {
        this.policy = value
    }

    /**
     * @param value The URL of the SQS Queue to which to attach the policy
     */
    @JvmName("saowqqljcnkcqjnd")
    public suspend fun queueUrl(`value`: Output<String>) {
        this.queueUrl = value
    }

    /**
     * @param value The JSON policy for the SQS queue.
     */
    @JvmName("hbdhdqtcxqtuvxpk")
    public suspend fun policy(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.policy = mapped
    }

    /**
     * @param value The URL of the SQS Queue to which to attach the policy
     */
    @JvmName("atpacuypqdacyovx")
    public suspend fun queueUrl(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.queueUrl = mapped
    }

    internal fun build(): QueuePolicyArgs = QueuePolicyArgs(
        policy = policy,
        queueUrl = queueUrl,
    )
}
