@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.ssoadmin.kotlin

import com.pulumi.aws.ssoadmin.ManagedPolicyAttachmentArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Provides an IAM managed policy for a Single Sign-On (SSO) Permission Set resource
 * > **NOTE:** Creating this resource will automatically [Provision the Permission Set](https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ProvisionPermissionSet.html) to apply the corresponding updates to all assigned accounts.
 * ## Example Usage
 * ### Basic Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = aws.ssoadmin.getInstances({});
 * const examplePermissionSet = new aws.ssoadmin.PermissionSet("example", {
 *     name: "Example",
 *     instanceArn: example.then(example => example.arns?.[0]),
 * });
 * const exampleManagedPolicyAttachment = new aws.ssoadmin.ManagedPolicyAttachment("example", {
 *     instanceArn: example.then(example => example.arns?.[0]),
 *     managedPolicyArn: "arn:aws:iam::aws:policy/AlexaForBusinessDeviceSetup",
 *     permissionSetArn: examplePermissionSet.arn,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.ssoadmin.get_instances()
 * example_permission_set = aws.ssoadmin.PermissionSet("example",
 *     name="Example",
 *     instance_arn=example.arns[0])
 * example_managed_policy_attachment = aws.ssoadmin.ManagedPolicyAttachment("example",
 *     instance_arn=example.arns[0],
 *     managed_policy_arn="arn:aws:iam::aws:policy/AlexaForBusinessDeviceSetup",
 *     permission_set_arn=example_permission_set.arn)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = Aws.SsoAdmin.GetInstances.Invoke();
 *     var examplePermissionSet = new Aws.SsoAdmin.PermissionSet("example", new()
 *     {
 *         Name = "Example",
 *         InstanceArn = example.Apply(getInstancesResult => getInstancesResult.Arns[0]),
 *     });
 *     var exampleManagedPolicyAttachment = new Aws.SsoAdmin.ManagedPolicyAttachment("example", new()
 *     {
 *         InstanceArn = example.Apply(getInstancesResult => getInstancesResult.Arns[0]),
 *         ManagedPolicyArn = "arn:aws:iam::aws:policy/AlexaForBusinessDeviceSetup",
 *         PermissionSetArn = examplePermissionSet.Arn,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ssoadmin"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := ssoadmin.GetInstances(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		examplePermissionSet, err := ssoadmin.NewPermissionSet(ctx, "example", &ssoadmin.PermissionSetArgs{
 * 			Name:        pulumi.String("Example"),
 * 			InstanceArn: pulumi.String(example.Arns[0]),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ssoadmin.NewManagedPolicyAttachment(ctx, "example", &ssoadmin.ManagedPolicyAttachmentArgs{
 * 			InstanceArn:      pulumi.String(example.Arns[0]),
 * 			ManagedPolicyArn: pulumi.String("arn:aws:iam::aws:policy/AlexaForBusinessDeviceSetup"),
 * 			PermissionSetArn: examplePermissionSet.Arn,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ssoadmin.SsoadminFunctions;
 * import com.pulumi.aws.ssoadmin.PermissionSet;
 * import com.pulumi.aws.ssoadmin.PermissionSetArgs;
 * import com.pulumi.aws.ssoadmin.ManagedPolicyAttachment;
 * import com.pulumi.aws.ssoadmin.ManagedPolicyAttachmentArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var example = SsoadminFunctions.getInstances();
 *         var examplePermissionSet = new PermissionSet("examplePermissionSet", PermissionSetArgs.builder()
 *             .name("Example")
 *             .instanceArn(example.applyValue(getInstancesResult -> getInstancesResult.arns()[0]))
 *             .build());
 *         var exampleManagedPolicyAttachment = new ManagedPolicyAttachment("exampleManagedPolicyAttachment", ManagedPolicyAttachmentArgs.builder()
 *             .instanceArn(example.applyValue(getInstancesResult -> getInstancesResult.arns()[0]))
 *             .managedPolicyArn("arn:aws:iam::aws:policy/AlexaForBusinessDeviceSetup")
 *             .permissionSetArn(examplePermissionSet.arn())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   examplePermissionSet:
 *     type: aws:ssoadmin:PermissionSet
 *     name: example
 *     properties:
 *       name: Example
 *       instanceArn: ${example.arns[0]}
 *   exampleManagedPolicyAttachment:
 *     type: aws:ssoadmin:ManagedPolicyAttachment
 *     name: example
 *     properties:
 *       instanceArn: ${example.arns[0]}
 *       managedPolicyArn: arn:aws:iam::aws:policy/AlexaForBusinessDeviceSetup
 *       permissionSetArn: ${examplePermissionSet.arn}
 * variables:
 *   example:
 *     fn::invoke:
 *       Function: aws:ssoadmin:getInstances
 *       Arguments: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### With Account Assignment
 * > Because destruction of a managed policy attachment resource also re-provisions the associated permission set to all accounts, explicitly indicating the dependency with the account assignment resource via the `depends_on` meta argument is necessary to ensure proper deletion order when these resources are used together.
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = aws.ssoadmin.getInstances({});
 * const examplePermissionSet = new aws.ssoadmin.PermissionSet("example", {
 *     name: "Example",
 *     instanceArn: example.then(example => example.arns?.[0]),
 * });
 * const exampleGroup = new aws.identitystore.Group("example", {
 *     identityStoreId: example.then(example => example.identityStoreIds?.[0]),
 *     displayName: "Admin",
 *     description: "Admin Group",
 * });
 * const accountAssignment = new aws.ssoadmin.AccountAssignment("account_assignment", {
 *     instanceArn: example.then(example => example.arns?.[0]),
 *     permissionSetArn: examplePermissionSet.arn,
 *     principalId: exampleGroup.groupId,
 *     principalType: "GROUP",
 *     targetId: "123456789012",
 *     targetType: "AWS_ACCOUNT",
 * });
 * const exampleManagedPolicyAttachment = new aws.ssoadmin.ManagedPolicyAttachment("example", {
 *     instanceArn: example.then(example => example.arns?.[0]),
 *     managedPolicyArn: "arn:aws:iam::aws:policy/AlexaForBusinessDeviceSetup",
 *     permissionSetArn: examplePermissionSet.arn,
 * }, {
 *     dependsOn: [exampleAwsSsoadminAccountAssignment],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.ssoadmin.get_instances()
 * example_permission_set = aws.ssoadmin.PermissionSet("example",
 *     name="Example",
 *     instance_arn=example.arns[0])
 * example_group = aws.identitystore.Group("example",
 *     identity_store_id=example.identity_store_ids[0],
 *     display_name="Admin",
 *     description="Admin Group")
 * account_assignment = aws.ssoadmin.AccountAssignment("account_assignment",
 *     instance_arn=example.arns[0],
 *     permission_set_arn=example_permission_set.arn,
 *     principal_id=example_group.group_id,
 *     principal_type="GROUP",
 *     target_id="123456789012",
 *     target_type="AWS_ACCOUNT")
 * example_managed_policy_attachment = aws.ssoadmin.ManagedPolicyAttachment("example",
 *     instance_arn=example.arns[0],
 *     managed_policy_arn="arn:aws:iam::aws:policy/AlexaForBusinessDeviceSetup",
 *     permission_set_arn=example_permission_set.arn,
 *     opts = pulumi.ResourceOptions(depends_on=[example_aws_ssoadmin_account_assignment]))
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = Aws.SsoAdmin.GetInstances.Invoke();
 *     var examplePermissionSet = new Aws.SsoAdmin.PermissionSet("example", new()
 *     {
 *         Name = "Example",
 *         InstanceArn = example.Apply(getInstancesResult => getInstancesResult.Arns[0]),
 *     });
 *     var exampleGroup = new Aws.IdentityStore.Group("example", new()
 *     {
 *         IdentityStoreId = example.Apply(getInstancesResult => getInstancesResult.IdentityStoreIds[0]),
 *         DisplayName = "Admin",
 *         Description = "Admin Group",
 *     });
 *     var accountAssignment = new Aws.SsoAdmin.AccountAssignment("account_assignment", new()
 *     {
 *         InstanceArn = example.Apply(getInstancesResult => getInstancesResult.Arns[0]),
 *         PermissionSetArn = examplePermissionSet.Arn,
 *         PrincipalId = exampleGroup.GroupId,
 *         PrincipalType = "GROUP",
 *         TargetId = "123456789012",
 *         TargetType = "AWS_ACCOUNT",
 *     });
 *     var exampleManagedPolicyAttachment = new Aws.SsoAdmin.ManagedPolicyAttachment("example", new()
 *     {
 *         InstanceArn = example.Apply(getInstancesResult => getInstancesResult.Arns[0]),
 *         ManagedPolicyArn = "arn:aws:iam::aws:policy/AlexaForBusinessDeviceSetup",
 *         PermissionSetArn = examplePermissionSet.Arn,
 *     }, new CustomResourceOptions
 *     {
 *         DependsOn =
 *         {
 *             exampleAwsSsoadminAccountAssignment,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/identitystore"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/ssoadmin"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := ssoadmin.GetInstances(ctx, nil, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		examplePermissionSet, err := ssoadmin.NewPermissionSet(ctx, "example", &ssoadmin.PermissionSetArgs{
 * 			Name:        pulumi.String("Example"),
 * 			InstanceArn: pulumi.String(example.Arns[0]),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		exampleGroup, err := identitystore.NewGroup(ctx, "example", &identitystore.GroupArgs{
 * 			IdentityStoreId: pulumi.String(example.IdentityStoreIds[0]),
 * 			DisplayName:     pulumi.String("Admin"),
 * 			Description:     pulumi.String("Admin Group"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ssoadmin.NewAccountAssignment(ctx, "account_assignment", &ssoadmin.AccountAssignmentArgs{
 * 			InstanceArn:      pulumi.String(example.Arns[0]),
 * 			PermissionSetArn: examplePermissionSet.Arn,
 * 			PrincipalId:      exampleGroup.GroupId,
 * 			PrincipalType:    pulumi.String("GROUP"),
 * 			TargetId:         pulumi.String("123456789012"),
 * 			TargetType:       pulumi.String("AWS_ACCOUNT"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = ssoadmin.NewManagedPolicyAttachment(ctx, "example", &ssoadmin.ManagedPolicyAttachmentArgs{
 * 			InstanceArn:      pulumi.String(example.Arns[0]),
 * 			ManagedPolicyArn: pulumi.String("arn:aws:iam::aws:policy/AlexaForBusinessDeviceSetup"),
 * 			PermissionSetArn: examplePermissionSet.Arn,
 * 		}, pulumi.DependsOn([]pulumi.Resource{
 * 			exampleAwsSsoadminAccountAssignment,
 * 		}))
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.ssoadmin.SsoadminFunctions;
 * import com.pulumi.aws.ssoadmin.PermissionSet;
 * import com.pulumi.aws.ssoadmin.PermissionSetArgs;
 * import com.pulumi.aws.identitystore.Group;
 * import com.pulumi.aws.identitystore.GroupArgs;
 * import com.pulumi.aws.ssoadmin.AccountAssignment;
 * import com.pulumi.aws.ssoadmin.AccountAssignmentArgs;
 * import com.pulumi.aws.ssoadmin.ManagedPolicyAttachment;
 * import com.pulumi.aws.ssoadmin.ManagedPolicyAttachmentArgs;
 * import com.pulumi.resources.CustomResourceOptions;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var example = SsoadminFunctions.getInstances();
 *         var examplePermissionSet = new PermissionSet("examplePermissionSet", PermissionSetArgs.builder()
 *             .name("Example")
 *             .instanceArn(example.applyValue(getInstancesResult -> getInstancesResult.arns()[0]))
 *             .build());
 *         var exampleGroup = new Group("exampleGroup", GroupArgs.builder()
 *             .identityStoreId(example.applyValue(getInstancesResult -> getInstancesResult.identityStoreIds()[0]))
 *             .displayName("Admin")
 *             .description("Admin Group")
 *             .build());
 *         var accountAssignment = new AccountAssignment("accountAssignment", AccountAssignmentArgs.builder()
 *             .instanceArn(example.applyValue(getInstancesResult -> getInstancesResult.arns()[0]))
 *             .permissionSetArn(examplePermissionSet.arn())
 *             .principalId(exampleGroup.groupId())
 *             .principalType("GROUP")
 *             .targetId("123456789012")
 *             .targetType("AWS_ACCOUNT")
 *             .build());
 *         var exampleManagedPolicyAttachment = new ManagedPolicyAttachment("exampleManagedPolicyAttachment", ManagedPolicyAttachmentArgs.builder()
 *             .instanceArn(example.applyValue(getInstancesResult -> getInstancesResult.arns()[0]))
 *             .managedPolicyArn("arn:aws:iam::aws:policy/AlexaForBusinessDeviceSetup")
 *             .permissionSetArn(examplePermissionSet.arn())
 *             .build(), CustomResourceOptions.builder()
 *                 .dependsOn(exampleAwsSsoadminAccountAssignment)
 *                 .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   examplePermissionSet:
 *     type: aws:ssoadmin:PermissionSet
 *     name: example
 *     properties:
 *       name: Example
 *       instanceArn: ${example.arns[0]}
 *   exampleGroup:
 *     type: aws:identitystore:Group
 *     name: example
 *     properties:
 *       identityStoreId: ${example.identityStoreIds[0]}
 *       displayName: Admin
 *       description: Admin Group
 *   accountAssignment:
 *     type: aws:ssoadmin:AccountAssignment
 *     name: account_assignment
 *     properties:
 *       instanceArn: ${example.arns[0]}
 *       permissionSetArn: ${examplePermissionSet.arn}
 *       principalId: ${exampleGroup.groupId}
 *       principalType: GROUP
 *       targetId: '123456789012'
 *       targetType: AWS_ACCOUNT
 *   exampleManagedPolicyAttachment:
 *     type: aws:ssoadmin:ManagedPolicyAttachment
 *     name: example
 *     properties:
 *       instanceArn: ${example.arns[0]}
 *       managedPolicyArn: arn:aws:iam::aws:policy/AlexaForBusinessDeviceSetup
 *       permissionSetArn: ${examplePermissionSet.arn}
 *     options:
 *       dependson:
 *         - ${exampleAwsSsoadminAccountAssignment}
 * variables:
 *   example:
 *     fn::invoke:
 *       Function: aws:ssoadmin:getInstances
 *       Arguments: {}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import SSO Managed Policy Attachments using the `managed_policy_arn`, `permission_set_arn`, and `instance_arn` separated by a comma (`,`). For example:
 * ```sh
 * $ pulumi import aws:ssoadmin/managedPolicyAttachment:ManagedPolicyAttachment example arn:aws:iam::aws:policy/AlexaForBusinessDeviceSetup,arn:aws:sso:::permissionSet/ssoins-2938j0x8920sbj72/ps-80383020jr9302rk,arn:aws:sso:::instance/ssoins-2938j0x8920sbj72
 * ```
 * @property instanceArn The Amazon Resource Name (ARN) of the SSO Instance under which the operation will be executed.
 * @property managedPolicyArn The IAM managed policy Amazon Resource Name (ARN) to be attached to the Permission Set.
 * @property permissionSetArn The Amazon Resource Name (ARN) of the Permission Set.
 */
public data class ManagedPolicyAttachmentArgs(
    public val instanceArn: Output<String>? = null,
    public val managedPolicyArn: Output<String>? = null,
    public val permissionSetArn: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.ssoadmin.ManagedPolicyAttachmentArgs> {
    override fun toJava(): com.pulumi.aws.ssoadmin.ManagedPolicyAttachmentArgs =
        com.pulumi.aws.ssoadmin.ManagedPolicyAttachmentArgs.builder()
            .instanceArn(instanceArn?.applyValue({ args0 -> args0 }))
            .managedPolicyArn(managedPolicyArn?.applyValue({ args0 -> args0 }))
            .permissionSetArn(permissionSetArn?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [ManagedPolicyAttachmentArgs].
 */
@PulumiTagMarker
public class ManagedPolicyAttachmentArgsBuilder internal constructor() {
    private var instanceArn: Output<String>? = null

    private var managedPolicyArn: Output<String>? = null

    private var permissionSetArn: Output<String>? = null

    /**
     * @param value The Amazon Resource Name (ARN) of the SSO Instance under which the operation will be executed.
     */
    @JvmName("tpgtilnnfpvlxytc")
    public suspend fun instanceArn(`value`: Output<String>) {
        this.instanceArn = value
    }

    /**
     * @param value The IAM managed policy Amazon Resource Name (ARN) to be attached to the Permission Set.
     */
    @JvmName("nnodtmeaprbvdddy")
    public suspend fun managedPolicyArn(`value`: Output<String>) {
        this.managedPolicyArn = value
    }

    /**
     * @param value The Amazon Resource Name (ARN) of the Permission Set.
     */
    @JvmName("jauerwsixipklehe")
    public suspend fun permissionSetArn(`value`: Output<String>) {
        this.permissionSetArn = value
    }

    /**
     * @param value The Amazon Resource Name (ARN) of the SSO Instance under which the operation will be executed.
     */
    @JvmName("uwhjsgwolmbkgcds")
    public suspend fun instanceArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.instanceArn = mapped
    }

    /**
     * @param value The IAM managed policy Amazon Resource Name (ARN) to be attached to the Permission Set.
     */
    @JvmName("udpaisvjjorweydh")
    public suspend fun managedPolicyArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.managedPolicyArn = mapped
    }

    /**
     * @param value The Amazon Resource Name (ARN) of the Permission Set.
     */
    @JvmName("unomouoncbgjeobc")
    public suspend fun permissionSetArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.permissionSetArn = mapped
    }

    internal fun build(): ManagedPolicyAttachmentArgs = ManagedPolicyAttachmentArgs(
        instanceArn = instanceArn,
        managedPolicyArn = managedPolicyArn,
        permissionSetArn = permissionSetArn,
    )
}
