@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.transfer.kotlin

import com.pulumi.aws.transfer.ServerArgs.builder
import com.pulumi.aws.transfer.kotlin.inputs.ServerEndpointDetailsArgs
import com.pulumi.aws.transfer.kotlin.inputs.ServerEndpointDetailsArgsBuilder
import com.pulumi.aws.transfer.kotlin.inputs.ServerProtocolDetailsArgs
import com.pulumi.aws.transfer.kotlin.inputs.ServerProtocolDetailsArgsBuilder
import com.pulumi.aws.transfer.kotlin.inputs.ServerS3StorageOptionsArgs
import com.pulumi.aws.transfer.kotlin.inputs.ServerS3StorageOptionsArgsBuilder
import com.pulumi.aws.transfer.kotlin.inputs.ServerWorkflowDetailsArgs
import com.pulumi.aws.transfer.kotlin.inputs.ServerWorkflowDetailsArgsBuilder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * Provides a AWS Transfer Server resource.
 * > **NOTE on AWS IAM permissions:** If the `endpoint_type` is set to `VPC`, the `ec2:DescribeVpcEndpoints` and `ec2:ModifyVpcEndpoint` [actions](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonec2.html#amazonec2-actions-as-permissions) are used.
 * > **NOTE:** Use the `aws.transfer.Tag` resource to manage the system tags used for [custom hostnames](https://docs.aws.amazon.com/transfer/latest/userguide/requirements-dns.html#tag-custom-hostname-cdk).
 * ## Example Usage
 * ### Basic
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.transfer.Server("example", {tags: {
 *     Name: "Example",
 * }});
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.transfer.Server("example", tags={
 *     "Name": "Example",
 * })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Transfer.Server("example", new()
 *     {
 *         Tags =
 *         {
 *             { "Name", "Example" },
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/transfer"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := transfer.NewServer(ctx, "example", &transfer.ServerArgs{
 * 			Tags: pulumi.StringMap{
 * 				"Name": pulumi.String("Example"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.transfer.Server;
 * import com.pulumi.aws.transfer.ServerArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Server("example", ServerArgs.builder()
 *             .tags(Map.of("Name", "Example"))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:transfer:Server
 *     properties:
 *       tags:
 *         Name: Example
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Security Policy Name
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.transfer.Server("example", {securityPolicyName: "TransferSecurityPolicy-2020-06"});
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.transfer.Server("example", security_policy_name="TransferSecurityPolicy-2020-06")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Transfer.Server("example", new()
 *     {
 *         SecurityPolicyName = "TransferSecurityPolicy-2020-06",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/transfer"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := transfer.NewServer(ctx, "example", &transfer.ServerArgs{
 * 			SecurityPolicyName: pulumi.String("TransferSecurityPolicy-2020-06"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.transfer.Server;
 * import com.pulumi.aws.transfer.ServerArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Server("example", ServerArgs.builder()
 *             .securityPolicyName("TransferSecurityPolicy-2020-06")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:transfer:Server
 *     properties:
 *       securityPolicyName: TransferSecurityPolicy-2020-06
 * ```
 * <!--End PulumiCodeChooser -->
 * ### VPC Endpoint
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.transfer.Server("example", {
 *     endpointType: "VPC",
 *     endpointDetails: {
 *         addressAllocationIds: [exampleAwsEip&#46;id],
 *         subnetIds: [exampleAwsSubnet&#46;id],
 *         vpcId: exampleAwsVpc.id,
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.transfer.Server("example",
 *     endpoint_type="VPC",
 *     endpoint_details={
 *         "address_allocation_ids": [example_aws_eip["id"]],
 *         "subnet_ids": [example_aws_subnet["id"]],
 *         "vpc_id": example_aws_vpc["id"],
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Transfer.Server("example", new()
 *     {
 *         EndpointType = "VPC",
 *         EndpointDetails = new Aws.Transfer.Inputs.ServerEndpointDetailsArgs
 *         {
 *             AddressAllocationIds = new[]
 *             {
 *                 exampleAwsEip.Id,
 *             },
 *             SubnetIds = new[]
 *             {
 *                 exampleAwsSubnet.Id,
 *             },
 *             VpcId = exampleAwsVpc.Id,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/transfer"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := transfer.NewServer(ctx, "example", &transfer.ServerArgs{
 * 			EndpointType: pulumi.String("VPC"),
 * 			EndpointDetails: &transfer.ServerEndpointDetailsArgs{
 * 				AddressAllocationIds: pulumi.StringArray{
 * 					exampleAwsEip.Id,
 * 				},
 * 				SubnetIds: pulumi.StringArray{
 * 					exampleAwsSubnet.Id,
 * 				},
 * 				VpcId: pulumi.Any(exampleAwsVpc.Id),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.transfer.Server;
 * import com.pulumi.aws.transfer.ServerArgs;
 * import com.pulumi.aws.transfer.inputs.ServerEndpointDetailsArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Server("example", ServerArgs.builder()
 *             .endpointType("VPC")
 *             .endpointDetails(ServerEndpointDetailsArgs.builder()
 *                 .addressAllocationIds(exampleAwsEip.id())
 *                 .subnetIds(exampleAwsSubnet.id())
 *                 .vpcId(exampleAwsVpc.id())
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:transfer:Server
 *     properties:
 *       endpointType: VPC
 *       endpointDetails:
 *         addressAllocationIds:
 *           - ${exampleAwsEip.id}
 *         subnetIds:
 *           - ${exampleAwsSubnet.id}
 *         vpcId: ${exampleAwsVpc.id}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### AWS Directory authentication
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.transfer.Server("example", {
 *     identityProviderType: "AWS_DIRECTORY_SERVICE",
 *     directoryId: exampleAwsDirectoryServiceDirectory.id,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.transfer.Server("example",
 *     identity_provider_type="AWS_DIRECTORY_SERVICE",
 *     directory_id=example_aws_directory_service_directory["id"])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Transfer.Server("example", new()
 *     {
 *         IdentityProviderType = "AWS_DIRECTORY_SERVICE",
 *         DirectoryId = exampleAwsDirectoryServiceDirectory.Id,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/transfer"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := transfer.NewServer(ctx, "example", &transfer.ServerArgs{
 * 			IdentityProviderType: pulumi.String("AWS_DIRECTORY_SERVICE"),
 * 			DirectoryId:          pulumi.Any(exampleAwsDirectoryServiceDirectory.Id),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.transfer.Server;
 * import com.pulumi.aws.transfer.ServerArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Server("example", ServerArgs.builder()
 *             .identityProviderType("AWS_DIRECTORY_SERVICE")
 *             .directoryId(exampleAwsDirectoryServiceDirectory.id())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:transfer:Server
 *     properties:
 *       identityProviderType: AWS_DIRECTORY_SERVICE
 *       directoryId: ${exampleAwsDirectoryServiceDirectory.id}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### AWS Lambda authentication
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.transfer.Server("example", {
 *     identityProviderType: "AWS_LAMBDA",
 *     "function": exampleAwsLambdaIdentityProvider.arn,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.transfer.Server("example",
 *     identity_provider_type="AWS_LAMBDA",
 *     function=example_aws_lambda_identity_provider["arn"])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Transfer.Server("example", new()
 *     {
 *         IdentityProviderType = "AWS_LAMBDA",
 *         Function = exampleAwsLambdaIdentityProvider.Arn,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/transfer"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := transfer.NewServer(ctx, "example", &transfer.ServerArgs{
 * 			IdentityProviderType: pulumi.String("AWS_LAMBDA"),
 * 			Function:             pulumi.Any(exampleAwsLambdaIdentityProvider.Arn),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.transfer.Server;
 * import com.pulumi.aws.transfer.ServerArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Server("example", ServerArgs.builder()
 *             .identityProviderType("AWS_LAMBDA")
 *             .function(exampleAwsLambdaIdentityProvider.arn())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:transfer:Server
 *     properties:
 *       identityProviderType: AWS_LAMBDA
 *       function: ${exampleAwsLambdaIdentityProvider.arn}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Protocols
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.transfer.Server("example", {
 *     endpointType: "VPC",
 *     endpointDetails: {
 *         subnetIds: [exampleAwsSubnet&#46;id],
 *         vpcId: exampleAwsVpc.id,
 *     },
 *     protocols: [
 *         "FTP",
 *         "FTPS",
 *     ],
 *     certificate: exampleAwsAcmCertificate.arn,
 *     identityProviderType: "API_GATEWAY",
 *     url: `${exampleAwsApiGatewayDeployment.invokeUrl}${exampleAwsApiGatewayResource.path}`,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.transfer.Server("example",
 *     endpoint_type="VPC",
 *     endpoint_details={
 *         "subnet_ids": [example_aws_subnet["id"]],
 *         "vpc_id": example_aws_vpc["id"],
 *     },
 *     protocols=[
 *         "FTP",
 *         "FTPS",
 *     ],
 *     certificate=example_aws_acm_certificate["arn"],
 *     identity_provider_type="API_GATEWAY",
 *     url=f"{example_aws_api_gateway_deployment['invokeUrl']}{example_aws_api_gateway_resource['path']}")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Transfer.Server("example", new()
 *     {
 *         EndpointType = "VPC",
 *         EndpointDetails = new Aws.Transfer.Inputs.ServerEndpointDetailsArgs
 *         {
 *             SubnetIds = new[]
 *             {
 *                 exampleAwsSubnet.Id,
 *             },
 *             VpcId = exampleAwsVpc.Id,
 *         },
 *         Protocols = new[]
 *         {
 *             "FTP",
 *             "FTPS",
 *         },
 *         Certificate = exampleAwsAcmCertificate.Arn,
 *         IdentityProviderType = "API_GATEWAY",
 *         Url = $"{exampleAwsApiGatewayDeployment.InvokeUrl}{exampleAwsApiGatewayResource.Path}",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/transfer"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := transfer.NewServer(ctx, "example", &transfer.ServerArgs{
 * 			EndpointType: pulumi.String("VPC"),
 * 			EndpointDetails: &transfer.ServerEndpointDetailsArgs{
 * 				SubnetIds: pulumi.StringArray{
 * 					exampleAwsSubnet.Id,
 * 				},
 * 				VpcId: pulumi.Any(exampleAwsVpc.Id),
 * 			},
 * 			Protocols: pulumi.StringArray{
 * 				pulumi.String("FTP"),
 * 				pulumi.String("FTPS"),
 * 			},
 * 			Certificate:          pulumi.Any(exampleAwsAcmCertificate.Arn),
 * 			IdentityProviderType: pulumi.String("API_GATEWAY"),
 * 			Url:                  pulumi.Sprintf("%v%v", exampleAwsApiGatewayDeployment.InvokeUrl, exampleAwsApiGatewayResource.Path),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.transfer.Server;
 * import com.pulumi.aws.transfer.ServerArgs;
 * import com.pulumi.aws.transfer.inputs.ServerEndpointDetailsArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Server("example", ServerArgs.builder()
 *             .endpointType("VPC")
 *             .endpointDetails(ServerEndpointDetailsArgs.builder()
 *                 .subnetIds(exampleAwsSubnet.id())
 *                 .vpcId(exampleAwsVpc.id())
 *                 .build())
 *             .protocols(
 *                 "FTP",
 *                 "FTPS")
 *             .certificate(exampleAwsAcmCertificate.arn())
 *             .identityProviderType("API_GATEWAY")
 *             .url(String.format("%s%s", exampleAwsApiGatewayDeployment.invokeUrl(),exampleAwsApiGatewayResource.path()))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:transfer:Server
 *     properties:
 *       endpointType: VPC
 *       endpointDetails:
 *         subnetIds:
 *           - ${exampleAwsSubnet.id}
 *         vpcId: ${exampleAwsVpc.id}
 *       protocols:
 *         - FTP
 *         - FTPS
 *       certificate: ${exampleAwsAcmCertificate.arn}
 *       identityProviderType: API_GATEWAY
 *       url: ${exampleAwsApiGatewayDeployment.invokeUrl}${exampleAwsApiGatewayResource.path}
 * ```
 * <!--End PulumiCodeChooser -->
 * ### Using Structured Logging Destinations
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const transfer = new aws.cloudwatch.LogGroup("transfer", {namePrefix: "transfer_test_"});
 * const transferAssumeRole = aws.iam.getPolicyDocument({
 *     statements: [{
 *         effect: "Allow",
 *         principals: [{
 *             type: "Service",
 *             identifiers: ["transfer&#46;amazonaws&#46;com"],
 *         }],
 *         actions: ["sts:AssumeRole"],
 *     }],
 * });
 * const iamForTransfer = new aws.iam.Role("iam_for_transfer", {
 *     namePrefix: "iam_for_transfer_",
 *     assumeRolePolicy: transferAssumeRole.then(transferAssumeRole => transferAssumeRole.json),
 *     managedPolicyArns: ["arn:aws:iam::aws:policy/service-role/AWSTransferLoggingAccess"],
 * });
 * const transferServer = new aws.transfer.Server("transfer", {
 *     endpointType: "PUBLIC",
 *     loggingRole: iamForTransfer.arn,
 *     protocols: ["SFTP"],
 *     structuredLogDestinations: [pulumi&#46;interpolate`${transfer&#46;arn}:*`],
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * transfer = aws.cloudwatch.LogGroup("transfer", name_prefix="transfer_test_")
 * transfer_assume_role = aws.iam.get_policy_document(statements=[{
 *     "effect": "Allow",
 *     "principals": [{
 *         "type": "Service",
 *         "identifiers": ["transfer&#46;amazonaws&#46;com"],
 *     }],
 *     "actions": ["sts:AssumeRole"],
 * }])
 * iam_for_transfer = aws.iam.Role("iam_for_transfer",
 *     name_prefix="iam_for_transfer_",
 *     assume_role_policy=transfer_assume_role.json,
 *     managed_policy_arns=["arn:aws:iam::aws:policy/service-role/AWSTransferLoggingAccess"])
 * transfer_server = aws.transfer.Server("transfer",
 *     endpoint_type="PUBLIC",
 *     logging_role=iam_for_transfer.arn,
 *     protocols=["SFTP"],
 *     structured_log_destinations=[transfer&#46;arn&#46;apply(lambda arn: f"{arn}:*")])
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var transfer = new Aws.CloudWatch.LogGroup("transfer", new()
 *     {
 *         NamePrefix = "transfer_test_",
 *     });
 *     var transferAssumeRole = Aws.Iam.GetPolicyDocument.Invoke(new()
 *     {
 *         Statements = new[]
 *         {
 *             new Aws.Iam.Inputs.GetPolicyDocumentStatementInputArgs
 *             {
 *                 Effect = "Allow",
 *                 Principals = new[]
 *                 {
 *                     new Aws.Iam.Inputs.GetPolicyDocumentStatementPrincipalInputArgs
 *                     {
 *                         Type = "Service",
 *                         Identifiers = new[]
 *                         {
 *                             "transfer.amazonaws.com",
 *                         },
 *                     },
 *                 },
 *                 Actions = new[]
 *                 {
 *                     "sts:AssumeRole",
 *                 },
 *             },
 *         },
 *     });
 *     var iamForTransfer = new Aws.Iam.Role("iam_for_transfer", new()
 *     {
 *         NamePrefix = "iam_for_transfer_",
 *         AssumeRolePolicy = transferAssumeRole.Apply(getPolicyDocumentResult => getPolicyDocumentResult.Json),
 *         ManagedPolicyArns = new[]
 *         {
 *             "arn:aws:iam::aws:policy/service-role/AWSTransferLoggingAccess",
 *         },
 *     });
 *     var transferServer = new Aws.Transfer.Server("transfer", new()
 *     {
 *         EndpointType = "PUBLIC",
 *         LoggingRole = iamForTransfer.Arn,
 *         Protocols = new[]
 *         {
 *             "SFTP",
 *         },
 *         StructuredLogDestinations = new[]
 *         {
 *             transfer.Arn.Apply(arn => $"{arn}:*"),
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/cloudwatch"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/iam"
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/transfer"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		transfer, err := cloudwatch.NewLogGroup(ctx, "transfer", &cloudwatch.LogGroupArgs{
 * 			NamePrefix: pulumi.String("transfer_test_"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		transferAssumeRole, err := iam.GetPolicyDocument(ctx, &iam.GetPolicyDocumentArgs{
 * 			Statements: []iam.GetPolicyDocumentStatement{
 * 				{
 * 					Effect: pulumi.StringRef("Allow"),
 * 					Principals: []iam.GetPolicyDocumentStatementPrincipal{
 * 						{
 * 							Type: "Service",
 * 							Identifiers: []string{
 * 								"transfer.amazonaws.com",
 * 							},
 * 						},
 * 					},
 * 					Actions: []string{
 * 						"sts:AssumeRole",
 * 					},
 * 				},
 * 			},
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		iamForTransfer, err := iam.NewRole(ctx, "iam_for_transfer", &iam.RoleArgs{
 * 			NamePrefix:       pulumi.String("iam_for_transfer_"),
 * 			AssumeRolePolicy: pulumi.String(transferAssumeRole.Json),
 * 			ManagedPolicyArns: pulumi.StringArray{
 * 				pulumi.String("arn:aws:iam::aws:policy/service-role/AWSTransferLoggingAccess"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = transfer.NewServer(ctx, "transfer", &transfer.ServerArgs{
 * 			EndpointType: pulumi.String("PUBLIC"),
 * 			LoggingRole:  iamForTransfer.Arn,
 * 			Protocols: pulumi.StringArray{
 * 				pulumi.String("SFTP"),
 * 			},
 * 			StructuredLogDestinations: pulumi.StringArray{
 * 				transfer.Arn.ApplyT(func(arn string) (string, error) {
 * 					return fmt.Sprintf("%v:*", arn), nil
 * 				}).(pulumi.StringOutput),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.cloudwatch.LogGroup;
 * import com.pulumi.aws.cloudwatch.LogGroupArgs;
 * import com.pulumi.aws.iam.IamFunctions;
 * import com.pulumi.aws.iam.inputs.GetPolicyDocumentArgs;
 * import com.pulumi.aws.iam.Role;
 * import com.pulumi.aws.iam.RoleArgs;
 * import com.pulumi.aws.transfer.Server;
 * import com.pulumi.aws.transfer.ServerArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var transfer = new LogGroup("transfer", LogGroupArgs.builder()
 *             .namePrefix("transfer_test_")
 *             .build());
 *         final var transferAssumeRole = IamFunctions.getPolicyDocument(GetPolicyDocumentArgs.builder()
 *             .statements(GetPolicyDocumentStatementArgs.builder()
 *                 .effect("Allow")
 *                 .principals(GetPolicyDocumentStatementPrincipalArgs.builder()
 *                     .type("Service")
 *                     .identifiers("transfer.amazonaws.com")
 *                     .build())
 *                 .actions("sts:AssumeRole")
 *                 .build())
 *             .build());
 *         var iamForTransfer = new Role("iamForTransfer", RoleArgs.builder()
 *             .namePrefix("iam_for_transfer_")
 *             .assumeRolePolicy(transferAssumeRole.applyValue(getPolicyDocumentResult -> getPolicyDocumentResult.json()))
 *             .managedPolicyArns("arn:aws:iam::aws:policy/service-role/AWSTransferLoggingAccess")
 *             .build());
 *         var transferServer = new Server("transferServer", ServerArgs.builder()
 *             .endpointType("PUBLIC")
 *             .loggingRole(iamForTransfer.arn())
 *             .protocols("SFTP")
 *             .structuredLogDestinations(transfer.arn().applyValue(arn -> String.format("%s:*", arn)))
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   transfer:
 *     type: aws:cloudwatch:LogGroup
 *     properties:
 *       namePrefix: transfer_test_
 *   iamForTransfer:
 *     type: aws:iam:Role
 *     name: iam_for_transfer
 *     properties:
 *       namePrefix: iam_for_transfer_
 *       assumeRolePolicy: ${transferAssumeRole.json}
 *       managedPolicyArns:
 *         - arn:aws:iam::aws:policy/service-role/AWSTransferLoggingAccess
 *   transferServer:
 *     type: aws:transfer:Server
 *     name: transfer
 *     properties:
 *       endpointType: PUBLIC
 *       loggingRole: ${iamForTransfer.arn}
 *       protocols:
 *         - SFTP
 *       structuredLogDestinations:
 *         - ${transfer.arn}:*
 * variables:
 *   transferAssumeRole:
 *     fn::invoke:
 *       Function: aws:iam:getPolicyDocument
 *       Arguments:
 *         statements:
 *           - effect: Allow
 *             principals:
 *               - type: Service
 *                 identifiers:
 *                   - transfer.amazonaws.com
 *             actions:
 *               - sts:AssumeRole
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * In Terraform v1.5.0 and later, use an `import` Block to import Transfer Servers using the server `id`. For example:
 * Using `pulumi import`, import Transfer Servers using the server `id`. For example:
 * ```sh
 * $ pulumi import aws:transfer/server:Server example s-12345678
 * ```
 * Certain resource arguments, such as `host_key`, cannot be read via the API and imported into the provider. This provider will display a difference for these arguments the first run after import if declared in the provider configuration for an imported resource.
 * @property certificate The Amazon Resource Name (ARN) of the AWS Certificate Manager (ACM) certificate. This is required when `protocols` is set to `FTPS`
 * @property directoryId The directory service ID of the directory service you want to connect to with an `identity_provider_type` of `AWS_DIRECTORY_SERVICE`.
 * @property domain The domain of the storage system that is used for file transfers. Valid values are: `S3` and `EFS`. The default value is `S3`.
 * @property endpointDetails The virtual private cloud (VPC) endpoint settings that you want to configure for your SFTP server. See `endpoint_details` Block below for details.
 * @property endpointType The type of endpoint that you want your SFTP server connect to. If you connect to a `VPC` (or `VPC_ENDPOINT`), your SFTP server isn't accessible over the public internet. If you want to connect your SFTP server via public internet, set `PUBLIC`.  Defaults to `PUBLIC`.
 * @property forceDestroy A boolean that indicates all users associated with the server should be deleted so that the Server can be destroyed without error. The default value is `false`. This option only applies to servers configured with a `SERVICE_MANAGED` `identity_provider_type`.
 * @property function The ARN for a lambda function to use for the Identity provider.
 * @property hostKey RSA, ECDSA, or ED25519 private key (e.g., as generated by the `ssh-keygen -t rsa -b 2048 -N "" -m PEM -f my-new-server-key`, `ssh-keygen -t ecdsa -b 256 -N "" -m PEM -f my-new-server-key` or `ssh-keygen -t ed25519 -N "" -f my-new-server-key` commands).
 * @property identityProviderType The mode of authentication enabled for this service. The default value is `SERVICE_MANAGED`, which allows you to store and access SFTP user credentials within the service. `API_GATEWAY` indicates that user authentication requires a call to an API Gateway endpoint URL provided by you to integrate an identity provider of your choice. Using `AWS_DIRECTORY_SERVICE` will allow for authentication against AWS Managed Active Directory or Microsoft Active Directory in your on-premises environment, or in AWS using AD Connectors. Use the `AWS_LAMBDA` value to directly use a Lambda function as your identity provider. If you choose this value, you must specify the ARN for the lambda function in the `function` argument.
 * @property invocationRole Amazon Resource Name (ARN) of the IAM role used to authenticate the user account with an `identity_provider_type` of `API_GATEWAY`.
 * @property loggingRole Amazon Resource Name (ARN) of an IAM role that allows the service to write your SFTP users’ activity to your Amazon CloudWatch logs for monitoring and auditing purposes.
 * @property postAuthenticationLoginBanner Specify a string to display when users connect to a server. This string is displayed after the user authenticates. The SFTP protocol does not support post-authentication display banners.
 * @property preAuthenticationLoginBanner Specify a string to display when users connect to a server. This string is displayed before the user authenticates.
 * @property protocolDetails The protocol settings that are configured for your server. See `protocol_details` Block below for details.
 * @property protocols Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect to your server's endpoint. This defaults to `SFTP` . The available protocols are:
 * * `AS2`: File transfer over Applicability Statement 2
 * * `SFTP`: File transfer over SSH
 * * `FTPS`: File transfer with TLS encryption
 * * `FTP`: Unencrypted file transfer
 * @property s3StorageOptions Specifies whether or not performance for your Amazon S3 directories is optimized. This is disabled by default. See `s3_storage_options` Block below for details.
 * @property securityPolicyName Specifies the name of the security policy that is attached to the server. Default value is: `TransferSecurityPolicy-2018-11`. The available values are:
 * * `TransferSecurityPolicy-2018-11`
 * * `TransferSecurityPolicy-2020-06`
 * * `TransferSecurityPolicy-2022-03`
 * * `TransferSecurityPolicy-2023-05`
 * * `TransferSecurityPolicy-2024-01`
 * * `TransferSecurityPolicy-FIPS-2020-06`
 * * `TransferSecurityPolicy-FIPS-2023-05`
 * * `TransferSecurityPolicy-FIPS-2024-01`
 * * `TransferSecurityPolicy-FIPS-2024-05`
 * * `TransferSecurityPolicy-PQ-SSH-Experimental-2023-04`
 * * `TransferSecurityPolicy-PQ-SSH-FIPS-Experimental-2023-04`
 * * `TransferSecurityPolicy-Restricted-2018-11`
 * * `TransferSecurityPolicy-Restricted-2020-06`
 * See [Security policies for AWS Transfer Family servers](https://docs.aws.amazon.com/transfer/latest/userguide/security-policies.html) for details.
 * @property sftpAuthenticationMethods For SFTP-enabled servers, and for custom identity providers only. Valid values are `PASSWORD`, `PUBLIC_KEY`, `PUBLIC_KEY_OR_PASSWORD` and `PUBLIC_KEY_AND_PASSWORD`. Default value is: `PUBLIC_KEY_OR_PASSWORD`.
 * @property structuredLogDestinations A set of ARNs of destinations that will receive structured logs from the transfer server such as CloudWatch Log Group ARNs. If provided this enables the transfer server to emit structured logs to the specified locations.
 * @property tags A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
 * @property url URL of the service endpoint used to authenticate users with an `identity_provider_type` of `API_GATEWAY`.
 * @property workflowDetails Specifies the workflow details. See `workflow_details` Block below for details.
 */
public data class ServerArgs(
    public val certificate: Output<String>? = null,
    public val directoryId: Output<String>? = null,
    public val domain: Output<String>? = null,
    public val endpointDetails: Output<ServerEndpointDetailsArgs>? = null,
    public val endpointType: Output<String>? = null,
    public val forceDestroy: Output<Boolean>? = null,
    public val function: Output<String>? = null,
    public val hostKey: Output<String>? = null,
    public val identityProviderType: Output<String>? = null,
    public val invocationRole: Output<String>? = null,
    public val loggingRole: Output<String>? = null,
    public val postAuthenticationLoginBanner: Output<String>? = null,
    public val preAuthenticationLoginBanner: Output<String>? = null,
    public val protocolDetails: Output<ServerProtocolDetailsArgs>? = null,
    public val protocols: Output<List<String>>? = null,
    public val s3StorageOptions: Output<ServerS3StorageOptionsArgs>? = null,
    public val securityPolicyName: Output<String>? = null,
    public val sftpAuthenticationMethods: Output<String>? = null,
    public val structuredLogDestinations: Output<List<String>>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val url: Output<String>? = null,
    public val workflowDetails: Output<ServerWorkflowDetailsArgs>? = null,
) : ConvertibleToJava<com.pulumi.aws.transfer.ServerArgs> {
    override fun toJava(): com.pulumi.aws.transfer.ServerArgs =
        com.pulumi.aws.transfer.ServerArgs.builder()
            .certificate(certificate?.applyValue({ args0 -> args0 }))
            .directoryId(directoryId?.applyValue({ args0 -> args0 }))
            .domain(domain?.applyValue({ args0 -> args0 }))
            .endpointDetails(endpointDetails?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .endpointType(endpointType?.applyValue({ args0 -> args0 }))
            .forceDestroy(forceDestroy?.applyValue({ args0 -> args0 }))
            .function(function?.applyValue({ args0 -> args0 }))
            .hostKey(hostKey?.applyValue({ args0 -> args0 }))
            .identityProviderType(identityProviderType?.applyValue({ args0 -> args0 }))
            .invocationRole(invocationRole?.applyValue({ args0 -> args0 }))
            .loggingRole(loggingRole?.applyValue({ args0 -> args0 }))
            .postAuthenticationLoginBanner(postAuthenticationLoginBanner?.applyValue({ args0 -> args0 }))
            .preAuthenticationLoginBanner(preAuthenticationLoginBanner?.applyValue({ args0 -> args0 }))
            .protocolDetails(protocolDetails?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .protocols(protocols?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .s3StorageOptions(s3StorageOptions?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .securityPolicyName(securityPolicyName?.applyValue({ args0 -> args0 }))
            .sftpAuthenticationMethods(sftpAuthenticationMethods?.applyValue({ args0 -> args0 }))
            .structuredLogDestinations(
                structuredLogDestinations?.applyValue({ args0 ->
                    args0.map({ args0 ->
                        args0
                    })
                }),
            )
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .url(url?.applyValue({ args0 -> args0 }))
            .workflowDetails(
                workflowDetails?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            ).build()
}

/**
 * Builder for [ServerArgs].
 */
@PulumiTagMarker
public class ServerArgsBuilder internal constructor() {
    private var certificate: Output<String>? = null

    private var directoryId: Output<String>? = null

    private var domain: Output<String>? = null

    private var endpointDetails: Output<ServerEndpointDetailsArgs>? = null

    private var endpointType: Output<String>? = null

    private var forceDestroy: Output<Boolean>? = null

    private var function: Output<String>? = null

    private var hostKey: Output<String>? = null

    private var identityProviderType: Output<String>? = null

    private var invocationRole: Output<String>? = null

    private var loggingRole: Output<String>? = null

    private var postAuthenticationLoginBanner: Output<String>? = null

    private var preAuthenticationLoginBanner: Output<String>? = null

    private var protocolDetails: Output<ServerProtocolDetailsArgs>? = null

    private var protocols: Output<List<String>>? = null

    private var s3StorageOptions: Output<ServerS3StorageOptionsArgs>? = null

    private var securityPolicyName: Output<String>? = null

    private var sftpAuthenticationMethods: Output<String>? = null

    private var structuredLogDestinations: Output<List<String>>? = null

    private var tags: Output<Map<String, String>>? = null

    private var url: Output<String>? = null

    private var workflowDetails: Output<ServerWorkflowDetailsArgs>? = null

    /**
     * @param value The Amazon Resource Name (ARN) of the AWS Certificate Manager (ACM) certificate. This is required when `protocols` is set to `FTPS`
     */
    @JvmName("isncjnkmigawwsfn")
    public suspend fun certificate(`value`: Output<String>) {
        this.certificate = value
    }

    /**
     * @param value The directory service ID of the directory service you want to connect to with an `identity_provider_type` of `AWS_DIRECTORY_SERVICE`.
     */
    @JvmName("wqdbypymcxckhddk")
    public suspend fun directoryId(`value`: Output<String>) {
        this.directoryId = value
    }

    /**
     * @param value The domain of the storage system that is used for file transfers. Valid values are: `S3` and `EFS`. The default value is `S3`.
     */
    @JvmName("qurvrxswuvqddgie")
    public suspend fun domain(`value`: Output<String>) {
        this.domain = value
    }

    /**
     * @param value The virtual private cloud (VPC) endpoint settings that you want to configure for your SFTP server. See `endpoint_details` Block below for details.
     */
    @JvmName("vofrrqewdsteugxb")
    public suspend fun endpointDetails(`value`: Output<ServerEndpointDetailsArgs>) {
        this.endpointDetails = value
    }

    /**
     * @param value The type of endpoint that you want your SFTP server connect to. If you connect to a `VPC` (or `VPC_ENDPOINT`), your SFTP server isn't accessible over the public internet. If you want to connect your SFTP server via public internet, set `PUBLIC`.  Defaults to `PUBLIC`.
     */
    @JvmName("buyvwrlfpddqnyyn")
    public suspend fun endpointType(`value`: Output<String>) {
        this.endpointType = value
    }

    /**
     * @param value A boolean that indicates all users associated with the server should be deleted so that the Server can be destroyed without error. The default value is `false`. This option only applies to servers configured with a `SERVICE_MANAGED` `identity_provider_type`.
     */
    @JvmName("ghykbqtyitwqweky")
    public suspend fun forceDestroy(`value`: Output<Boolean>) {
        this.forceDestroy = value
    }

    /**
     * @param value The ARN for a lambda function to use for the Identity provider.
     */
    @JvmName("wygqnkkfhusrndpt")
    public suspend fun function(`value`: Output<String>) {
        this.function = value
    }

    /**
     * @param value RSA, ECDSA, or ED25519 private key (e.g., as generated by the `ssh-keygen -t rsa -b 2048 -N "" -m PEM -f my-new-server-key`, `ssh-keygen -t ecdsa -b 256 -N "" -m PEM -f my-new-server-key` or `ssh-keygen -t ed25519 -N "" -f my-new-server-key` commands).
     */
    @JvmName("hgklafgcvofymuxd")
    public suspend fun hostKey(`value`: Output<String>) {
        this.hostKey = value
    }

    /**
     * @param value The mode of authentication enabled for this service. The default value is `SERVICE_MANAGED`, which allows you to store and access SFTP user credentials within the service. `API_GATEWAY` indicates that user authentication requires a call to an API Gateway endpoint URL provided by you to integrate an identity provider of your choice. Using `AWS_DIRECTORY_SERVICE` will allow for authentication against AWS Managed Active Directory or Microsoft Active Directory in your on-premises environment, or in AWS using AD Connectors. Use the `AWS_LAMBDA` value to directly use a Lambda function as your identity provider. If you choose this value, you must specify the ARN for the lambda function in the `function` argument.
     */
    @JvmName("lejbfjrquhtqxqud")
    public suspend fun identityProviderType(`value`: Output<String>) {
        this.identityProviderType = value
    }

    /**
     * @param value Amazon Resource Name (ARN) of the IAM role used to authenticate the user account with an `identity_provider_type` of `API_GATEWAY`.
     */
    @JvmName("pcngpnjlkbfbqqry")
    public suspend fun invocationRole(`value`: Output<String>) {
        this.invocationRole = value
    }

    /**
     * @param value Amazon Resource Name (ARN) of an IAM role that allows the service to write your SFTP users’ activity to your Amazon CloudWatch logs for monitoring and auditing purposes.
     */
    @JvmName("jtnrbkhntwiwbabt")
    public suspend fun loggingRole(`value`: Output<String>) {
        this.loggingRole = value
    }

    /**
     * @param value Specify a string to display when users connect to a server. This string is displayed after the user authenticates. The SFTP protocol does not support post-authentication display banners.
     */
    @JvmName("hccjdfyoaxgsodhu")
    public suspend fun postAuthenticationLoginBanner(`value`: Output<String>) {
        this.postAuthenticationLoginBanner = value
    }

    /**
     * @param value Specify a string to display when users connect to a server. This string is displayed before the user authenticates.
     */
    @JvmName("mokvaebrlckdlhfc")
    public suspend fun preAuthenticationLoginBanner(`value`: Output<String>) {
        this.preAuthenticationLoginBanner = value
    }

    /**
     * @param value The protocol settings that are configured for your server. See `protocol_details` Block below for details.
     */
    @JvmName("rnaejshsfnqplofu")
    public suspend fun protocolDetails(`value`: Output<ServerProtocolDetailsArgs>) {
        this.protocolDetails = value
    }

    /**
     * @param value Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect to your server's endpoint. This defaults to `SFTP` . The available protocols are:
     * * `AS2`: File transfer over Applicability Statement 2
     * * `SFTP`: File transfer over SSH
     * * `FTPS`: File transfer with TLS encryption
     * * `FTP`: Unencrypted file transfer
     */
    @JvmName("wqnspqrwjdxnrsyu")
    public suspend fun protocols(`value`: Output<List<String>>) {
        this.protocols = value
    }

    @JvmName("qugdpgrahxtgkqdc")
    public suspend fun protocols(vararg values: Output<String>) {
        this.protocols = Output.all(values.asList())
    }

    /**
     * @param values Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect to your server's endpoint. This defaults to `SFTP` . The available protocols are:
     * * `AS2`: File transfer over Applicability Statement 2
     * * `SFTP`: File transfer over SSH
     * * `FTPS`: File transfer with TLS encryption
     * * `FTP`: Unencrypted file transfer
     */
    @JvmName("nklvywoulluekbnh")
    public suspend fun protocols(values: List<Output<String>>) {
        this.protocols = Output.all(values)
    }

    /**
     * @param value Specifies whether or not performance for your Amazon S3 directories is optimized. This is disabled by default. See `s3_storage_options` Block below for details.
     */
    @JvmName("ioyxbojuehhaggrk")
    public suspend fun s3StorageOptions(`value`: Output<ServerS3StorageOptionsArgs>) {
        this.s3StorageOptions = value
    }

    /**
     * @param value Specifies the name of the security policy that is attached to the server. Default value is: `TransferSecurityPolicy-2018-11`. The available values are:
     * * `TransferSecurityPolicy-2018-11`
     * * `TransferSecurityPolicy-2020-06`
     * * `TransferSecurityPolicy-2022-03`
     * * `TransferSecurityPolicy-2023-05`
     * * `TransferSecurityPolicy-2024-01`
     * * `TransferSecurityPolicy-FIPS-2020-06`
     * * `TransferSecurityPolicy-FIPS-2023-05`
     * * `TransferSecurityPolicy-FIPS-2024-01`
     * * `TransferSecurityPolicy-FIPS-2024-05`
     * * `TransferSecurityPolicy-PQ-SSH-Experimental-2023-04`
     * * `TransferSecurityPolicy-PQ-SSH-FIPS-Experimental-2023-04`
     * * `TransferSecurityPolicy-Restricted-2018-11`
     * * `TransferSecurityPolicy-Restricted-2020-06`
     * See [Security policies for AWS Transfer Family servers](https://docs.aws.amazon.com/transfer/latest/userguide/security-policies.html) for details.
     */
    @JvmName("pcebhafaakcubijx")
    public suspend fun securityPolicyName(`value`: Output<String>) {
        this.securityPolicyName = value
    }

    /**
     * @param value For SFTP-enabled servers, and for custom identity providers only. Valid values are `PASSWORD`, `PUBLIC_KEY`, `PUBLIC_KEY_OR_PASSWORD` and `PUBLIC_KEY_AND_PASSWORD`. Default value is: `PUBLIC_KEY_OR_PASSWORD`.
     */
    @JvmName("mkqkbhbbhdrlbscf")
    public suspend fun sftpAuthenticationMethods(`value`: Output<String>) {
        this.sftpAuthenticationMethods = value
    }

    /**
     * @param value A set of ARNs of destinations that will receive structured logs from the transfer server such as CloudWatch Log Group ARNs. If provided this enables the transfer server to emit structured logs to the specified locations.
     */
    @JvmName("cdvsyffkbigwqqft")
    public suspend fun structuredLogDestinations(`value`: Output<List<String>>) {
        this.structuredLogDestinations = value
    }

    @JvmName("qixtgoaktadovyag")
    public suspend fun structuredLogDestinations(vararg values: Output<String>) {
        this.structuredLogDestinations = Output.all(values.asList())
    }

    /**
     * @param values A set of ARNs of destinations that will receive structured logs from the transfer server such as CloudWatch Log Group ARNs. If provided this enables the transfer server to emit structured logs to the specified locations.
     */
    @JvmName("lrcyndkunkvejgcw")
    public suspend fun structuredLogDestinations(values: List<Output<String>>) {
        this.structuredLogDestinations = Output.all(values)
    }

    /**
     * @param value A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("jltknmwhmftxwlyg")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value URL of the service endpoint used to authenticate users with an `identity_provider_type` of `API_GATEWAY`.
     */
    @JvmName("rjxfnapttmonvtof")
    public suspend fun url(`value`: Output<String>) {
        this.url = value
    }

    /**
     * @param value Specifies the workflow details. See `workflow_details` Block below for details.
     */
    @JvmName("bsmpnonrexooovqy")
    public suspend fun workflowDetails(`value`: Output<ServerWorkflowDetailsArgs>) {
        this.workflowDetails = value
    }

    /**
     * @param value The Amazon Resource Name (ARN) of the AWS Certificate Manager (ACM) certificate. This is required when `protocols` is set to `FTPS`
     */
    @JvmName("imgfvsmphgeuyuno")
    public suspend fun certificate(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.certificate = mapped
    }

    /**
     * @param value The directory service ID of the directory service you want to connect to with an `identity_provider_type` of `AWS_DIRECTORY_SERVICE`.
     */
    @JvmName("rutttqvguucakimc")
    public suspend fun directoryId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.directoryId = mapped
    }

    /**
     * @param value The domain of the storage system that is used for file transfers. Valid values are: `S3` and `EFS`. The default value is `S3`.
     */
    @JvmName("tdhfndnopjhoduca")
    public suspend fun domain(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.domain = mapped
    }

    /**
     * @param value The virtual private cloud (VPC) endpoint settings that you want to configure for your SFTP server. See `endpoint_details` Block below for details.
     */
    @JvmName("mccfbohdtfbdcrqy")
    public suspend fun endpointDetails(`value`: ServerEndpointDetailsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.endpointDetails = mapped
    }

    /**
     * @param argument The virtual private cloud (VPC) endpoint settings that you want to configure for your SFTP server. See `endpoint_details` Block below for details.
     */
    @JvmName("xxwpbasrqmmqcucf")
    public suspend fun endpointDetails(argument: suspend ServerEndpointDetailsArgsBuilder.() -> Unit) {
        val toBeMapped = ServerEndpointDetailsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.endpointDetails = mapped
    }

    /**
     * @param value The type of endpoint that you want your SFTP server connect to. If you connect to a `VPC` (or `VPC_ENDPOINT`), your SFTP server isn't accessible over the public internet. If you want to connect your SFTP server via public internet, set `PUBLIC`.  Defaults to `PUBLIC`.
     */
    @JvmName("advotlgaycfkxags")
    public suspend fun endpointType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.endpointType = mapped
    }

    /**
     * @param value A boolean that indicates all users associated with the server should be deleted so that the Server can be destroyed without error. The default value is `false`. This option only applies to servers configured with a `SERVICE_MANAGED` `identity_provider_type`.
     */
    @JvmName("amufrjoagiyqxitr")
    public suspend fun forceDestroy(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.forceDestroy = mapped
    }

    /**
     * @param value The ARN for a lambda function to use for the Identity provider.
     */
    @JvmName("wseccjyyivrhnjcs")
    public suspend fun function(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.function = mapped
    }

    /**
     * @param value RSA, ECDSA, or ED25519 private key (e.g., as generated by the `ssh-keygen -t rsa -b 2048 -N "" -m PEM -f my-new-server-key`, `ssh-keygen -t ecdsa -b 256 -N "" -m PEM -f my-new-server-key` or `ssh-keygen -t ed25519 -N "" -f my-new-server-key` commands).
     */
    @JvmName("cxcyeeomlebkgquk")
    public suspend fun hostKey(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.hostKey = mapped
    }

    /**
     * @param value The mode of authentication enabled for this service. The default value is `SERVICE_MANAGED`, which allows you to store and access SFTP user credentials within the service. `API_GATEWAY` indicates that user authentication requires a call to an API Gateway endpoint URL provided by you to integrate an identity provider of your choice. Using `AWS_DIRECTORY_SERVICE` will allow for authentication against AWS Managed Active Directory or Microsoft Active Directory in your on-premises environment, or in AWS using AD Connectors. Use the `AWS_LAMBDA` value to directly use a Lambda function as your identity provider. If you choose this value, you must specify the ARN for the lambda function in the `function` argument.
     */
    @JvmName("bxdmjpryulsuhrnn")
    public suspend fun identityProviderType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.identityProviderType = mapped
    }

    /**
     * @param value Amazon Resource Name (ARN) of the IAM role used to authenticate the user account with an `identity_provider_type` of `API_GATEWAY`.
     */
    @JvmName("huukapefdsihkqrh")
    public suspend fun invocationRole(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.invocationRole = mapped
    }

    /**
     * @param value Amazon Resource Name (ARN) of an IAM role that allows the service to write your SFTP users’ activity to your Amazon CloudWatch logs for monitoring and auditing purposes.
     */
    @JvmName("fwflserwriyavpuv")
    public suspend fun loggingRole(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.loggingRole = mapped
    }

    /**
     * @param value Specify a string to display when users connect to a server. This string is displayed after the user authenticates. The SFTP protocol does not support post-authentication display banners.
     */
    @JvmName("ovmxrnubuuhfwhhr")
    public suspend fun postAuthenticationLoginBanner(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.postAuthenticationLoginBanner = mapped
    }

    /**
     * @param value Specify a string to display when users connect to a server. This string is displayed before the user authenticates.
     */
    @JvmName("wnqvvjhnvrsdovwg")
    public suspend fun preAuthenticationLoginBanner(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preAuthenticationLoginBanner = mapped
    }

    /**
     * @param value The protocol settings that are configured for your server. See `protocol_details` Block below for details.
     */
    @JvmName("utrasubkbammsljy")
    public suspend fun protocolDetails(`value`: ServerProtocolDetailsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.protocolDetails = mapped
    }

    /**
     * @param argument The protocol settings that are configured for your server. See `protocol_details` Block below for details.
     */
    @JvmName("chuomkoxomqbbwcp")
    public suspend fun protocolDetails(argument: suspend ServerProtocolDetailsArgsBuilder.() -> Unit) {
        val toBeMapped = ServerProtocolDetailsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.protocolDetails = mapped
    }

    /**
     * @param value Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect to your server's endpoint. This defaults to `SFTP` . The available protocols are:
     * * `AS2`: File transfer over Applicability Statement 2
     * * `SFTP`: File transfer over SSH
     * * `FTPS`: File transfer with TLS encryption
     * * `FTP`: Unencrypted file transfer
     */
    @JvmName("itkdwyadugtouilf")
    public suspend fun protocols(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.protocols = mapped
    }

    /**
     * @param values Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect to your server's endpoint. This defaults to `SFTP` . The available protocols are:
     * * `AS2`: File transfer over Applicability Statement 2
     * * `SFTP`: File transfer over SSH
     * * `FTPS`: File transfer with TLS encryption
     * * `FTP`: Unencrypted file transfer
     */
    @JvmName("xamibvwsawsabxqy")
    public suspend fun protocols(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.protocols = mapped
    }

    /**
     * @param value Specifies whether or not performance for your Amazon S3 directories is optimized. This is disabled by default. See `s3_storage_options` Block below for details.
     */
    @JvmName("aywqoukkdeucqkqy")
    public suspend fun s3StorageOptions(`value`: ServerS3StorageOptionsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.s3StorageOptions = mapped
    }

    /**
     * @param argument Specifies whether or not performance for your Amazon S3 directories is optimized. This is disabled by default. See `s3_storage_options` Block below for details.
     */
    @JvmName("sqlaokfwnahxtbsk")
    public suspend fun s3StorageOptions(argument: suspend ServerS3StorageOptionsArgsBuilder.() -> Unit) {
        val toBeMapped = ServerS3StorageOptionsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.s3StorageOptions = mapped
    }

    /**
     * @param value Specifies the name of the security policy that is attached to the server. Default value is: `TransferSecurityPolicy-2018-11`. The available values are:
     * * `TransferSecurityPolicy-2018-11`
     * * `TransferSecurityPolicy-2020-06`
     * * `TransferSecurityPolicy-2022-03`
     * * `TransferSecurityPolicy-2023-05`
     * * `TransferSecurityPolicy-2024-01`
     * * `TransferSecurityPolicy-FIPS-2020-06`
     * * `TransferSecurityPolicy-FIPS-2023-05`
     * * `TransferSecurityPolicy-FIPS-2024-01`
     * * `TransferSecurityPolicy-FIPS-2024-05`
     * * `TransferSecurityPolicy-PQ-SSH-Experimental-2023-04`
     * * `TransferSecurityPolicy-PQ-SSH-FIPS-Experimental-2023-04`
     * * `TransferSecurityPolicy-Restricted-2018-11`
     * * `TransferSecurityPolicy-Restricted-2020-06`
     * See [Security policies for AWS Transfer Family servers](https://docs.aws.amazon.com/transfer/latest/userguide/security-policies.html) for details.
     */
    @JvmName("hpkgwcvtadundakq")
    public suspend fun securityPolicyName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.securityPolicyName = mapped
    }

    /**
     * @param value For SFTP-enabled servers, and for custom identity providers only. Valid values are `PASSWORD`, `PUBLIC_KEY`, `PUBLIC_KEY_OR_PASSWORD` and `PUBLIC_KEY_AND_PASSWORD`. Default value is: `PUBLIC_KEY_OR_PASSWORD`.
     */
    @JvmName("hysfijramgsstkrt")
    public suspend fun sftpAuthenticationMethods(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sftpAuthenticationMethods = mapped
    }

    /**
     * @param value A set of ARNs of destinations that will receive structured logs from the transfer server such as CloudWatch Log Group ARNs. If provided this enables the transfer server to emit structured logs to the specified locations.
     */
    @JvmName("kkqrqghijlwiefqd")
    public suspend fun structuredLogDestinations(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.structuredLogDestinations = mapped
    }

    /**
     * @param values A set of ARNs of destinations that will receive structured logs from the transfer server such as CloudWatch Log Group ARNs. If provided this enables the transfer server to emit structured logs to the specified locations.
     */
    @JvmName("ytemlhdjtveskmlb")
    public suspend fun structuredLogDestinations(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.structuredLogDestinations = mapped
    }

    /**
     * @param value A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("hpacndfojsmalsdr")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values A map of tags to assign to the resource. If configured with a provider `default_tags` configuration block present, tags with matching keys will overwrite those defined at the provider-level.
     */
    @JvmName("kpevoibdnrpkqsee")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value URL of the service endpoint used to authenticate users with an `identity_provider_type` of `API_GATEWAY`.
     */
    @JvmName("ollrawhqdimlcxur")
    public suspend fun url(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.url = mapped
    }

    /**
     * @param value Specifies the workflow details. See `workflow_details` Block below for details.
     */
    @JvmName("qembjyktfowqamcb")
    public suspend fun workflowDetails(`value`: ServerWorkflowDetailsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.workflowDetails = mapped
    }

    /**
     * @param argument Specifies the workflow details. See `workflow_details` Block below for details.
     */
    @JvmName("svaeawbmskiwivwt")
    public suspend fun workflowDetails(argument: suspend ServerWorkflowDetailsArgsBuilder.() -> Unit) {
        val toBeMapped = ServerWorkflowDetailsArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.workflowDetails = mapped
    }

    internal fun build(): ServerArgs = ServerArgs(
        certificate = certificate,
        directoryId = directoryId,
        domain = domain,
        endpointDetails = endpointDetails,
        endpointType = endpointType,
        forceDestroy = forceDestroy,
        function = function,
        hostKey = hostKey,
        identityProviderType = identityProviderType,
        invocationRole = invocationRole,
        loggingRole = loggingRole,
        postAuthenticationLoginBanner = postAuthenticationLoginBanner,
        preAuthenticationLoginBanner = preAuthenticationLoginBanner,
        protocolDetails = protocolDetails,
        protocols = protocols,
        s3StorageOptions = s3StorageOptions,
        securityPolicyName = securityPolicyName,
        sftpAuthenticationMethods = sftpAuthenticationMethods,
        structuredLogDestinations = structuredLogDestinations,
        tags = tags,
        url = url,
        workflowDetails = workflowDetails,
    )
}
