@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.aws.transfer.kotlin

import com.pulumi.aws.transfer.TagArgs.builder
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * Manages an individual Transfer Family resource tag. This resource should only be used in cases where Transfer Family resources are created outside the provider (e.g., Servers without AWS Management Console) or the tag key has the `aws:` prefix.
 * > **NOTE:** This tagging resource should not be combined with the resource for managing the parent resource. For example, using `aws.transfer.Server` and `aws.transfer.Tag` to manage tags of the same server will cause a perpetual difference where the `aws.transfer.Server` resource will try to remove the tag being added by the `aws.transfer.Tag` resource.
 * > **NOTE:** This tagging resource does not use the provider `ignore_tags` configuration.
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as aws from "@pulumi/aws";
 * const example = new aws.transfer.Server("example", {identityProviderType: "SERVICE_MANAGED"});
 * const zoneId = new aws.transfer.Tag("zone_id", {
 *     resourceArn: example.arn,
 *     key: "aws:transfer:route53HostedZoneId",
 *     value: "/hostedzone/MyHostedZoneId",
 * });
 * const hostname = new aws.transfer.Tag("hostname", {
 *     resourceArn: example.arn,
 *     key: "aws:transfer:customHostname",
 *     value: "example.com",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_aws as aws
 * example = aws.transfer.Server("example", identity_provider_type="SERVICE_MANAGED")
 * zone_id = aws.transfer.Tag("zone_id",
 *     resource_arn=example.arn,
 *     key="aws:transfer:route53HostedZoneId",
 *     value="/hostedzone/MyHostedZoneId")
 * hostname = aws.transfer.Tag("hostname",
 *     resource_arn=example.arn,
 *     key="aws:transfer:customHostname",
 *     value="example.com")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using Aws = Pulumi.Aws;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new Aws.Transfer.Server("example", new()
 *     {
 *         IdentityProviderType = "SERVICE_MANAGED",
 *     });
 *     var zoneId = new Aws.Transfer.Tag("zone_id", new()
 *     {
 *         ResourceArn = example.Arn,
 *         Key = "aws:transfer:route53HostedZoneId",
 *         Value = "/hostedzone/MyHostedZoneId",
 *     });
 *     var hostname = new Aws.Transfer.Tag("hostname", new()
 *     {
 *         ResourceArn = example.Arn,
 *         Key = "aws:transfer:customHostname",
 *         Value = "example.com",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-aws/sdk/v6/go/aws/transfer"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := transfer.NewServer(ctx, "example", &transfer.ServerArgs{
 * 			IdentityProviderType: pulumi.String("SERVICE_MANAGED"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = transfer.NewTag(ctx, "zone_id", &transfer.TagArgs{
 * 			ResourceArn: example.Arn,
 * 			Key:         pulumi.String("aws:transfer:route53HostedZoneId"),
 * 			Value:       pulumi.String("/hostedzone/MyHostedZoneId"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = transfer.NewTag(ctx, "hostname", &transfer.TagArgs{
 * 			ResourceArn: example.Arn,
 * 			Key:         pulumi.String("aws:transfer:customHostname"),
 * 			Value:       pulumi.String("example.com"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.aws.transfer.Server;
 * import com.pulumi.aws.transfer.ServerArgs;
 * import com.pulumi.aws.transfer.Tag;
 * import com.pulumi.aws.transfer.TagArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Server("example", ServerArgs.builder()
 *             .identityProviderType("SERVICE_MANAGED")
 *             .build());
 *         var zoneId = new Tag("zoneId", TagArgs.builder()
 *             .resourceArn(example.arn())
 *             .key("aws:transfer:route53HostedZoneId")
 *             .value("/hostedzone/MyHostedZoneId")
 *             .build());
 *         var hostname = new Tag("hostname", TagArgs.builder()
 *             .resourceArn(example.arn())
 *             .key("aws:transfer:customHostname")
 *             .value("example.com")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: aws:transfer:Server
 *     properties:
 *       identityProviderType: SERVICE_MANAGED
 *   zoneId:
 *     type: aws:transfer:Tag
 *     name: zone_id
 *     properties:
 *       resourceArn: ${example.arn}
 *       key: aws:transfer:route53HostedZoneId
 *       value: /hostedzone/MyHostedZoneId
 *   hostname:
 *     type: aws:transfer:Tag
 *     properties:
 *       resourceArn: ${example.arn}
 *       key: aws:transfer:customHostname
 *       value: example.com
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Using `pulumi import`, import `aws_transfer_tag` using the Transfer Family resource identifier and key, separated by a comma (`,`). For example:
 * ```sh
 * $ pulumi import aws:transfer/tag:Tag example arn:aws:transfer:us-east-1:123456789012:server/s-1234567890abcdef0,Name
 * ```
 * @property key Tag name.
 * @property resourceArn Amazon Resource Name (ARN) of the Transfer Family resource to tag.
 * @property value Tag value.
 */
public data class TagArgs(
    public val key: Output<String>? = null,
    public val resourceArn: Output<String>? = null,
    public val `value`: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.aws.transfer.TagArgs> {
    override fun toJava(): com.pulumi.aws.transfer.TagArgs = com.pulumi.aws.transfer.TagArgs.builder()
        .key(key?.applyValue({ args0 -> args0 }))
        .resourceArn(resourceArn?.applyValue({ args0 -> args0 }))
        .`value`(`value`?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [TagArgs].
 */
@PulumiTagMarker
public class TagArgsBuilder internal constructor() {
    private var key: Output<String>? = null

    private var resourceArn: Output<String>? = null

    private var `value`: Output<String>? = null

    /**
     * @param value Tag name.
     */
    @JvmName("smfcssmwhoxmdgcv")
    public suspend fun key(`value`: Output<String>) {
        this.key = value
    }

    /**
     * @param value Amazon Resource Name (ARN) of the Transfer Family resource to tag.
     */
    @JvmName("hkjgbgyvqxavqudm")
    public suspend fun resourceArn(`value`: Output<String>) {
        this.resourceArn = value
    }

    /**
     * @param value Tag value.
     */
    @JvmName("eaytiynteparwrae")
    public suspend fun `value`(`value`: Output<String>) {
        this.`value` = value
    }

    /**
     * @param value Tag name.
     */
    @JvmName("didnuxhohvqhjgjp")
    public suspend fun key(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.key = mapped
    }

    /**
     * @param value Amazon Resource Name (ARN) of the Transfer Family resource to tag.
     */
    @JvmName("ljjybfralxbaxvde")
    public suspend fun resourceArn(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.resourceArn = mapped
    }

    /**
     * @param value Tag value.
     */
    @JvmName("bwsdkwupppvnvfnm")
    public suspend fun `value`(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.`value` = mapped
    }

    internal fun build(): TagArgs = TagArgs(
        key = key,
        resourceArn = resourceArn,
        `value` = `value`,
    )
}
