@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.kotlin.KotlinCustomResource
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.ResourceMapper
import com.pulumi.kotlin.options.CustomResourceOptions
import com.pulumi.kotlin.options.CustomResourceOptionsBuilder
import com.pulumi.resources.Resource
import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List

/**
 * Builder for [Application].
 */
@PulumiTagMarker
public class ApplicationResourceBuilder internal constructor() {
    public var name: String? = null

    public var args: ApplicationArgs = ApplicationArgs()

    public var opts: CustomResourceOptions = CustomResourceOptions()

    /**
     * @param name The _unique_ name of the resulting resource.
     */
    public fun name(`value`: String) {
        this.name = value
    }

    /**
     * @param block The arguments to use to populate this resource's properties.
     */
    public suspend fun args(block: suspend ApplicationArgsBuilder.() -> Unit) {
        val builder = ApplicationArgsBuilder()
        block(builder)
        this.args = builder.build()
    }

    /**
     * @param block A bag of options that control this resource's behavior.
     */
    public suspend fun opts(block: suspend CustomResourceOptionsBuilder.() -> Unit) {
        this.opts = com.pulumi.kotlin.options.CustomResourceOptions.opts(block)
    }

    internal fun build(): Application {
        val builtJavaResource = com.pulumi.gitlab.Application(
            this.name,
            this.args.toJava(),
            this.opts.toJava(),
        )
        return Application(builtJavaResource)
    }
}

/**
 * The `gitlab.Application` resource allows to manage the lifecycle of applications in gitlab.
 * > In order to use a user for a user to create an application, they must have admin privileges at the instance level.
 * To create an OIDC application, a scope of "openid".
 * **Upstream API**: [GitLab REST API docs](https://docs.gitlab.com/ee/api/applications.html)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * const oidc = new gitlab.Application("oidc", {
 *     confidential: true,
 *     scopes: ["openid"],
 *     name: "company_oidc",
 *     redirectUrl: "https://mycompany.com",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * oidc = gitlab.Application("oidc",
 *     confidential=True,
 *     scopes=["openid"],
 *     name="company_oidc",
 *     redirect_url="https://mycompany.com")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     var oidc = new GitLab.Application("oidc", new()
 *     {
 *         Confidential = true,
 *         Scopes = new[]
 *         {
 *             "openid",
 *         },
 *         Name = "company_oidc",
 *         RedirectUrl = "https://mycompany.com",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := gitlab.NewApplication(ctx, "oidc", &gitlab.ApplicationArgs{
 * 			Confidential: pulumi.Bool(true),
 * 			Scopes: pulumi.StringArray{
 * 				pulumi.String("openid"),
 * 			},
 * 			Name:        pulumi.String("company_oidc"),
 * 			RedirectUrl: pulumi.String("https://mycompany.com"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.Application;
 * import com.pulumi.gitlab.ApplicationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var oidc = new Application("oidc", ApplicationArgs.builder()
 *             .confidential(true)
 *             .scopes("openid")
 *             .name("company_oidc")
 *             .redirectUrl("https://mycompany.com")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   oidc:
 *     type: gitlab:Application
 *     properties:
 *       confidential: true
 *       scopes:
 *         - openid
 *       name: company_oidc
 *       redirectUrl: https://mycompany.com
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Gitlab applications can be imported with their id, e.g.
 * ```sh
 * $ pulumi import gitlab:index/application:Application example "1"
 * ```
 * NOTE: the secret and scopes cannot be imported
 */
public class Application internal constructor(
    override val javaResource: com.pulumi.gitlab.Application,
) : KotlinCustomResource(javaResource, ApplicationMapper) {
    /**
     * Internal name of the application.
     */
    public val applicationId: Output<String>
        get() = javaResource.applicationId().applyValue({ args0 -> args0 })

    /**
     * The application is used where the client secret can be kept confidential. Native mobile apps and Single Page Apps are considered non-confidential. Defaults to true if not supplied
     */
    public val confidential: Output<Boolean>
        get() = javaResource.confidential().applyValue({ args0 -> args0 })

    /**
     * Name of the application.
     */
    public val name: Output<String>
        get() = javaResource.name().applyValue({ args0 -> args0 })

    /**
     * The URL gitlab should send the user to after authentication.
     */
    public val redirectUrl: Output<String>
        get() = javaResource.redirectUrl().applyValue({ args0 -> args0 })

    /**
     * Scopes of the application. Use "openid" if you plan to use this as an oidc authentication application. Valid options are: `api`, `read_api`, `read_user`, `read_repository`, `write_repository`, `read_registry`, `write_registry`, `sudo`, `admin_mode`, `openid`, `profile`, `email`.
     * This is only populated when creating a new application. This attribute is not available for imported resources
     */
    public val scopes: Output<List<String>>
        get() = javaResource.scopes().applyValue({ args0 -> args0.map({ args0 -> args0 }) })

    /**
     * Application secret. Sensitive and must be kept secret. This is only populated when creating a new application. This attribute is not available for imported resources.
     */
    public val secret: Output<String>
        get() = javaResource.secret().applyValue({ args0 -> args0 })
}

public object ApplicationMapper : ResourceMapper<Application> {
    override fun supportsMappingOfType(javaResource: Resource): Boolean =
        com.pulumi.gitlab.Application::class == javaResource::class

    override fun map(javaResource: Resource): Application = Application(
        javaResource as
            com.pulumi.gitlab.Application,
    )
}

/**
 * @see [Application].
 * @param name The _unique_ name of the resulting resource.
 * @param block Builder for [Application].
 */
public suspend fun application(name: String, block: suspend ApplicationResourceBuilder.() -> Unit): Application {
    val builder = ApplicationResourceBuilder()
    builder.name(name)
    block(builder)
    return builder.build()
}

/**
 * @see [Application].
 * @param name The _unique_ name of the resulting resource.
 */
public fun application(name: String): Application {
    val builder = ApplicationResourceBuilder()
    builder.name(name)
    return builder.build()
}
