@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.gitlab.ApplicationArgs.builder
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlin.jvm.JvmName

/**
 * The `gitlab.Application` resource allows to manage the lifecycle of applications in gitlab.
 * > In order to use a user for a user to create an application, they must have admin privileges at the instance level.
 * To create an OIDC application, a scope of "openid".
 * **Upstream API**: [GitLab REST API docs](https://docs.gitlab.com/ee/api/applications.html)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * const oidc = new gitlab.Application("oidc", {
 *     confidential: true,
 *     scopes: ["openid"],
 *     name: "company_oidc",
 *     redirectUrl: "https://mycompany.com",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * oidc = gitlab.Application("oidc",
 *     confidential=True,
 *     scopes=["openid"],
 *     name="company_oidc",
 *     redirect_url="https://mycompany.com")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     var oidc = new GitLab.Application("oidc", new()
 *     {
 *         Confidential = true,
 *         Scopes = new[]
 *         {
 *             "openid",
 *         },
 *         Name = "company_oidc",
 *         RedirectUrl = "https://mycompany.com",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := gitlab.NewApplication(ctx, "oidc", &gitlab.ApplicationArgs{
 * 			Confidential: pulumi.Bool(true),
 * 			Scopes: pulumi.StringArray{
 * 				pulumi.String("openid"),
 * 			},
 * 			Name:        pulumi.String("company_oidc"),
 * 			RedirectUrl: pulumi.String("https://mycompany.com"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.Application;
 * import com.pulumi.gitlab.ApplicationArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var oidc = new Application("oidc", ApplicationArgs.builder()
 *             .confidential(true)
 *             .scopes("openid")
 *             .name("company_oidc")
 *             .redirectUrl("https://mycompany.com")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   oidc:
 *     type: gitlab:Application
 *     properties:
 *       confidential: true
 *       scopes:
 *         - openid
 *       name: company_oidc
 *       redirectUrl: https://mycompany.com
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Gitlab applications can be imported with their id, e.g.
 * ```sh
 * $ pulumi import gitlab:index/application:Application example "1"
 * ```
 * NOTE: the secret and scopes cannot be imported
 * @property confidential The application is used where the client secret can be kept confidential. Native mobile apps and Single Page Apps are considered non-confidential. Defaults to true if not supplied
 * @property name Name of the application.
 * @property redirectUrl The URL gitlab should send the user to after authentication.
 * @property scopes Scopes of the application. Use "openid" if you plan to use this as an oidc authentication application. Valid options are: `api`, `read_api`, `read_user`, `read_repository`, `write_repository`, `read_registry`, `write_registry`, `sudo`, `admin_mode`, `openid`, `profile`, `email`.
 * This is only populated when creating a new application. This attribute is not available for imported resources
 */
public data class ApplicationArgs(
    public val confidential: Output<Boolean>? = null,
    public val name: Output<String>? = null,
    public val redirectUrl: Output<String>? = null,
    public val scopes: Output<List<String>>? = null,
) : ConvertibleToJava<com.pulumi.gitlab.ApplicationArgs> {
    override fun toJava(): com.pulumi.gitlab.ApplicationArgs =
        com.pulumi.gitlab.ApplicationArgs.builder()
            .confidential(confidential?.applyValue({ args0 -> args0 }))
            .name(name?.applyValue({ args0 -> args0 }))
            .redirectUrl(redirectUrl?.applyValue({ args0 -> args0 }))
            .scopes(scopes?.applyValue({ args0 -> args0.map({ args0 -> args0 }) })).build()
}

/**
 * Builder for [ApplicationArgs].
 */
@PulumiTagMarker
public class ApplicationArgsBuilder internal constructor() {
    private var confidential: Output<Boolean>? = null

    private var name: Output<String>? = null

    private var redirectUrl: Output<String>? = null

    private var scopes: Output<List<String>>? = null

    /**
     * @param value The application is used where the client secret can be kept confidential. Native mobile apps and Single Page Apps are considered non-confidential. Defaults to true if not supplied
     */
    @JvmName("filphxdyiaocvuhg")
    public suspend fun confidential(`value`: Output<Boolean>) {
        this.confidential = value
    }

    /**
     * @param value Name of the application.
     */
    @JvmName("gsrvnrwxighbjhfn")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value The URL gitlab should send the user to after authentication.
     */
    @JvmName("oxwqfmpcpypbstdc")
    public suspend fun redirectUrl(`value`: Output<String>) {
        this.redirectUrl = value
    }

    /**
     * @param value Scopes of the application. Use "openid" if you plan to use this as an oidc authentication application. Valid options are: `api`, `read_api`, `read_user`, `read_repository`, `write_repository`, `read_registry`, `write_registry`, `sudo`, `admin_mode`, `openid`, `profile`, `email`.
     * This is only populated when creating a new application. This attribute is not available for imported resources
     */
    @JvmName("kkqwtaykwofjmvqq")
    public suspend fun scopes(`value`: Output<List<String>>) {
        this.scopes = value
    }

    @JvmName("khfajsmsreldlcid")
    public suspend fun scopes(vararg values: Output<String>) {
        this.scopes = Output.all(values.asList())
    }

    /**
     * @param values Scopes of the application. Use "openid" if you plan to use this as an oidc authentication application. Valid options are: `api`, `read_api`, `read_user`, `read_repository`, `write_repository`, `read_registry`, `write_registry`, `sudo`, `admin_mode`, `openid`, `profile`, `email`.
     * This is only populated when creating a new application. This attribute is not available for imported resources
     */
    @JvmName("drkwhtqfaonddiig")
    public suspend fun scopes(values: List<Output<String>>) {
        this.scopes = Output.all(values)
    }

    /**
     * @param value The application is used where the client secret can be kept confidential. Native mobile apps and Single Page Apps are considered non-confidential. Defaults to true if not supplied
     */
    @JvmName("njytwwopbiwgrmfb")
    public suspend fun confidential(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.confidential = mapped
    }

    /**
     * @param value Name of the application.
     */
    @JvmName("drfxcccupesjibqv")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value The URL gitlab should send the user to after authentication.
     */
    @JvmName("pbuqmdibwxaidldl")
    public suspend fun redirectUrl(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.redirectUrl = mapped
    }

    /**
     * @param value Scopes of the application. Use "openid" if you plan to use this as an oidc authentication application. Valid options are: `api`, `read_api`, `read_user`, `read_repository`, `write_repository`, `read_registry`, `write_registry`, `sudo`, `admin_mode`, `openid`, `profile`, `email`.
     * This is only populated when creating a new application. This attribute is not available for imported resources
     */
    @JvmName("qehknfirnrqilqks")
    public suspend fun scopes(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.scopes = mapped
    }

    /**
     * @param values Scopes of the application. Use "openid" if you plan to use this as an oidc authentication application. Valid options are: `api`, `read_api`, `read_user`, `read_repository`, `write_repository`, `read_registry`, `write_registry`, `sudo`, `admin_mode`, `openid`, `profile`, `email`.
     * This is only populated when creating a new application. This attribute is not available for imported resources
     */
    @JvmName("ikwwwgclnhuglmfq")
    public suspend fun scopes(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.scopes = mapped
    }

    internal fun build(): ApplicationArgs = ApplicationArgs(
        confidential = confidential,
        name = name,
        redirectUrl = redirectUrl,
        scopes = scopes,
    )
}
