@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.gitlab.ComplianceFrameworkArgs.builder
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * The `gitlab.ComplianceFramework` resource allows to manage the lifecycle of a compliance framework on top-level groups.
 * There can be only one `default` compliance framework. Of all the configured compliance frameworks marked as default, the last one applied will be the default compliance framework.
 * > This resource requires a GitLab Enterprise instance with a Premium license to create the compliance framework.
 * > This resource requires a GitLab Enterprise instance with an Ultimate license to specify a compliance pipeline configuration in the compliance framework.
 * **Upstream API**: [GitLab GraphQL API docs](https://docs.gitlab.com/ee/api/graphql/reference/#mutationcreatecomplianceframework)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * const sample = new gitlab.ComplianceFramework("sample", {
 *     namespacePath: "top-level-group",
 *     name: "HIPAA",
 *     description: "A HIPAA Compliance Framework",
 *     color: "#87BEEF",
 *     "default": false,
 *     pipelineConfigurationFullPath: ".hipaa.yml@top-level-group/compliance-frameworks",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * sample = gitlab.ComplianceFramework("sample",
 *     namespace_path="top-level-group",
 *     name="HIPAA",
 *     description="A HIPAA Compliance Framework",
 *     color="#87BEEF",
 *     default=False,
 *     pipeline_configuration_full_path=".hipaa.yml@top-level-group/compliance-frameworks")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     var sample = new GitLab.ComplianceFramework("sample", new()
 *     {
 *         NamespacePath = "top-level-group",
 *         Name = "HIPAA",
 *         Description = "A HIPAA Compliance Framework",
 *         Color = "#87BEEF",
 *         Default = false,
 *         PipelineConfigurationFullPath = ".hipaa.yml@top-level-group/compliance-frameworks",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := gitlab.NewComplianceFramework(ctx, "sample", &gitlab.ComplianceFrameworkArgs{
 * 			NamespacePath:                 pulumi.String("top-level-group"),
 * 			Name:                          pulumi.String("HIPAA"),
 * 			Description:                   pulumi.String("A HIPAA Compliance Framework"),
 * 			Color:                         pulumi.String("#87BEEF"),
 * 			Default:                       pulumi.Bool(false),
 * 			PipelineConfigurationFullPath: pulumi.String(".hipaa.yml@top-level-group/compliance-frameworks"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.ComplianceFramework;
 * import com.pulumi.gitlab.ComplianceFrameworkArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var sample = new ComplianceFramework("sample", ComplianceFrameworkArgs.builder()
 *             .namespacePath("top-level-group")
 *             .name("HIPAA")
 *             .description("A HIPAA Compliance Framework")
 *             .color("#87BEEF")
 *             .default_(false)
 *             .pipelineConfigurationFullPath(".hipaa.yml@top-level-group/compliance-frameworks")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   sample:
 *     type: gitlab:ComplianceFramework
 *     properties:
 *       namespacePath: top-level-group
 *       name: HIPAA
 *       description: A HIPAA Compliance Framework
 *       color: '#87BEEF'
 *       default: false
 *       pipelineConfigurationFullPath: .hipaa.yml@top-level-group/compliance-frameworks
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Gitlab compliance frameworks can be imported with a key composed of `<namespace_path>:<framework_id>`, e.g.
 * ```sh
 * $ pulumi import gitlab:index/complianceFramework:ComplianceFramework sample "top-level-group:gid://gitlab/ComplianceManagement::Framework/12345"
 * ```
 * @property color New color representation of the compliance framework in hex format. e.g. #FCA121.
 * @property default Set this compliance framework as the default framework for the group. Default: `false`
 * @property description Description for the compliance framework.
 * @property name Name for the compliance framework.
 * @property namespacePath Full path of the namespace to add the compliance framework to.
 * @property pipelineConfigurationFullPath Full path of the compliance pipeline configuration stored in a project repository, such as `.gitlab/.compliance-gitlab-ci.yml@compliance/hipaa`. Required format: `path/file.y[a]ml@group-name/project-name` **Note**: Ultimate license required.
 */
public data class ComplianceFrameworkArgs(
    public val color: Output<String>? = null,
    public val default: Output<Boolean>? = null,
    public val description: Output<String>? = null,
    public val name: Output<String>? = null,
    public val namespacePath: Output<String>? = null,
    public val pipelineConfigurationFullPath: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.gitlab.ComplianceFrameworkArgs> {
    override fun toJava(): com.pulumi.gitlab.ComplianceFrameworkArgs =
        com.pulumi.gitlab.ComplianceFrameworkArgs.builder()
            .color(color?.applyValue({ args0 -> args0 }))
            .default_(default?.applyValue({ args0 -> args0 }))
            .description(description?.applyValue({ args0 -> args0 }))
            .name(name?.applyValue({ args0 -> args0 }))
            .namespacePath(namespacePath?.applyValue({ args0 -> args0 }))
            .pipelineConfigurationFullPath(
                pipelineConfigurationFullPath?.applyValue({ args0 ->
                    args0
                }),
            ).build()
}

/**
 * Builder for [ComplianceFrameworkArgs].
 */
@PulumiTagMarker
public class ComplianceFrameworkArgsBuilder internal constructor() {
    private var color: Output<String>? = null

    private var default: Output<Boolean>? = null

    private var description: Output<String>? = null

    private var name: Output<String>? = null

    private var namespacePath: Output<String>? = null

    private var pipelineConfigurationFullPath: Output<String>? = null

    /**
     * @param value New color representation of the compliance framework in hex format. e.g. #FCA121.
     */
    @JvmName("kvhmlofvecpltdlr")
    public suspend fun color(`value`: Output<String>) {
        this.color = value
    }

    /**
     * @param value Set this compliance framework as the default framework for the group. Default: `false`
     */
    @JvmName("pvabfbnnvgiyppyq")
    public suspend fun default(`value`: Output<Boolean>) {
        this.default = value
    }

    /**
     * @param value Description for the compliance framework.
     */
    @JvmName("ttaaohtpqrpnksch")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value Name for the compliance framework.
     */
    @JvmName("ckqurwuklbufpoum")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Full path of the namespace to add the compliance framework to.
     */
    @JvmName("ofqbonjuhjvvysxl")
    public suspend fun namespacePath(`value`: Output<String>) {
        this.namespacePath = value
    }

    /**
     * @param value Full path of the compliance pipeline configuration stored in a project repository, such as `.gitlab/.compliance-gitlab-ci.yml@compliance/hipaa`. Required format: `path/file.y[a]ml@group-name/project-name` **Note**: Ultimate license required.
     */
    @JvmName("ihtgpramxpivbggb")
    public suspend fun pipelineConfigurationFullPath(`value`: Output<String>) {
        this.pipelineConfigurationFullPath = value
    }

    /**
     * @param value New color representation of the compliance framework in hex format. e.g. #FCA121.
     */
    @JvmName("jdxftkjwcwsqpkln")
    public suspend fun color(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.color = mapped
    }

    /**
     * @param value Set this compliance framework as the default framework for the group. Default: `false`
     */
    @JvmName("pkcsunibxytiamsy")
    public suspend fun default(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.default = mapped
    }

    /**
     * @param value Description for the compliance framework.
     */
    @JvmName("kikpgvjebppsogop")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value Name for the compliance framework.
     */
    @JvmName("olptmertstdouqkr")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Full path of the namespace to add the compliance framework to.
     */
    @JvmName("lxjymoyvnntuqewq")
    public suspend fun namespacePath(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.namespacePath = mapped
    }

    /**
     * @param value Full path of the compliance pipeline configuration stored in a project repository, such as `.gitlab/.compliance-gitlab-ci.yml@compliance/hipaa`. Required format: `path/file.y[a]ml@group-name/project-name` **Note**: Ultimate license required.
     */
    @JvmName("ddpplhreepxuwrky")
    public suspend fun pipelineConfigurationFullPath(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.pipelineConfigurationFullPath = mapped
    }

    internal fun build(): ComplianceFrameworkArgs = ComplianceFrameworkArgs(
        color = color,
        default = default,
        description = description,
        name = name,
        namespacePath = namespacePath,
        pipelineConfigurationFullPath = pipelineConfigurationFullPath,
    )
}
