@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.gitlab.DeployKeyArgs.builder
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * The `gitlab.DeployKey` resource allows to manage the lifecycle of a deploy key.
 * > To enable an already existing deploy key for another project use the `gitlab.DeployKeyEnable` resource.
 * **Upstream API**: [GitLab REST API docs](https://docs.gitlab.com/ee/api/deploy_keys.html)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * const example = new gitlab.DeployKey("example", {
 *     project: "example/deploying",
 *     title: "Example deploy key",
 *     key: "ssh-ed25519 AAAA...",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * example = gitlab.DeployKey("example",
 *     project="example/deploying",
 *     title="Example deploy key",
 *     key="ssh-ed25519 AAAA...")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new GitLab.DeployKey("example", new()
 *     {
 *         Project = "example/deploying",
 *         Title = "Example deploy key",
 *         Key = "ssh-ed25519 AAAA...",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := gitlab.NewDeployKey(ctx, "example", &gitlab.DeployKeyArgs{
 * 			Project: pulumi.String("example/deploying"),
 * 			Title:   pulumi.String("Example deploy key"),
 * 			Key:     pulumi.String("ssh-ed25519 AAAA..."),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.DeployKey;
 * import com.pulumi.gitlab.DeployKeyArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new DeployKey("example", DeployKeyArgs.builder()
 *             .project("example/deploying")
 *             .title("Example deploy key")
 *             .key("ssh-ed25519 AAAA...")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: gitlab:DeployKey
 *     properties:
 *       project: example/deploying
 *       title: Example deploy key
 *       key: ssh-ed25519 AAAA...
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * GitLab deploy keys can be imported using an id made up of `{project_id}:{deploy_key_id}`, e.g.
 * `project_id` can be whatever the [get single project api][get_single_project] takes for
 * its `:id` value, so for example:
 * ```sh
 * $ pulumi import gitlab:index/deployKey:DeployKey test 1:3
 * ```
 * ```sh
 * $ pulumi import gitlab:index/deployKey:DeployKey test richardc/example:3
 * ```
 * @property canPush Allow this deploy key to be used to push changes to the project. Defaults to `false`.
 * @property key The public ssh key body.
 * @property project The name or id of the project to add the deploy key to.
 * @property title A title to describe the deploy key with.
 */
public data class DeployKeyArgs(
    public val canPush: Output<Boolean>? = null,
    public val key: Output<String>? = null,
    public val project: Output<String>? = null,
    public val title: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.gitlab.DeployKeyArgs> {
    override fun toJava(): com.pulumi.gitlab.DeployKeyArgs = com.pulumi.gitlab.DeployKeyArgs.builder()
        .canPush(canPush?.applyValue({ args0 -> args0 }))
        .key(key?.applyValue({ args0 -> args0 }))
        .project(project?.applyValue({ args0 -> args0 }))
        .title(title?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [DeployKeyArgs].
 */
@PulumiTagMarker
public class DeployKeyArgsBuilder internal constructor() {
    private var canPush: Output<Boolean>? = null

    private var key: Output<String>? = null

    private var project: Output<String>? = null

    private var title: Output<String>? = null

    /**
     * @param value Allow this deploy key to be used to push changes to the project. Defaults to `false`.
     */
    @JvmName("rvwifodlnsdrqbse")
    public suspend fun canPush(`value`: Output<Boolean>) {
        this.canPush = value
    }

    /**
     * @param value The public ssh key body.
     */
    @JvmName("hwvywhceldmurnpi")
    public suspend fun key(`value`: Output<String>) {
        this.key = value
    }

    /**
     * @param value The name or id of the project to add the deploy key to.
     */
    @JvmName("itmvctlovrsysnvh")
    public suspend fun project(`value`: Output<String>) {
        this.project = value
    }

    /**
     * @param value A title to describe the deploy key with.
     */
    @JvmName("sdivnegjacswfqjc")
    public suspend fun title(`value`: Output<String>) {
        this.title = value
    }

    /**
     * @param value Allow this deploy key to be used to push changes to the project. Defaults to `false`.
     */
    @JvmName("qmgffaglncbmovbh")
    public suspend fun canPush(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.canPush = mapped
    }

    /**
     * @param value The public ssh key body.
     */
    @JvmName("pmamhohdlhokrask")
    public suspend fun key(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.key = mapped
    }

    /**
     * @param value The name or id of the project to add the deploy key to.
     */
    @JvmName("vvbgpeunsqjjarhq")
    public suspend fun project(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.project = mapped
    }

    /**
     * @param value A title to describe the deploy key with.
     */
    @JvmName("rdrurdavnodgvkhi")
    public suspend fun title(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.title = mapped
    }

    internal fun build(): DeployKeyArgs = DeployKeyArgs(
        canPush = canPush,
        key = key,
        project = project,
        title = title,
    )
}
