@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.kotlin.KotlinCustomResource
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.ResourceMapper
import com.pulumi.kotlin.options.CustomResourceOptions
import com.pulumi.kotlin.options.CustomResourceOptionsBuilder
import com.pulumi.resources.Resource
import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.Unit

/**
 * Builder for [DeployKeyEnable].
 */
@PulumiTagMarker
public class DeployKeyEnableResourceBuilder internal constructor() {
    public var name: String? = null

    public var args: DeployKeyEnableArgs = DeployKeyEnableArgs()

    public var opts: CustomResourceOptions = CustomResourceOptions()

    /**
     * @param name The _unique_ name of the resulting resource.
     */
    public fun name(`value`: String) {
        this.name = value
    }

    /**
     * @param block The arguments to use to populate this resource's properties.
     */
    public suspend fun args(block: suspend DeployKeyEnableArgsBuilder.() -> Unit) {
        val builder = DeployKeyEnableArgsBuilder()
        block(builder)
        this.args = builder.build()
    }

    /**
     * @param block A bag of options that control this resource's behavior.
     */
    public suspend fun opts(block: suspend CustomResourceOptionsBuilder.() -> Unit) {
        this.opts = com.pulumi.kotlin.options.CustomResourceOptions.opts(block)
    }

    internal fun build(): DeployKeyEnable {
        val builtJavaResource = com.pulumi.gitlab.DeployKeyEnable(
            this.name,
            this.args.toJava(),
            this.opts.toJava(),
        )
        return DeployKeyEnable(builtJavaResource)
    }
}

/**
 * The `gitlab.DeployKeyEnable` resource allows to enable an already existing deploy key (see `gitlab.DeployKey resource`) for a specific project.
 * **Upstream API**: [GitLab REST API docs](https://docs.gitlab.com/ee/api/deploy_keys.html#enable-a-deploy-key)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * // A repo to host the deployment key
 * const parent = new gitlab.Project("parent", {name: "parent_project"});
 * // A second repo to use the deployment key from the parent project
 * const foo = new gitlab.Project("foo", {name: "foo_project"});
 * // Upload a deployment key for the parent repo
 * const parentDeployKey = new gitlab.DeployKey("parent", {
 *     project: parent.id,
 *     title: "Example deploy key",
 *     key: "ssh-ed25519 AAAA...",
 * });
 * // Enable the deployment key on the second repo
 * const fooDeployKeyEnable = new gitlab.DeployKeyEnable("foo", {
 *     project: foo.id,
 *     keyId: parentDeployKey.deployKeyId,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * # A repo to host the deployment key
 * parent = gitlab.Project("parent", name="parent_project")
 * # A second repo to use the deployment key from the parent project
 * foo = gitlab.Project("foo", name="foo_project")
 * # Upload a deployment key for the parent repo
 * parent_deploy_key = gitlab.DeployKey("parent",
 *     project=parent.id,
 *     title="Example deploy key",
 *     key="ssh-ed25519 AAAA...")
 * # Enable the deployment key on the second repo
 * foo_deploy_key_enable = gitlab.DeployKeyEnable("foo",
 *     project=foo.id,
 *     key_id=parent_deploy_key.deploy_key_id)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     // A repo to host the deployment key
 *     var parent = new GitLab.Project("parent", new()
 *     {
 *         Name = "parent_project",
 *     });
 *     // A second repo to use the deployment key from the parent project
 *     var foo = new GitLab.Project("foo", new()
 *     {
 *         Name = "foo_project",
 *     });
 *     // Upload a deployment key for the parent repo
 *     var parentDeployKey = new GitLab.DeployKey("parent", new()
 *     {
 *         Project = parent.Id,
 *         Title = "Example deploy key",
 *         Key = "ssh-ed25519 AAAA...",
 *     });
 *     // Enable the deployment key on the second repo
 *     var fooDeployKeyEnable = new GitLab.DeployKeyEnable("foo", new()
 *     {
 *         Project = foo.Id,
 *         KeyId = parentDeployKey.DeployKeyId,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		// A repo to host the deployment key
 * 		parent, err := gitlab.NewProject(ctx, "parent", &gitlab.ProjectArgs{
 * 			Name: pulumi.String("parent_project"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// A second repo to use the deployment key from the parent project
 * 		foo, err := gitlab.NewProject(ctx, "foo", &gitlab.ProjectArgs{
 * 			Name: pulumi.String("foo_project"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Upload a deployment key for the parent repo
 * 		parentDeployKey, err := gitlab.NewDeployKey(ctx, "parent", &gitlab.DeployKeyArgs{
 * 			Project: parent.ID(),
 * 			Title:   pulumi.String("Example deploy key"),
 * 			Key:     pulumi.String("ssh-ed25519 AAAA..."),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Enable the deployment key on the second repo
 * 		_, err = gitlab.NewDeployKeyEnable(ctx, "foo", &gitlab.DeployKeyEnableArgs{
 * 			Project: foo.ID(),
 * 			KeyId:   parentDeployKey.DeployKeyId,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.Project;
 * import com.pulumi.gitlab.ProjectArgs;
 * import com.pulumi.gitlab.DeployKey;
 * import com.pulumi.gitlab.DeployKeyArgs;
 * import com.pulumi.gitlab.DeployKeyEnable;
 * import com.pulumi.gitlab.DeployKeyEnableArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         // A repo to host the deployment key
 *         var parent = new Project("parent", ProjectArgs.builder()
 *             .name("parent_project")
 *             .build());
 *         // A second repo to use the deployment key from the parent project
 *         var foo = new Project("foo", ProjectArgs.builder()
 *             .name("foo_project")
 *             .build());
 *         // Upload a deployment key for the parent repo
 *         var parentDeployKey = new DeployKey("parentDeployKey", DeployKeyArgs.builder()
 *             .project(parent.id())
 *             .title("Example deploy key")
 *             .key("ssh-ed25519 AAAA...")
 *             .build());
 *         // Enable the deployment key on the second repo
 *         var fooDeployKeyEnable = new DeployKeyEnable("fooDeployKeyEnable", DeployKeyEnableArgs.builder()
 *             .project(foo.id())
 *             .keyId(parentDeployKey.deployKeyId())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   # A repo to host the deployment key
 *   parent:
 *     type: gitlab:Project
 *     properties:
 *       name: parent_project
 *   # A second repo to use the deployment key from the parent project
 *   foo:
 *     type: gitlab:Project
 *     properties:
 *       name: foo_project
 *   # Upload a deployment key for the parent repo
 *   parentDeployKey:
 *     type: gitlab:DeployKey
 *     name: parent
 *     properties:
 *       project: ${parent.id}
 *       title: Example deploy key
 *       key: ssh-ed25519 AAAA...
 *   # Enable the deployment key on the second repo
 *   fooDeployKeyEnable:
 *     type: gitlab:DeployKeyEnable
 *     name: foo
 *     properties:
 *       project: ${foo.id}
 *       keyId: ${parentDeployKey.deployKeyId}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * GitLab enabled deploy keys can be imported using an id made up of `{project_id}:{deploy_key_id}`, e.g.
 * `project_id` can be whatever the [get single project api][get_single_project] takes for
 * its `:id` value, so for example:
 * ```sh
 * $ pulumi import gitlab:index/deployKeyEnable:DeployKeyEnable example 12345:67890
 * ```
 * ```sh
 * $ pulumi import gitlab:index/deployKeyEnable:DeployKeyEnable example richardc/example:67890
 * ```
 */
public class DeployKeyEnable internal constructor(
    override val javaResource: com.pulumi.gitlab.DeployKeyEnable,
) : KotlinCustomResource(javaResource, DeployKeyEnableMapper) {
    /**
     * Can deploy key push to the project's repository.
     */
    public val canPush: Output<Boolean>?
        get() = javaResource.canPush().applyValue({ args0 -> args0.map({ args0 -> args0 }).orElse(null) })

    /**
     * Deploy key.
     */
    public val key: Output<String>
        get() = javaResource.key().applyValue({ args0 -> args0 })

    /**
     * The Gitlab key id for the pre-existing deploy key
     */
    public val keyId: Output<String>
        get() = javaResource.keyId().applyValue({ args0 -> args0 })

    /**
     * The name or id of the project to add the deploy key to.
     */
    public val project: Output<String>
        get() = javaResource.project().applyValue({ args0 -> args0 })

    /**
     * Deploy key's title.
     */
    public val title: Output<String>
        get() = javaResource.title().applyValue({ args0 -> args0 })
}

public object DeployKeyEnableMapper : ResourceMapper<DeployKeyEnable> {
    override fun supportsMappingOfType(javaResource: Resource): Boolean =
        com.pulumi.gitlab.DeployKeyEnable::class == javaResource::class

    override fun map(javaResource: Resource): DeployKeyEnable = DeployKeyEnable(
        javaResource as
            com.pulumi.gitlab.DeployKeyEnable,
    )
}

/**
 * @see [DeployKeyEnable].
 * @param name The _unique_ name of the resulting resource.
 * @param block Builder for [DeployKeyEnable].
 */
public suspend fun deployKeyEnable(
    name: String,
    block: suspend DeployKeyEnableResourceBuilder.() -> Unit,
): DeployKeyEnable {
    val builder = DeployKeyEnableResourceBuilder()
    builder.name(name)
    block(builder)
    return builder.build()
}

/**
 * @see [DeployKeyEnable].
 * @param name The _unique_ name of the resulting resource.
 */
public fun deployKeyEnable(name: String): DeployKeyEnable {
    val builder = DeployKeyEnableResourceBuilder()
    builder.name(name)
    return builder.build()
}
