@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.gitlab.GroupArgs.builder
import com.pulumi.gitlab.kotlin.inputs.GroupDefaultBranchProtectionDefaultsArgs
import com.pulumi.gitlab.kotlin.inputs.GroupDefaultBranchProtectionDefaultsArgsBuilder
import com.pulumi.gitlab.kotlin.inputs.GroupPushRulesArgs
import com.pulumi.gitlab.kotlin.inputs.GroupPushRulesArgsBuilder
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.jvm.JvmName

/**
 * The `gitlab.Group` resource allows to manage the lifecycle of a group.
 * > On GitLab SaaS, you must use the GitLab UI to create groups without a parent group. You cannot use this provider nor the API to do this.
 * **Upstream API**: [GitLab REST API docs](https://docs.gitlab.com/ee/api/groups.html)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * const example = new gitlab.Group("example", {
 *     name: "example",
 *     path: "example",
 *     description: "An example group",
 * });
 * // Create a project in the example group
 * const exampleProject = new gitlab.Project("example", {
 *     name: "example",
 *     description: "An example project",
 *     namespaceId: example.id,
 * });
 * // Group with custom push rules
 * const example_two = new gitlab.Group("example-two", {
 *     name: "example-two",
 *     path: "example-two",
 *     description: "An example group with push rules",
 *     pushRules: {
 *         authorEmailRegex: "@example\\.com$",
 *         commitCommitterCheck: true,
 *         memberCheck: true,
 *         preventSecrets: true,
 *     },
 * });
 * // Group with custom default branch protection defaults
 * const example_three = new gitlab.Group("example-three", {
 *     name: "example-three",
 *     path: "example-three",
 *     description: "An example group with default branch protection defaults",
 *     defaultBranchProtectionDefaults: {
 *         allowedToPushes: ["developer"],
 *         allowForcePush: true,
 *         allowedToMerges: [
 *             "developer",
 *             "maintainer",
 *         ],
 *         developerCanInitialPush: true,
 *     },
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * example = gitlab.Group("example",
 *     name="example",
 *     path="example",
 *     description="An example group")
 * # Create a project in the example group
 * example_project = gitlab.Project("example",
 *     name="example",
 *     description="An example project",
 *     namespace_id=example.id)
 * # Group with custom push rules
 * example_two = gitlab.Group("example-two",
 *     name="example-two",
 *     path="example-two",
 *     description="An example group with push rules",
 *     push_rules={
 *         "author_email_regex": "@example\\.com$",
 *         "commit_committer_check": True,
 *         "member_check": True,
 *         "prevent_secrets": True,
 *     })
 * # Group with custom default branch protection defaults
 * example_three = gitlab.Group("example-three",
 *     name="example-three",
 *     path="example-three",
 *     description="An example group with default branch protection defaults",
 *     default_branch_protection_defaults={
 *         "allowed_to_pushes": ["developer"],
 *         "allow_force_push": True,
 *         "allowed_to_merges": [
 *             "developer",
 *             "maintainer",
 *         ],
 *         "developer_can_initial_push": True,
 *     })
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new GitLab.Group("example", new()
 *     {
 *         Name = "example",
 *         Path = "example",
 *         Description = "An example group",
 *     });
 *     // Create a project in the example group
 *     var exampleProject = new GitLab.Project("example", new()
 *     {
 *         Name = "example",
 *         Description = "An example project",
 *         NamespaceId = example.Id,
 *     });
 *     // Group with custom push rules
 *     var example_two = new GitLab.Group("example-two", new()
 *     {
 *         Name = "example-two",
 *         Path = "example-two",
 *         Description = "An example group with push rules",
 *         PushRules = new GitLab.Inputs.GroupPushRulesArgs
 *         {
 *             AuthorEmailRegex = "@example\\.com$",
 *             CommitCommitterCheck = true,
 *             MemberCheck = true,
 *             PreventSecrets = true,
 *         },
 *     });
 *     // Group with custom default branch protection defaults
 *     var example_three = new GitLab.Group("example-three", new()
 *     {
 *         Name = "example-three",
 *         Path = "example-three",
 *         Description = "An example group with default branch protection defaults",
 *         DefaultBranchProtectionDefaults = new GitLab.Inputs.GroupDefaultBranchProtectionDefaultsArgs
 *         {
 *             AllowedToPushes = new[]
 *             {
 *                 "developer",
 *             },
 *             AllowForcePush = true,
 *             AllowedToMerges = new[]
 *             {
 *                 "developer",
 *                 "maintainer",
 *             },
 *             DeveloperCanInitialPush = true,
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := gitlab.NewGroup(ctx, "example", &gitlab.GroupArgs{
 * 			Name:        pulumi.String("example"),
 * 			Path:        pulumi.String("example"),
 * 			Description: pulumi.String("An example group"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Create a project in the example group
 * 		_, err = gitlab.NewProject(ctx, "example", &gitlab.ProjectArgs{
 * 			Name:        pulumi.String("example"),
 * 			Description: pulumi.String("An example project"),
 * 			NamespaceId: example.ID(),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Group with custom push rules
 * 		_, err = gitlab.NewGroup(ctx, "example-two", &gitlab.GroupArgs{
 * 			Name:        pulumi.String("example-two"),
 * 			Path:        pulumi.String("example-two"),
 * 			Description: pulumi.String("An example group with push rules"),
 * 			PushRules: &gitlab.GroupPushRulesArgs{
 * 				AuthorEmailRegex:     pulumi.String("@example\\.com$"),
 * 				CommitCommitterCheck: pulumi.Bool(true),
 * 				MemberCheck:          pulumi.Bool(true),
 * 				PreventSecrets:       pulumi.Bool(true),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Group with custom default branch protection defaults
 * 		_, err = gitlab.NewGroup(ctx, "example-three", &gitlab.GroupArgs{
 * 			Name:        pulumi.String("example-three"),
 * 			Path:        pulumi.String("example-three"),
 * 			Description: pulumi.String("An example group with default branch protection defaults"),
 * 			DefaultBranchProtectionDefaults: &gitlab.GroupDefaultBranchProtectionDefaultsArgs{
 * 				AllowedToPushes: pulumi.StringArray{
 * 					pulumi.String("developer"),
 * 				},
 * 				AllowForcePush: pulumi.Bool(true),
 * 				AllowedToMerges: pulumi.StringArray{
 * 					pulumi.String("developer"),
 * 					pulumi.String("maintainer"),
 * 				},
 * 				DeveloperCanInitialPush: pulumi.Bool(true),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.Group;
 * import com.pulumi.gitlab.GroupArgs;
 * import com.pulumi.gitlab.Project;
 * import com.pulumi.gitlab.ProjectArgs;
 * import com.pulumi.gitlab.inputs.GroupPushRulesArgs;
 * import com.pulumi.gitlab.inputs.GroupDefaultBranchProtectionDefaultsArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new Group("example", GroupArgs.builder()
 *             .name("example")
 *             .path("example")
 *             .description("An example group")
 *             .build());
 *         // Create a project in the example group
 *         var exampleProject = new Project("exampleProject", ProjectArgs.builder()
 *             .name("example")
 *             .description("An example project")
 *             .namespaceId(example.id())
 *             .build());
 *         // Group with custom push rules
 *         var example_two = new Group("example-two", GroupArgs.builder()
 *             .name("example-two")
 *             .path("example-two")
 *             .description("An example group with push rules")
 *             .pushRules(GroupPushRulesArgs.builder()
 *                 .authorEmailRegex("@example\\.com$")
 *                 .commitCommitterCheck(true)
 *                 .memberCheck(true)
 *                 .preventSecrets(true)
 *                 .build())
 *             .build());
 *         // Group with custom default branch protection defaults
 *         var example_three = new Group("example-three", GroupArgs.builder()
 *             .name("example-three")
 *             .path("example-three")
 *             .description("An example group with default branch protection defaults")
 *             .defaultBranchProtectionDefaults(GroupDefaultBranchProtectionDefaultsArgs.builder()
 *                 .allowedToPushes("developer")
 *                 .allowForcePush(true)
 *                 .allowedToMerges(
 *                     "developer",
 *                     "maintainer")
 *                 .developerCanInitialPush(true)
 *                 .build())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: gitlab:Group
 *     properties:
 *       name: example
 *       path: example
 *       description: An example group
 *   # Create a project in the example group
 *   exampleProject:
 *     type: gitlab:Project
 *     name: example
 *     properties:
 *       name: example
 *       description: An example project
 *       namespaceId: ${example.id}
 *   # Group with custom push rules
 *   example-two:
 *     type: gitlab:Group
 *     properties:
 *       name: example-two
 *       path: example-two
 *       description: An example group with push rules
 *       pushRules:
 *         authorEmailRegex: '@example\.com$'
 *         commitCommitterCheck: true
 *         memberCheck: true
 *         preventSecrets: true
 *   # Group with custom default branch protection defaults
 *   example-three:
 *     type: gitlab:Group
 *     properties:
 *       name: example-three
 *       path: example-three
 *       description: An example group with default branch protection defaults
 *       defaultBranchProtectionDefaults:
 *         allowedToPushes:
 *           - developer
 *         allowForcePush: true
 *         allowedToMerges:
 *           - developer
 *           - maintainer
 *         developerCanInitialPush: true
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * ```sh
 * $ pulumi import gitlab:index/group:Group You can import a group state using `<resource> <id>`. The
 * ```
 * `id` can be whatever the [details of a group][details_of_a_group] api takes for
 * its `:id` value, so for example:
 * ```sh
 * $ pulumi import gitlab:index/group:Group example example
 * ```
 * @property autoDevopsEnabled Default to Auto DevOps pipeline for all projects within this group.
 * @property avatar A local path to the avatar image to upload. **Note**: not available for imported resources.
 * @property avatarHash The hash of the avatar image. Use `filesha256("path/to/avatar.png")` whenever possible. **Note**: this is used to trigger an update of the avatar. If it's not given, but an avatar is given, the avatar will be updated each time.
 * @property defaultBranchProtection See https://docs.gitlab.com/ee/api/groups.html#options-for-default*branch*protection. Valid values are: `0`, `1`, `2`, `3`, `4`.
 * @property defaultBranchProtectionDefaults The default branch protection defaults
 * @property description The group's description.
 * @property emailsEnabled Enable email notifications.
 * @property extraSharedRunnersMinutesLimit Can be set by administrators only. Additional CI/CD minutes for this group.
 * @property ipRestrictionRanges A list of IP addresses or subnet masks to restrict group access. Will be concatenated together into a comma separated string. Only allowed on top level groups.
 * @property lfsEnabled Enable/disable Large File Storage (LFS) for the projects in this group.
 * @property membershipLock Users cannot be added to projects in this group.
 * @property mentionsDisabled Disable the capability of a group from getting mentioned.
 * @property name The name of the group.
 * @property parentId Id of the parent group (creates a nested group).
 * @property path The path of the group.
 * @property permanentlyRemoveOnDelete Whether the group should be permanently removed during a `delete` operation. This only works with subgroups. Must be configured via an `apply` before the `destroy` is run.
 * @property preventForkingOutsideGroup Defaults to false. When enabled, users can not fork projects from this group to external namespaces.
 * @property projectCreationLevel Determine if developers can create projects in the group. Valid values are: `noone`, `maintainer`, `developer`
 * @property pushRules Push rules for the group.
 * @property requestAccessEnabled Allow users to request member access.
 * @property requireTwoFactorAuthentication Require all users in this group to setup Two-factor authentication.
 * @property shareWithGroupLock Prevent sharing a project with another group within this group.
 * @property sharedRunnersMinutesLimit Can be set by administrators only. Maximum number of monthly CI/CD minutes for this group. Can be nil (default; inherit system default), 0 (unlimited), or > 0.
 * @property sharedRunnersSetting Enable or disable shared runners for a group’s subgroups and projects. Valid values are: `enabled`, `disabled_and_overridable`, `disabled_and_unoverridable`, `disabled_with_override`.
 * @property subgroupCreationLevel Allowed to create subgroups. Valid values are: `owner`, `maintainer`.
 * @property twoFactorGracePeriod Defaults to 48. Time before Two-factor authentication is enforced (in hours).
 * @property visibilityLevel The group's visibility. Can be `private`, `internal`, or `public`. Valid values are: `private`, `internal`, `public`.
 * @property wikiAccessLevel The group's wiki access level. Only available on Premium and Ultimate plans. Valid values are `disabled`, `private`, `enabled`.
 */
public data class GroupArgs(
    public val autoDevopsEnabled: Output<Boolean>? = null,
    public val avatar: Output<String>? = null,
    public val avatarHash: Output<String>? = null,
    @Deprecated(
        message = """
  Deprecated in GitLab 17.0. Use default_branch_protection_defaults instead.
  """,
    )
    public val defaultBranchProtection: Output<Int>? = null,
    public val defaultBranchProtectionDefaults: Output<GroupDefaultBranchProtectionDefaultsArgs>? =
        null,
    public val description: Output<String>? = null,
    public val emailsEnabled: Output<Boolean>? = null,
    public val extraSharedRunnersMinutesLimit: Output<Int>? = null,
    public val ipRestrictionRanges: Output<List<String>>? = null,
    public val lfsEnabled: Output<Boolean>? = null,
    public val membershipLock: Output<Boolean>? = null,
    public val mentionsDisabled: Output<Boolean>? = null,
    public val name: Output<String>? = null,
    public val parentId: Output<Int>? = null,
    public val path: Output<String>? = null,
    public val permanentlyRemoveOnDelete: Output<Boolean>? = null,
    public val preventForkingOutsideGroup: Output<Boolean>? = null,
    public val projectCreationLevel: Output<String>? = null,
    public val pushRules: Output<GroupPushRulesArgs>? = null,
    public val requestAccessEnabled: Output<Boolean>? = null,
    public val requireTwoFactorAuthentication: Output<Boolean>? = null,
    public val shareWithGroupLock: Output<Boolean>? = null,
    public val sharedRunnersMinutesLimit: Output<Int>? = null,
    public val sharedRunnersSetting: Output<String>? = null,
    public val subgroupCreationLevel: Output<String>? = null,
    public val twoFactorGracePeriod: Output<Int>? = null,
    public val visibilityLevel: Output<String>? = null,
    public val wikiAccessLevel: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.gitlab.GroupArgs> {
    override fun toJava(): com.pulumi.gitlab.GroupArgs = com.pulumi.gitlab.GroupArgs.builder()
        .autoDevopsEnabled(autoDevopsEnabled?.applyValue({ args0 -> args0 }))
        .avatar(avatar?.applyValue({ args0 -> args0 }))
        .avatarHash(avatarHash?.applyValue({ args0 -> args0 }))
        .defaultBranchProtection(defaultBranchProtection?.applyValue({ args0 -> args0 }))
        .defaultBranchProtectionDefaults(
            defaultBranchProtectionDefaults?.applyValue({ args0 ->
                args0.let({ args0 -> args0.toJava() })
            }),
        )
        .description(description?.applyValue({ args0 -> args0 }))
        .emailsEnabled(emailsEnabled?.applyValue({ args0 -> args0 }))
        .extraSharedRunnersMinutesLimit(extraSharedRunnersMinutesLimit?.applyValue({ args0 -> args0 }))
        .ipRestrictionRanges(ipRestrictionRanges?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
        .lfsEnabled(lfsEnabled?.applyValue({ args0 -> args0 }))
        .membershipLock(membershipLock?.applyValue({ args0 -> args0 }))
        .mentionsDisabled(mentionsDisabled?.applyValue({ args0 -> args0 }))
        .name(name?.applyValue({ args0 -> args0 }))
        .parentId(parentId?.applyValue({ args0 -> args0 }))
        .path(path?.applyValue({ args0 -> args0 }))
        .permanentlyRemoveOnDelete(permanentlyRemoveOnDelete?.applyValue({ args0 -> args0 }))
        .preventForkingOutsideGroup(preventForkingOutsideGroup?.applyValue({ args0 -> args0 }))
        .projectCreationLevel(projectCreationLevel?.applyValue({ args0 -> args0 }))
        .pushRules(pushRules?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
        .requestAccessEnabled(requestAccessEnabled?.applyValue({ args0 -> args0 }))
        .requireTwoFactorAuthentication(requireTwoFactorAuthentication?.applyValue({ args0 -> args0 }))
        .shareWithGroupLock(shareWithGroupLock?.applyValue({ args0 -> args0 }))
        .sharedRunnersMinutesLimit(sharedRunnersMinutesLimit?.applyValue({ args0 -> args0 }))
        .sharedRunnersSetting(sharedRunnersSetting?.applyValue({ args0 -> args0 }))
        .subgroupCreationLevel(subgroupCreationLevel?.applyValue({ args0 -> args0 }))
        .twoFactorGracePeriod(twoFactorGracePeriod?.applyValue({ args0 -> args0 }))
        .visibilityLevel(visibilityLevel?.applyValue({ args0 -> args0 }))
        .wikiAccessLevel(wikiAccessLevel?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [GroupArgs].
 */
@PulumiTagMarker
public class GroupArgsBuilder internal constructor() {
    private var autoDevopsEnabled: Output<Boolean>? = null

    private var avatar: Output<String>? = null

    private var avatarHash: Output<String>? = null

    private var defaultBranchProtection: Output<Int>? = null

    private var defaultBranchProtectionDefaults: Output<GroupDefaultBranchProtectionDefaultsArgs>? =
        null

    private var description: Output<String>? = null

    private var emailsEnabled: Output<Boolean>? = null

    private var extraSharedRunnersMinutesLimit: Output<Int>? = null

    private var ipRestrictionRanges: Output<List<String>>? = null

    private var lfsEnabled: Output<Boolean>? = null

    private var membershipLock: Output<Boolean>? = null

    private var mentionsDisabled: Output<Boolean>? = null

    private var name: Output<String>? = null

    private var parentId: Output<Int>? = null

    private var path: Output<String>? = null

    private var permanentlyRemoveOnDelete: Output<Boolean>? = null

    private var preventForkingOutsideGroup: Output<Boolean>? = null

    private var projectCreationLevel: Output<String>? = null

    private var pushRules: Output<GroupPushRulesArgs>? = null

    private var requestAccessEnabled: Output<Boolean>? = null

    private var requireTwoFactorAuthentication: Output<Boolean>? = null

    private var shareWithGroupLock: Output<Boolean>? = null

    private var sharedRunnersMinutesLimit: Output<Int>? = null

    private var sharedRunnersSetting: Output<String>? = null

    private var subgroupCreationLevel: Output<String>? = null

    private var twoFactorGracePeriod: Output<Int>? = null

    private var visibilityLevel: Output<String>? = null

    private var wikiAccessLevel: Output<String>? = null

    /**
     * @param value Default to Auto DevOps pipeline for all projects within this group.
     */
    @JvmName("tslytvobqswmfjtj")
    public suspend fun autoDevopsEnabled(`value`: Output<Boolean>) {
        this.autoDevopsEnabled = value
    }

    /**
     * @param value A local path to the avatar image to upload. **Note**: not available for imported resources.
     */
    @JvmName("lprfsmjksdsmrvwu")
    public suspend fun avatar(`value`: Output<String>) {
        this.avatar = value
    }

    /**
     * @param value The hash of the avatar image. Use `filesha256("path/to/avatar.png")` whenever possible. **Note**: this is used to trigger an update of the avatar. If it's not given, but an avatar is given, the avatar will be updated each time.
     */
    @JvmName("yqygqbnpbrivedao")
    public suspend fun avatarHash(`value`: Output<String>) {
        this.avatarHash = value
    }

    /**
     * @param value See https://docs.gitlab.com/ee/api/groups.html#options-for-default*branch*protection. Valid values are: `0`, `1`, `2`, `3`, `4`.
     */
    @Deprecated(
        message = """
  Deprecated in GitLab 17.0. Use default_branch_protection_defaults instead.
  """,
    )
    @JvmName("rigewjbwpgjxokvo")
    public suspend fun defaultBranchProtection(`value`: Output<Int>) {
        this.defaultBranchProtection = value
    }

    /**
     * @param value The default branch protection defaults
     */
    @JvmName("egexwjyahxdggklv")
    public suspend fun defaultBranchProtectionDefaults(`value`: Output<GroupDefaultBranchProtectionDefaultsArgs>) {
        this.defaultBranchProtectionDefaults = value
    }

    /**
     * @param value The group's description.
     */
    @JvmName("bkdjswdtwciwotcc")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value Enable email notifications.
     */
    @JvmName("fkfmblktnigdksoe")
    public suspend fun emailsEnabled(`value`: Output<Boolean>) {
        this.emailsEnabled = value
    }

    /**
     * @param value Can be set by administrators only. Additional CI/CD minutes for this group.
     */
    @JvmName("jaoqeqgijhpibxsu")
    public suspend fun extraSharedRunnersMinutesLimit(`value`: Output<Int>) {
        this.extraSharedRunnersMinutesLimit = value
    }

    /**
     * @param value A list of IP addresses or subnet masks to restrict group access. Will be concatenated together into a comma separated string. Only allowed on top level groups.
     */
    @JvmName("rghvjybcdmofwrvj")
    public suspend fun ipRestrictionRanges(`value`: Output<List<String>>) {
        this.ipRestrictionRanges = value
    }

    @JvmName("dvcyylcpkmgcrtot")
    public suspend fun ipRestrictionRanges(vararg values: Output<String>) {
        this.ipRestrictionRanges = Output.all(values.asList())
    }

    /**
     * @param values A list of IP addresses or subnet masks to restrict group access. Will be concatenated together into a comma separated string. Only allowed on top level groups.
     */
    @JvmName("vnvrdnuuguokwrhs")
    public suspend fun ipRestrictionRanges(values: List<Output<String>>) {
        this.ipRestrictionRanges = Output.all(values)
    }

    /**
     * @param value Enable/disable Large File Storage (LFS) for the projects in this group.
     */
    @JvmName("fumplttnagnoocqu")
    public suspend fun lfsEnabled(`value`: Output<Boolean>) {
        this.lfsEnabled = value
    }

    /**
     * @param value Users cannot be added to projects in this group.
     */
    @JvmName("dwxdidmmtegrijnq")
    public suspend fun membershipLock(`value`: Output<Boolean>) {
        this.membershipLock = value
    }

    /**
     * @param value Disable the capability of a group from getting mentioned.
     */
    @JvmName("onnquhrxvvrbpodb")
    public suspend fun mentionsDisabled(`value`: Output<Boolean>) {
        this.mentionsDisabled = value
    }

    /**
     * @param value The name of the group.
     */
    @JvmName("nuaahiclhslbpyvs")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value Id of the parent group (creates a nested group).
     */
    @JvmName("aimhsukcodfwihyj")
    public suspend fun parentId(`value`: Output<Int>) {
        this.parentId = value
    }

    /**
     * @param value The path of the group.
     */
    @JvmName("teujqipdxmxyjdns")
    public suspend fun path(`value`: Output<String>) {
        this.path = value
    }

    /**
     * @param value Whether the group should be permanently removed during a `delete` operation. This only works with subgroups. Must be configured via an `apply` before the `destroy` is run.
     */
    @JvmName("vxaebncqgntjwjmb")
    public suspend fun permanentlyRemoveOnDelete(`value`: Output<Boolean>) {
        this.permanentlyRemoveOnDelete = value
    }

    /**
     * @param value Defaults to false. When enabled, users can not fork projects from this group to external namespaces.
     */
    @JvmName("suspbpfsbkekhblh")
    public suspend fun preventForkingOutsideGroup(`value`: Output<Boolean>) {
        this.preventForkingOutsideGroup = value
    }

    /**
     * @param value Determine if developers can create projects in the group. Valid values are: `noone`, `maintainer`, `developer`
     */
    @JvmName("mpbutmfohaanskyl")
    public suspend fun projectCreationLevel(`value`: Output<String>) {
        this.projectCreationLevel = value
    }

    /**
     * @param value Push rules for the group.
     */
    @JvmName("fjioygykbbcouxox")
    public suspend fun pushRules(`value`: Output<GroupPushRulesArgs>) {
        this.pushRules = value
    }

    /**
     * @param value Allow users to request member access.
     */
    @JvmName("kyidxypxeerjmynd")
    public suspend fun requestAccessEnabled(`value`: Output<Boolean>) {
        this.requestAccessEnabled = value
    }

    /**
     * @param value Require all users in this group to setup Two-factor authentication.
     */
    @JvmName("ecncxcdeoprvmvvw")
    public suspend fun requireTwoFactorAuthentication(`value`: Output<Boolean>) {
        this.requireTwoFactorAuthentication = value
    }

    /**
     * @param value Prevent sharing a project with another group within this group.
     */
    @JvmName("pxriujhjmtpjwcuu")
    public suspend fun shareWithGroupLock(`value`: Output<Boolean>) {
        this.shareWithGroupLock = value
    }

    /**
     * @param value Can be set by administrators only. Maximum number of monthly CI/CD minutes for this group. Can be nil (default; inherit system default), 0 (unlimited), or > 0.
     */
    @JvmName("gertmvraslwsrghr")
    public suspend fun sharedRunnersMinutesLimit(`value`: Output<Int>) {
        this.sharedRunnersMinutesLimit = value
    }

    /**
     * @param value Enable or disable shared runners for a group’s subgroups and projects. Valid values are: `enabled`, `disabled_and_overridable`, `disabled_and_unoverridable`, `disabled_with_override`.
     */
    @JvmName("pnyvrsvcragcbteg")
    public suspend fun sharedRunnersSetting(`value`: Output<String>) {
        this.sharedRunnersSetting = value
    }

    /**
     * @param value Allowed to create subgroups. Valid values are: `owner`, `maintainer`.
     */
    @JvmName("btaddtlnyoapesfm")
    public suspend fun subgroupCreationLevel(`value`: Output<String>) {
        this.subgroupCreationLevel = value
    }

    /**
     * @param value Defaults to 48. Time before Two-factor authentication is enforced (in hours).
     */
    @JvmName("siqjjtweuhyeltpu")
    public suspend fun twoFactorGracePeriod(`value`: Output<Int>) {
        this.twoFactorGracePeriod = value
    }

    /**
     * @param value The group's visibility. Can be `private`, `internal`, or `public`. Valid values are: `private`, `internal`, `public`.
     */
    @JvmName("efsgqlsqgaodstcl")
    public suspend fun visibilityLevel(`value`: Output<String>) {
        this.visibilityLevel = value
    }

    /**
     * @param value The group's wiki access level. Only available on Premium and Ultimate plans. Valid values are `disabled`, `private`, `enabled`.
     */
    @JvmName("haebtleacuofibqn")
    public suspend fun wikiAccessLevel(`value`: Output<String>) {
        this.wikiAccessLevel = value
    }

    /**
     * @param value Default to Auto DevOps pipeline for all projects within this group.
     */
    @JvmName("uwjjtutpvkonvmli")
    public suspend fun autoDevopsEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoDevopsEnabled = mapped
    }

    /**
     * @param value A local path to the avatar image to upload. **Note**: not available for imported resources.
     */
    @JvmName("ytjyrvtckdlfgpgw")
    public suspend fun avatar(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.avatar = mapped
    }

    /**
     * @param value The hash of the avatar image. Use `filesha256("path/to/avatar.png")` whenever possible. **Note**: this is used to trigger an update of the avatar. If it's not given, but an avatar is given, the avatar will be updated each time.
     */
    @JvmName("xkxfnbwvgjxgcvvr")
    public suspend fun avatarHash(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.avatarHash = mapped
    }

    /**
     * @param value See https://docs.gitlab.com/ee/api/groups.html#options-for-default*branch*protection. Valid values are: `0`, `1`, `2`, `3`, `4`.
     */
    @Deprecated(
        message = """
  Deprecated in GitLab 17.0. Use default_branch_protection_defaults instead.
  """,
    )
    @JvmName("qtiuxylrneuabcwn")
    public suspend fun defaultBranchProtection(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.defaultBranchProtection = mapped
    }

    /**
     * @param value The default branch protection defaults
     */
    @JvmName("hkqgymtknikrmtym")
    public suspend fun defaultBranchProtectionDefaults(`value`: GroupDefaultBranchProtectionDefaultsArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.defaultBranchProtectionDefaults = mapped
    }

    /**
     * @param argument The default branch protection defaults
     */
    @JvmName("slnpbgfyofchogss")
    public suspend fun defaultBranchProtectionDefaults(argument: suspend GroupDefaultBranchProtectionDefaultsArgsBuilder.() -> Unit) {
        val toBeMapped = GroupDefaultBranchProtectionDefaultsArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.defaultBranchProtectionDefaults = mapped
    }

    /**
     * @param value The group's description.
     */
    @JvmName("fnpryjtrfmhbvjuu")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value Enable email notifications.
     */
    @JvmName("wqdrypewoaleknmo")
    public suspend fun emailsEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.emailsEnabled = mapped
    }

    /**
     * @param value Can be set by administrators only. Additional CI/CD minutes for this group.
     */
    @JvmName("wopvurmkgarsyxyv")
    public suspend fun extraSharedRunnersMinutesLimit(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.extraSharedRunnersMinutesLimit = mapped
    }

    /**
     * @param value A list of IP addresses or subnet masks to restrict group access. Will be concatenated together into a comma separated string. Only allowed on top level groups.
     */
    @JvmName("yatfnqfowuxrwmbk")
    public suspend fun ipRestrictionRanges(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ipRestrictionRanges = mapped
    }

    /**
     * @param values A list of IP addresses or subnet masks to restrict group access. Will be concatenated together into a comma separated string. Only allowed on top level groups.
     */
    @JvmName("unulinhpkxpllqvv")
    public suspend fun ipRestrictionRanges(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.ipRestrictionRanges = mapped
    }

    /**
     * @param value Enable/disable Large File Storage (LFS) for the projects in this group.
     */
    @JvmName("ncyslskebmxajbjn")
    public suspend fun lfsEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.lfsEnabled = mapped
    }

    /**
     * @param value Users cannot be added to projects in this group.
     */
    @JvmName("ktynhgvkyfwhexqq")
    public suspend fun membershipLock(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.membershipLock = mapped
    }

    /**
     * @param value Disable the capability of a group from getting mentioned.
     */
    @JvmName("lorxjihsyydumawi")
    public suspend fun mentionsDisabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.mentionsDisabled = mapped
    }

    /**
     * @param value The name of the group.
     */
    @JvmName("dnpebuoocpusjxvy")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value Id of the parent group (creates a nested group).
     */
    @JvmName("ahyytnyqasknrqfc")
    public suspend fun parentId(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.parentId = mapped
    }

    /**
     * @param value The path of the group.
     */
    @JvmName("xcmkovhpeaseoird")
    public suspend fun path(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.path = mapped
    }

    /**
     * @param value Whether the group should be permanently removed during a `delete` operation. This only works with subgroups. Must be configured via an `apply` before the `destroy` is run.
     */
    @JvmName("ojkyaaxdijgetulp")
    public suspend fun permanentlyRemoveOnDelete(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.permanentlyRemoveOnDelete = mapped
    }

    /**
     * @param value Defaults to false. When enabled, users can not fork projects from this group to external namespaces.
     */
    @JvmName("kbxjofwoflkjnftu")
    public suspend fun preventForkingOutsideGroup(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.preventForkingOutsideGroup = mapped
    }

    /**
     * @param value Determine if developers can create projects in the group. Valid values are: `noone`, `maintainer`, `developer`
     */
    @JvmName("eldpqrrjcihcyltu")
    public suspend fun projectCreationLevel(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.projectCreationLevel = mapped
    }

    /**
     * @param value Push rules for the group.
     */
    @JvmName("etgmledsqklgpifb")
    public suspend fun pushRules(`value`: GroupPushRulesArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.pushRules = mapped
    }

    /**
     * @param argument Push rules for the group.
     */
    @JvmName("umxlevyrgcgypjta")
    public suspend fun pushRules(argument: suspend GroupPushRulesArgsBuilder.() -> Unit) {
        val toBeMapped = GroupPushRulesArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.pushRules = mapped
    }

    /**
     * @param value Allow users to request member access.
     */
    @JvmName("rbmehqejwmkdxsnt")
    public suspend fun requestAccessEnabled(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.requestAccessEnabled = mapped
    }

    /**
     * @param value Require all users in this group to setup Two-factor authentication.
     */
    @JvmName("rbjvqkqsvlyuctri")
    public suspend fun requireTwoFactorAuthentication(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.requireTwoFactorAuthentication = mapped
    }

    /**
     * @param value Prevent sharing a project with another group within this group.
     */
    @JvmName("fkssharfdjnwrmro")
    public suspend fun shareWithGroupLock(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.shareWithGroupLock = mapped
    }

    /**
     * @param value Can be set by administrators only. Maximum number of monthly CI/CD minutes for this group. Can be nil (default; inherit system default), 0 (unlimited), or > 0.
     */
    @JvmName("bawvxardwsfvrwbk")
    public suspend fun sharedRunnersMinutesLimit(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sharedRunnersMinutesLimit = mapped
    }

    /**
     * @param value Enable or disable shared runners for a group’s subgroups and projects. Valid values are: `enabled`, `disabled_and_overridable`, `disabled_and_unoverridable`, `disabled_with_override`.
     */
    @JvmName("dkpdsfgtdkjbpwkt")
    public suspend fun sharedRunnersSetting(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sharedRunnersSetting = mapped
    }

    /**
     * @param value Allowed to create subgroups. Valid values are: `owner`, `maintainer`.
     */
    @JvmName("fmiospygwdqgcwma")
    public suspend fun subgroupCreationLevel(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.subgroupCreationLevel = mapped
    }

    /**
     * @param value Defaults to 48. Time before Two-factor authentication is enforced (in hours).
     */
    @JvmName("qbkgehmgnwakgrnw")
    public suspend fun twoFactorGracePeriod(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.twoFactorGracePeriod = mapped
    }

    /**
     * @param value The group's visibility. Can be `private`, `internal`, or `public`. Valid values are: `private`, `internal`, `public`.
     */
    @JvmName("nswnymqsuqxalbpl")
    public suspend fun visibilityLevel(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.visibilityLevel = mapped
    }

    /**
     * @param value The group's wiki access level. Only available on Premium and Ultimate plans. Valid values are `disabled`, `private`, `enabled`.
     */
    @JvmName("jvglmkpwgraswfrb")
    public suspend fun wikiAccessLevel(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.wikiAccessLevel = mapped
    }

    internal fun build(): GroupArgs = GroupArgs(
        autoDevopsEnabled = autoDevopsEnabled,
        avatar = avatar,
        avatarHash = avatarHash,
        defaultBranchProtection = defaultBranchProtection,
        defaultBranchProtectionDefaults = defaultBranchProtectionDefaults,
        description = description,
        emailsEnabled = emailsEnabled,
        extraSharedRunnersMinutesLimit = extraSharedRunnersMinutesLimit,
        ipRestrictionRanges = ipRestrictionRanges,
        lfsEnabled = lfsEnabled,
        membershipLock = membershipLock,
        mentionsDisabled = mentionsDisabled,
        name = name,
        parentId = parentId,
        path = path,
        permanentlyRemoveOnDelete = permanentlyRemoveOnDelete,
        preventForkingOutsideGroup = preventForkingOutsideGroup,
        projectCreationLevel = projectCreationLevel,
        pushRules = pushRules,
        requestAccessEnabled = requestAccessEnabled,
        requireTwoFactorAuthentication = requireTwoFactorAuthentication,
        shareWithGroupLock = shareWithGroupLock,
        sharedRunnersMinutesLimit = sharedRunnersMinutesLimit,
        sharedRunnersSetting = sharedRunnersSetting,
        subgroupCreationLevel = subgroupCreationLevel,
        twoFactorGracePeriod = twoFactorGracePeriod,
        visibilityLevel = visibilityLevel,
        wikiAccessLevel = wikiAccessLevel,
    )
}
