@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.gitlab.IntegrationExternalWikiArgs.builder
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * The `gitlab.IntegrationExternalWiki` resource allows to manage the lifecycle of a project integration with External Wiki Service.
 * **Upstream API**: [GitLab REST API docs](https://docs.gitlab.com/ee/api/integrations.html#external-wiki)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * const awesomeProject = new gitlab.Project("awesome_project", {
 *     name: "awesome_project",
 *     description: "My awesome project.",
 *     visibilityLevel: "public",
 * });
 * const wiki = new gitlab.IntegrationExternalWiki("wiki", {
 *     project: awesomeProject.id,
 *     externalWikiUrl: "https://MyAwesomeExternalWikiURL.com",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * awesome_project = gitlab.Project("awesome_project",
 *     name="awesome_project",
 *     description="My awesome project.",
 *     visibility_level="public")
 * wiki = gitlab.IntegrationExternalWiki("wiki",
 *     project=awesome_project.id,
 *     external_wiki_url="https://MyAwesomeExternalWikiURL.com")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     var awesomeProject = new GitLab.Project("awesome_project", new()
 *     {
 *         Name = "awesome_project",
 *         Description = "My awesome project.",
 *         VisibilityLevel = "public",
 *     });
 *     var wiki = new GitLab.IntegrationExternalWiki("wiki", new()
 *     {
 *         Project = awesomeProject.Id,
 *         ExternalWikiUrl = "https://MyAwesomeExternalWikiURL.com",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		awesomeProject, err := gitlab.NewProject(ctx, "awesome_project", &gitlab.ProjectArgs{
 * 			Name:            pulumi.String("awesome_project"),
 * 			Description:     pulumi.String("My awesome project."),
 * 			VisibilityLevel: pulumi.String("public"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = gitlab.NewIntegrationExternalWiki(ctx, "wiki", &gitlab.IntegrationExternalWikiArgs{
 * 			Project:         awesomeProject.ID(),
 * 			ExternalWikiUrl: pulumi.String("https://MyAwesomeExternalWikiURL.com"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.Project;
 * import com.pulumi.gitlab.ProjectArgs;
 * import com.pulumi.gitlab.IntegrationExternalWiki;
 * import com.pulumi.gitlab.IntegrationExternalWikiArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var awesomeProject = new Project("awesomeProject", ProjectArgs.builder()
 *             .name("awesome_project")
 *             .description("My awesome project.")
 *             .visibilityLevel("public")
 *             .build());
 *         var wiki = new IntegrationExternalWiki("wiki", IntegrationExternalWikiArgs.builder()
 *             .project(awesomeProject.id())
 *             .externalWikiUrl("https://MyAwesomeExternalWikiURL.com")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   awesomeProject:
 *     type: gitlab:Project
 *     name: awesome_project
 *     properties:
 *       name: awesome_project
 *       description: My awesome project.
 *       visibilityLevel: public
 *   wiki:
 *     type: gitlab:IntegrationExternalWiki
 *     properties:
 *       project: ${awesomeProject.id}
 *       externalWikiUrl: https://MyAwesomeExternalWikiURL.com
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * You can import a gitlab_integration_external_wiki state using the project ID, e.g.
 * ```sh
 * $ pulumi import gitlab:index/integrationExternalWiki:IntegrationExternalWiki wiki 1
 * ```
 * @property externalWikiUrl The URL of the external wiki.
 * @property project ID of the project you want to activate integration on.
 */
public data class IntegrationExternalWikiArgs(
    public val externalWikiUrl: Output<String>? = null,
    public val project: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.gitlab.IntegrationExternalWikiArgs> {
    override fun toJava(): com.pulumi.gitlab.IntegrationExternalWikiArgs =
        com.pulumi.gitlab.IntegrationExternalWikiArgs.builder()
            .externalWikiUrl(externalWikiUrl?.applyValue({ args0 -> args0 }))
            .project(project?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [IntegrationExternalWikiArgs].
 */
@PulumiTagMarker
public class IntegrationExternalWikiArgsBuilder internal constructor() {
    private var externalWikiUrl: Output<String>? = null

    private var project: Output<String>? = null

    /**
     * @param value The URL of the external wiki.
     */
    @JvmName("mxomspgjbwodbigf")
    public suspend fun externalWikiUrl(`value`: Output<String>) {
        this.externalWikiUrl = value
    }

    /**
     * @param value ID of the project you want to activate integration on.
     */
    @JvmName("pbubrqcaoenucakf")
    public suspend fun project(`value`: Output<String>) {
        this.project = value
    }

    /**
     * @param value The URL of the external wiki.
     */
    @JvmName("ljhgmsvevfpfprjf")
    public suspend fun externalWikiUrl(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.externalWikiUrl = mapped
    }

    /**
     * @param value ID of the project you want to activate integration on.
     */
    @JvmName("ndhkeaaqvefclusj")
    public suspend fun project(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.project = mapped
    }

    internal fun build(): IntegrationExternalWikiArgs = IntegrationExternalWikiArgs(
        externalWikiUrl = externalWikiUrl,
        project = project,
    )
}
