@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.gitlab.IntegrationGithubArgs.builder
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * The `gitlab.IntegrationGithub` resource allows to manage the lifecycle of a project integration with GitHub.
 * > This resource requires a GitLab Enterprise instance.
 * **Upstream API**: [GitLab REST API docs](https://docs.gitlab.com/ee/api/integrations.html#github)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * const awesomeProject = new gitlab.Project("awesome_project", {
 *     name: "awesome_project",
 *     description: "My awesome project.",
 *     visibilityLevel: "public",
 * });
 * const github = new gitlab.IntegrationGithub("github", {
 *     project: awesomeProject.id,
 *     token: "REDACTED",
 *     repositoryUrl: "https://github.com/gitlabhq/terraform-provider-gitlab",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * awesome_project = gitlab.Project("awesome_project",
 *     name="awesome_project",
 *     description="My awesome project.",
 *     visibility_level="public")
 * github = gitlab.IntegrationGithub("github",
 *     project=awesome_project.id,
 *     token="REDACTED",
 *     repository_url="https://github.com/gitlabhq/terraform-provider-gitlab")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     var awesomeProject = new GitLab.Project("awesome_project", new()
 *     {
 *         Name = "awesome_project",
 *         Description = "My awesome project.",
 *         VisibilityLevel = "public",
 *     });
 *     var github = new GitLab.IntegrationGithub("github", new()
 *     {
 *         Project = awesomeProject.Id,
 *         Token = "REDACTED",
 *         RepositoryUrl = "https://github.com/gitlabhq/terraform-provider-gitlab",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		awesomeProject, err := gitlab.NewProject(ctx, "awesome_project", &gitlab.ProjectArgs{
 * 			Name:            pulumi.String("awesome_project"),
 * 			Description:     pulumi.String("My awesome project."),
 * 			VisibilityLevel: pulumi.String("public"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = gitlab.NewIntegrationGithub(ctx, "github", &gitlab.IntegrationGithubArgs{
 * 			Project:       awesomeProject.ID(),
 * 			Token:         pulumi.String("REDACTED"),
 * 			RepositoryUrl: pulumi.String("https://github.com/gitlabhq/terraform-provider-gitlab"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.Project;
 * import com.pulumi.gitlab.ProjectArgs;
 * import com.pulumi.gitlab.IntegrationGithub;
 * import com.pulumi.gitlab.IntegrationGithubArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var awesomeProject = new Project("awesomeProject", ProjectArgs.builder()
 *             .name("awesome_project")
 *             .description("My awesome project.")
 *             .visibilityLevel("public")
 *             .build());
 *         var github = new IntegrationGithub("github", IntegrationGithubArgs.builder()
 *             .project(awesomeProject.id())
 *             .token("REDACTED")
 *             .repositoryUrl("https://github.com/gitlabhq/terraform-provider-gitlab")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   awesomeProject:
 *     type: gitlab:Project
 *     name: awesome_project
 *     properties:
 *       name: awesome_project
 *       description: My awesome project.
 *       visibilityLevel: public
 *   github:
 *     type: gitlab:IntegrationGithub
 *     properties:
 *       project: ${awesomeProject.id}
 *       token: REDACTED
 *       repositoryUrl: https://github.com/gitlabhq/terraform-provider-gitlab
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * ```sh
 * $ pulumi import gitlab:index/integrationGithub:IntegrationGithub You can import a gitlab_integration_github state using `<resource> <project_id>`:
 * ```
 * ```sh
 * $ pulumi import gitlab:index/integrationGithub:IntegrationGithub github 1
 * ```
 * @property project ID of the project you want to activate integration on.
 * @property repositoryUrl
 * @property staticContext Append instance name instead of branch to the status. Must enable to set a GitLab status check as *required* in GitHub. See [Static / dynamic status check names] to learn more.
 * @property token A GitHub personal access token with at least `repo:status` scope.
 */
public data class IntegrationGithubArgs(
    public val project: Output<String>? = null,
    public val repositoryUrl: Output<String>? = null,
    public val staticContext: Output<Boolean>? = null,
    public val token: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.gitlab.IntegrationGithubArgs> {
    override fun toJava(): com.pulumi.gitlab.IntegrationGithubArgs =
        com.pulumi.gitlab.IntegrationGithubArgs.builder()
            .project(project?.applyValue({ args0 -> args0 }))
            .repositoryUrl(repositoryUrl?.applyValue({ args0 -> args0 }))
            .staticContext(staticContext?.applyValue({ args0 -> args0 }))
            .token(token?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [IntegrationGithubArgs].
 */
@PulumiTagMarker
public class IntegrationGithubArgsBuilder internal constructor() {
    private var project: Output<String>? = null

    private var repositoryUrl: Output<String>? = null

    private var staticContext: Output<Boolean>? = null

    private var token: Output<String>? = null

    /**
     * @param value ID of the project you want to activate integration on.
     */
    @JvmName("hjicwiibiodsihfi")
    public suspend fun project(`value`: Output<String>) {
        this.project = value
    }

    /**
     * @param value
     */
    @JvmName("wrhgunjpmsohlhbd")
    public suspend fun repositoryUrl(`value`: Output<String>) {
        this.repositoryUrl = value
    }

    /**
     * @param value Append instance name instead of branch to the status. Must enable to set a GitLab status check as *required* in GitHub. See [Static / dynamic status check names] to learn more.
     */
    @JvmName("wmjjfuwmhguepwca")
    public suspend fun staticContext(`value`: Output<Boolean>) {
        this.staticContext = value
    }

    /**
     * @param value A GitHub personal access token with at least `repo:status` scope.
     */
    @JvmName("xiqsatxtusefkncp")
    public suspend fun token(`value`: Output<String>) {
        this.token = value
    }

    /**
     * @param value ID of the project you want to activate integration on.
     */
    @JvmName("ouupjomgygbpecuf")
    public suspend fun project(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.project = mapped
    }

    /**
     * @param value
     */
    @JvmName("psnloenfhpbkibxs")
    public suspend fun repositoryUrl(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.repositoryUrl = mapped
    }

    /**
     * @param value Append instance name instead of branch to the status. Must enable to set a GitLab status check as *required* in GitHub. See [Static / dynamic status check names] to learn more.
     */
    @JvmName("ojphgswmrncsqjel")
    public suspend fun staticContext(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.staticContext = mapped
    }

    /**
     * @param value A GitHub personal access token with at least `repo:status` scope.
     */
    @JvmName("eeeyxkjlpfyukunw")
    public suspend fun token(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.token = mapped
    }

    internal fun build(): IntegrationGithubArgs = IntegrationGithubArgs(
        project = project,
        repositoryUrl = repositoryUrl,
        staticContext = staticContext,
        token = token,
    )
}
