@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.gitlab.PersonalAccessTokenArgs.builder
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlin.jvm.JvmName

/**
 * The `gitlab.PersonalAccessToken` resource allows to manage the lifecycle of a personal access token.
 * > This resource requires administration privileges.
 * > Use of the `timestamp()` function with expires_at will cause the resource to be re-created with every apply, it's recommended to use `plantimestamp()` or a static value instead.
 * > Observability scopes are in beta and may not work on all instances. See more details in [the documentation](https://docs.gitlab.com/ee/operations/tracing.html)
 * > Due to [Automatic reuse detection](https://docs.gitlab.com/ee/api/personal_access_tokens.html#automatic-reuse-detection) it's possible that a new Personal Access Token will immediately be revoked. Check if an old process using the old token is running if this happens.
 * **Upstream API**: [GitLab API docs](https://docs.gitlab.com/ee/api/personal_access_tokens.html)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * const example = new gitlab.PersonalAccessToken("example", {
 *     userId: 25,
 *     name: "Example personal access token",
 *     expiresAt: "2020-03-14",
 *     scopes: ["api"],
 * });
 * const exampleProjectVariable = new gitlab.ProjectVariable("example", {
 *     project: exampleGitlabProject.id,
 *     key: "pat",
 *     value: example.token,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * example = gitlab.PersonalAccessToken("example",
 *     user_id=25,
 *     name="Example personal access token",
 *     expires_at="2020-03-14",
 *     scopes=["api"])
 * example_project_variable = gitlab.ProjectVariable("example",
 *     project=example_gitlab_project["id"],
 *     key="pat",
 *     value=example.token)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new GitLab.PersonalAccessToken("example", new()
 *     {
 *         UserId = 25,
 *         Name = "Example personal access token",
 *         ExpiresAt = "2020-03-14",
 *         Scopes = new[]
 *         {
 *             "api",
 *         },
 *     });
 *     var exampleProjectVariable = new GitLab.ProjectVariable("example", new()
 *     {
 *         Project = exampleGitlabProject.Id,
 *         Key = "pat",
 *         Value = example.Token,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := gitlab.NewPersonalAccessToken(ctx, "example", &gitlab.PersonalAccessTokenArgs{
 * 			UserId:    pulumi.Int(25),
 * 			Name:      pulumi.String("Example personal access token"),
 * 			ExpiresAt: pulumi.String("2020-03-14"),
 * 			Scopes: pulumi.StringArray{
 * 				pulumi.String("api"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = gitlab.NewProjectVariable(ctx, "example", &gitlab.ProjectVariableArgs{
 * 			Project: pulumi.Any(exampleGitlabProject.Id),
 * 			Key:     pulumi.String("pat"),
 * 			Value:   example.Token,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.PersonalAccessToken;
 * import com.pulumi.gitlab.PersonalAccessTokenArgs;
 * import com.pulumi.gitlab.ProjectVariable;
 * import com.pulumi.gitlab.ProjectVariableArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new PersonalAccessToken("example", PersonalAccessTokenArgs.builder()
 *             .userId("25")
 *             .name("Example personal access token")
 *             .expiresAt("2020-03-14")
 *             .scopes("api")
 *             .build());
 *         var exampleProjectVariable = new ProjectVariable("exampleProjectVariable", ProjectVariableArgs.builder()
 *             .project(exampleGitlabProject.id())
 *             .key("pat")
 *             .value(example.token())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: gitlab:PersonalAccessToken
 *     properties:
 *       userId: '25'
 *       name: Example personal access token
 *       expiresAt: 2020-03-14
 *       scopes:
 *         - api
 *   exampleProjectVariable:
 *     type: gitlab:ProjectVariable
 *     name: example
 *     properties:
 *       project: ${exampleGitlabProject.id}
 *       key: pat
 *       value: ${example.token}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * A GitLab Personal Access Token can be imported using a key composed of `<user-id>:<token-id>`, e.g.
 * ```sh
 * $ pulumi import gitlab:index/personalAccessToken:PersonalAccessToken example "12345:1"
 * ```
 * NOTE: the `token` resource attribute is not available for imported resources as this information cannot be read from the GitLab API.
 * @property expiresAt When the token will expire, YYYY-MM-DD format.
 * @property name The name of the personal access token.
 * @property scopes The scopes of the personal access token. valid values are: `api`, `read_user`, `read_api`, `read_repository`, `write_repository`, `read_registry`, `write_registry`, `sudo`, `admin_mode`, `create_runner`, `manage_runner`, `ai_features`, `k8s_proxy`, `read_service_ping`
 * @property userId The ID of the user.
 */
public data class PersonalAccessTokenArgs(
    public val expiresAt: Output<String>? = null,
    public val name: Output<String>? = null,
    public val scopes: Output<List<String>>? = null,
    public val userId: Output<Int>? = null,
) : ConvertibleToJava<com.pulumi.gitlab.PersonalAccessTokenArgs> {
    override fun toJava(): com.pulumi.gitlab.PersonalAccessTokenArgs =
        com.pulumi.gitlab.PersonalAccessTokenArgs.builder()
            .expiresAt(expiresAt?.applyValue({ args0 -> args0 }))
            .name(name?.applyValue({ args0 -> args0 }))
            .scopes(scopes?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .userId(userId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [PersonalAccessTokenArgs].
 */
@PulumiTagMarker
public class PersonalAccessTokenArgsBuilder internal constructor() {
    private var expiresAt: Output<String>? = null

    private var name: Output<String>? = null

    private var scopes: Output<List<String>>? = null

    private var userId: Output<Int>? = null

    /**
     * @param value When the token will expire, YYYY-MM-DD format.
     */
    @JvmName("pctlpekhwjrcqxtf")
    public suspend fun expiresAt(`value`: Output<String>) {
        this.expiresAt = value
    }

    /**
     * @param value The name of the personal access token.
     */
    @JvmName("wyececaebifpagkr")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value The scopes of the personal access token. valid values are: `api`, `read_user`, `read_api`, `read_repository`, `write_repository`, `read_registry`, `write_registry`, `sudo`, `admin_mode`, `create_runner`, `manage_runner`, `ai_features`, `k8s_proxy`, `read_service_ping`
     */
    @JvmName("ysxonsgyifmqilcb")
    public suspend fun scopes(`value`: Output<List<String>>) {
        this.scopes = value
    }

    @JvmName("hofqbqdrxtpqrxce")
    public suspend fun scopes(vararg values: Output<String>) {
        this.scopes = Output.all(values.asList())
    }

    /**
     * @param values The scopes of the personal access token. valid values are: `api`, `read_user`, `read_api`, `read_repository`, `write_repository`, `read_registry`, `write_registry`, `sudo`, `admin_mode`, `create_runner`, `manage_runner`, `ai_features`, `k8s_proxy`, `read_service_ping`
     */
    @JvmName("exmnlsbpebpekymq")
    public suspend fun scopes(values: List<Output<String>>) {
        this.scopes = Output.all(values)
    }

    /**
     * @param value The ID of the user.
     */
    @JvmName("omjwxiaxhtxgrrmw")
    public suspend fun userId(`value`: Output<Int>) {
        this.userId = value
    }

    /**
     * @param value When the token will expire, YYYY-MM-DD format.
     */
    @JvmName("dtcxcdimtpflgvci")
    public suspend fun expiresAt(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.expiresAt = mapped
    }

    /**
     * @param value The name of the personal access token.
     */
    @JvmName("hnlxtfqvwnqcseps")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value The scopes of the personal access token. valid values are: `api`, `read_user`, `read_api`, `read_repository`, `write_repository`, `read_registry`, `write_registry`, `sudo`, `admin_mode`, `create_runner`, `manage_runner`, `ai_features`, `k8s_proxy`, `read_service_ping`
     */
    @JvmName("vxaekwryghnlddet")
    public suspend fun scopes(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.scopes = mapped
    }

    /**
     * @param values The scopes of the personal access token. valid values are: `api`, `read_user`, `read_api`, `read_repository`, `write_repository`, `read_registry`, `write_registry`, `sudo`, `admin_mode`, `create_runner`, `manage_runner`, `ai_features`, `k8s_proxy`, `read_service_ping`
     */
    @JvmName("vqowrkurqrtxfndu")
    public suspend fun scopes(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.scopes = mapped
    }

    /**
     * @param value The ID of the user.
     */
    @JvmName("nsokcnsiamqwefsu")
    public suspend fun userId(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.userId = mapped
    }

    internal fun build(): PersonalAccessTokenArgs = PersonalAccessTokenArgs(
        expiresAt = expiresAt,
        name = name,
        scopes = scopes,
        userId = userId,
    )
}
