@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.gitlab.ProjectAccessTokenArgs.builder
import com.pulumi.gitlab.kotlin.inputs.ProjectAccessTokenRotationConfigurationArgs
import com.pulumi.gitlab.kotlin.inputs.ProjectAccessTokenRotationConfigurationArgsBuilder
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.jvm.JvmName

/**
 * The `gitlab.ProjectAccessToken` resource allows to manage the lifecycle of a project access token.
 * > Observability scopes are in beta and may not work on all instances. See more details in [the documentation](https://docs.gitlab.com/ee/operations/tracing.html)
 * > Use `rotation_configuration` to automatically rotate tokens instead of using `timestamp()` as timestamp will cause changes with every plan. `pulumi up` must still be run to rotate the token.
 * > Due to [Automatic reuse detection](https://docs.gitlab.com/ee/api/project_access_tokens.html#automatic-reuse-detection) it's possible that a new Project Access Token will immediately be revoked. Check if an old process using the old token is running if this happens.
 * **Upstream API**: [GitLab API docs](https://docs.gitlab.com/ee/api/project_access_tokens.html)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * const example = new gitlab.ProjectAccessToken("example", {
 *     project: "25",
 *     name: "Example project access token",
 *     expiresAt: "2020-03-14",
 *     accessLevel: "reporter",
 *     scopes: ["api"],
 * });
 * const exampleProjectVariable = new gitlab.ProjectVariable("example", {
 *     project: exampleGitlabProject.id,
 *     key: "pat",
 *     value: example.token,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * example = gitlab.ProjectAccessToken("example",
 *     project="25",
 *     name="Example project access token",
 *     expires_at="2020-03-14",
 *     access_level="reporter",
 *     scopes=["api"])
 * example_project_variable = gitlab.ProjectVariable("example",
 *     project=example_gitlab_project["id"],
 *     key="pat",
 *     value=example.token)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = new GitLab.ProjectAccessToken("example", new()
 *     {
 *         Project = "25",
 *         Name = "Example project access token",
 *         ExpiresAt = "2020-03-14",
 *         AccessLevel = "reporter",
 *         Scopes = new[]
 *         {
 *             "api",
 *         },
 *     });
 *     var exampleProjectVariable = new GitLab.ProjectVariable("example", new()
 *     {
 *         Project = exampleGitlabProject.Id,
 *         Key = "pat",
 *         Value = example.Token,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := gitlab.NewProjectAccessToken(ctx, "example", &gitlab.ProjectAccessTokenArgs{
 * 			Project:     pulumi.String("25"),
 * 			Name:        pulumi.String("Example project access token"),
 * 			ExpiresAt:   pulumi.String("2020-03-14"),
 * 			AccessLevel: pulumi.String("reporter"),
 * 			Scopes: pulumi.StringArray{
 * 				pulumi.String("api"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = gitlab.NewProjectVariable(ctx, "example", &gitlab.ProjectVariableArgs{
 * 			Project: pulumi.Any(exampleGitlabProject.Id),
 * 			Key:     pulumi.String("pat"),
 * 			Value:   example.Token,
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.ProjectAccessToken;
 * import com.pulumi.gitlab.ProjectAccessTokenArgs;
 * import com.pulumi.gitlab.ProjectVariable;
 * import com.pulumi.gitlab.ProjectVariableArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var example = new ProjectAccessToken("example", ProjectAccessTokenArgs.builder()
 *             .project("25")
 *             .name("Example project access token")
 *             .expiresAt("2020-03-14")
 *             .accessLevel("reporter")
 *             .scopes("api")
 *             .build());
 *         var exampleProjectVariable = new ProjectVariable("exampleProjectVariable", ProjectVariableArgs.builder()
 *             .project(exampleGitlabProject.id())
 *             .key("pat")
 *             .value(example.token())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   example:
 *     type: gitlab:ProjectAccessToken
 *     properties:
 *       project: '25'
 *       name: Example project access token
 *       expiresAt: 2020-03-14
 *       accessLevel: reporter
 *       scopes:
 *         - api
 *   exampleProjectVariable:
 *     type: gitlab:ProjectVariable
 *     name: example
 *     properties:
 *       project: ${exampleGitlabProject.id}
 *       key: pat
 *       value: ${example.token}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * A GitLab Project Access Token can be imported using a key composed of `<project-id>:<token-id>`, e.g.
 * ```sh
 * $ pulumi import gitlab:index/projectAccessToken:ProjectAccessToken example "12345:1"
 * ```
 * NOTE: the `token` resource attribute is not available for imported resources as this information cannot be read from the GitLab API.
 * @property accessLevel The access level for the project access token. Valid values are: `no one`, `minimal`, `guest`, `reporter`, `developer`, `maintainer`, `owner`. Default is `maintainer`.
 * @property expiresAt When the token will expire, YYYY-MM-DD format. Is automatically set when `rotation_configuration` is used.
 * @property name The name of the project access token.
 * @property project The ID or full path of the project.
 * @property rotationConfiguration The configuration for when to rotate a token automatically. Will not rotate a token until `pulumi up` is run.
 * @property scopes The scopes of the project access token. valid values are: `api`, `read_api`, `read_registry`, `write_registry`, `read_repository`, `write_repository`, `create_runner`, `manage_runner`, `ai_features`, `k8s_proxy`, `read_observability`, `write_observability`
 */
public data class ProjectAccessTokenArgs(
    public val accessLevel: Output<String>? = null,
    public val expiresAt: Output<String>? = null,
    public val name: Output<String>? = null,
    public val project: Output<String>? = null,
    public val rotationConfiguration: Output<ProjectAccessTokenRotationConfigurationArgs>? = null,
    public val scopes: Output<List<String>>? = null,
) : ConvertibleToJava<com.pulumi.gitlab.ProjectAccessTokenArgs> {
    override fun toJava(): com.pulumi.gitlab.ProjectAccessTokenArgs =
        com.pulumi.gitlab.ProjectAccessTokenArgs.builder()
            .accessLevel(accessLevel?.applyValue({ args0 -> args0 }))
            .expiresAt(expiresAt?.applyValue({ args0 -> args0 }))
            .name(name?.applyValue({ args0 -> args0 }))
            .project(project?.applyValue({ args0 -> args0 }))
            .rotationConfiguration(
                rotationConfiguration?.applyValue({ args0 ->
                    args0.let({ args0 ->
                        args0.toJava()
                    })
                }),
            )
            .scopes(scopes?.applyValue({ args0 -> args0.map({ args0 -> args0 }) })).build()
}

/**
 * Builder for [ProjectAccessTokenArgs].
 */
@PulumiTagMarker
public class ProjectAccessTokenArgsBuilder internal constructor() {
    private var accessLevel: Output<String>? = null

    private var expiresAt: Output<String>? = null

    private var name: Output<String>? = null

    private var project: Output<String>? = null

    private var rotationConfiguration: Output<ProjectAccessTokenRotationConfigurationArgs>? = null

    private var scopes: Output<List<String>>? = null

    /**
     * @param value The access level for the project access token. Valid values are: `no one`, `minimal`, `guest`, `reporter`, `developer`, `maintainer`, `owner`. Default is `maintainer`.
     */
    @JvmName("jdhxigkpxmhwjhsh")
    public suspend fun accessLevel(`value`: Output<String>) {
        this.accessLevel = value
    }

    /**
     * @param value When the token will expire, YYYY-MM-DD format. Is automatically set when `rotation_configuration` is used.
     */
    @JvmName("vunhsdyssjtekjps")
    public suspend fun expiresAt(`value`: Output<String>) {
        this.expiresAt = value
    }

    /**
     * @param value The name of the project access token.
     */
    @JvmName("oofyktxreimsugfk")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value The ID or full path of the project.
     */
    @JvmName("anhkfbpserfcgvqu")
    public suspend fun project(`value`: Output<String>) {
        this.project = value
    }

    /**
     * @param value The configuration for when to rotate a token automatically. Will not rotate a token until `pulumi up` is run.
     */
    @JvmName("evlvrniqnaddjqbh")
    public suspend fun rotationConfiguration(`value`: Output<ProjectAccessTokenRotationConfigurationArgs>) {
        this.rotationConfiguration = value
    }

    /**
     * @param value The scopes of the project access token. valid values are: `api`, `read_api`, `read_registry`, `write_registry`, `read_repository`, `write_repository`, `create_runner`, `manage_runner`, `ai_features`, `k8s_proxy`, `read_observability`, `write_observability`
     */
    @JvmName("bkcxxrxininfodbo")
    public suspend fun scopes(`value`: Output<List<String>>) {
        this.scopes = value
    }

    @JvmName("qhcmnasgsufqjmws")
    public suspend fun scopes(vararg values: Output<String>) {
        this.scopes = Output.all(values.asList())
    }

    /**
     * @param values The scopes of the project access token. valid values are: `api`, `read_api`, `read_registry`, `write_registry`, `read_repository`, `write_repository`, `create_runner`, `manage_runner`, `ai_features`, `k8s_proxy`, `read_observability`, `write_observability`
     */
    @JvmName("igycxieyccodergt")
    public suspend fun scopes(values: List<Output<String>>) {
        this.scopes = Output.all(values)
    }

    /**
     * @param value The access level for the project access token. Valid values are: `no one`, `minimal`, `guest`, `reporter`, `developer`, `maintainer`, `owner`. Default is `maintainer`.
     */
    @JvmName("nicdteoejnnswihw")
    public suspend fun accessLevel(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.accessLevel = mapped
    }

    /**
     * @param value When the token will expire, YYYY-MM-DD format. Is automatically set when `rotation_configuration` is used.
     */
    @JvmName("xrwlkljbmpfyxfpu")
    public suspend fun expiresAt(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.expiresAt = mapped
    }

    /**
     * @param value The name of the project access token.
     */
    @JvmName("sqaqdjpqaswxfowa")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value The ID or full path of the project.
     */
    @JvmName("leldjfmmwwyrlmrw")
    public suspend fun project(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.project = mapped
    }

    /**
     * @param value The configuration for when to rotate a token automatically. Will not rotate a token until `pulumi up` is run.
     */
    @JvmName("johnfaqjnyydcsaa")
    public suspend fun rotationConfiguration(`value`: ProjectAccessTokenRotationConfigurationArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.rotationConfiguration = mapped
    }

    /**
     * @param argument The configuration for when to rotate a token automatically. Will not rotate a token until `pulumi up` is run.
     */
    @JvmName("fpgcaafsvfmypmdi")
    public suspend fun rotationConfiguration(argument: suspend ProjectAccessTokenRotationConfigurationArgsBuilder.() -> Unit) {
        val toBeMapped = ProjectAccessTokenRotationConfigurationArgsBuilder().applySuspend {
            argument()
        }.build()
        val mapped = of(toBeMapped)
        this.rotationConfiguration = mapped
    }

    /**
     * @param value The scopes of the project access token. valid values are: `api`, `read_api`, `read_registry`, `write_registry`, `read_repository`, `write_repository`, `create_runner`, `manage_runner`, `ai_features`, `k8s_proxy`, `read_observability`, `write_observability`
     */
    @JvmName("xtkcvcadacdcwhst")
    public suspend fun scopes(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.scopes = mapped
    }

    /**
     * @param values The scopes of the project access token. valid values are: `api`, `read_api`, `read_registry`, `write_registry`, `read_repository`, `write_repository`, `create_runner`, `manage_runner`, `ai_features`, `k8s_proxy`, `read_observability`, `write_observability`
     */
    @JvmName("peoygtaaayxbfdxb")
    public suspend fun scopes(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.scopes = mapped
    }

    internal fun build(): ProjectAccessTokenArgs = ProjectAccessTokenArgs(
        accessLevel = accessLevel,
        expiresAt = expiresAt,
        name = name,
        project = project,
        rotationConfiguration = rotationConfiguration,
        scopes = scopes,
    )
}
