@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.gitlab.ProjectLabelArgs.builder
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * The `gitlab.ProjectLabel` resource allows to manage the lifecycle of a project label.
 * **Upstream API**: [GitLab REST API docs](https://docs.gitlab.com/ee/api/labels.html#project-labels)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * const fixme = new gitlab.ProjectLabel("fixme", {
 *     project: "example",
 *     name: "fixme",
 *     description: "issue with failing tests",
 *     color: "#ffcc00",
 * });
 * // Scoped label
 * const devopsCreate = new gitlab.ProjectLabel("devops_create", {
 *     project: example.id,
 *     name: "devops::create",
 *     description: "issue for creating infrastructure resources",
 *     color: "#ffa500",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * fixme = gitlab.ProjectLabel("fixme",
 *     project="example",
 *     name="fixme",
 *     description="issue with failing tests",
 *     color="#ffcc00")
 * # Scoped label
 * devops_create = gitlab.ProjectLabel("devops_create",
 *     project=example["id"],
 *     name="devops::create",
 *     description="issue for creating infrastructure resources",
 *     color="#ffa500")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     var fixme = new GitLab.ProjectLabel("fixme", new()
 *     {
 *         Project = "example",
 *         Name = "fixme",
 *         Description = "issue with failing tests",
 *         Color = "#ffcc00",
 *     });
 *     // Scoped label
 *     var devopsCreate = new GitLab.ProjectLabel("devops_create", new()
 *     {
 *         Project = example.Id,
 *         Name = "devops::create",
 *         Description = "issue for creating infrastructure resources",
 *         Color = "#ffa500",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := gitlab.NewProjectLabel(ctx, "fixme", &gitlab.ProjectLabelArgs{
 * 			Project:     pulumi.String("example"),
 * 			Name:        pulumi.String("fixme"),
 * 			Description: pulumi.String("issue with failing tests"),
 * 			Color:       pulumi.String("#ffcc00"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Scoped label
 * 		_, err = gitlab.NewProjectLabel(ctx, "devops_create", &gitlab.ProjectLabelArgs{
 * 			Project:     pulumi.Any(example.Id),
 * 			Name:        pulumi.String("devops::create"),
 * 			Description: pulumi.String("issue for creating infrastructure resources"),
 * 			Color:       pulumi.String("#ffa500"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.ProjectLabel;
 * import com.pulumi.gitlab.ProjectLabelArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var fixme = new ProjectLabel("fixme", ProjectLabelArgs.builder()
 *             .project("example")
 *             .name("fixme")
 *             .description("issue with failing tests")
 *             .color("#ffcc00")
 *             .build());
 *         // Scoped label
 *         var devopsCreate = new ProjectLabel("devopsCreate", ProjectLabelArgs.builder()
 *             .project(example.id())
 *             .name("devops::create")
 *             .description("issue for creating infrastructure resources")
 *             .color("#ffa500")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   fixme:
 *     type: gitlab:ProjectLabel
 *     properties:
 *       project: example
 *       name: fixme
 *       description: issue with failing tests
 *       color: '#ffcc00'
 *   # Scoped label
 *   devopsCreate:
 *     type: gitlab:ProjectLabel
 *     name: devops_create
 *     properties:
 *       project: ${example.id}
 *       name: devops::create
 *       description: issue for creating infrastructure resources
 *       color: '#ffa500'
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Gitlab Project labels can be imported using an id made up of `{project_id}:{group_label_id}`, e.g.
 * ```sh
 * $ pulumi import gitlab:index/projectLabel:ProjectLabel example 12345:fixme
 * ```
 * @property color The color of the label given in 6-digit hex notation with leading '#' sign (e.g. #FFAABB) or one of the [CSS color names](https://developer.mozilla.org/en-US/docs/Web/CSS/color_value#Color_keywords).
 * @property description The description of the label.
 * @property name The name of the label.
 * @property project The name or id of the project to add the label to.
 */
public data class ProjectLabelArgs(
    public val color: Output<String>? = null,
    public val description: Output<String>? = null,
    public val name: Output<String>? = null,
    public val project: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.gitlab.ProjectLabelArgs> {
    override fun toJava(): com.pulumi.gitlab.ProjectLabelArgs =
        com.pulumi.gitlab.ProjectLabelArgs.builder()
            .color(color?.applyValue({ args0 -> args0 }))
            .description(description?.applyValue({ args0 -> args0 }))
            .name(name?.applyValue({ args0 -> args0 }))
            .project(project?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [ProjectLabelArgs].
 */
@PulumiTagMarker
public class ProjectLabelArgsBuilder internal constructor() {
    private var color: Output<String>? = null

    private var description: Output<String>? = null

    private var name: Output<String>? = null

    private var project: Output<String>? = null

    /**
     * @param value The color of the label given in 6-digit hex notation with leading '#' sign (e.g. #FFAABB) or one of the [CSS color names](https://developer.mozilla.org/en-US/docs/Web/CSS/color_value#Color_keywords).
     */
    @JvmName("cnywpwnejdmtmyii")
    public suspend fun color(`value`: Output<String>) {
        this.color = value
    }

    /**
     * @param value The description of the label.
     */
    @JvmName("kswbqlbjpgvilojb")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value The name of the label.
     */
    @JvmName("tnbhyeibshfnspuv")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value The name or id of the project to add the label to.
     */
    @JvmName("advqumbhykhrqkot")
    public suspend fun project(`value`: Output<String>) {
        this.project = value
    }

    /**
     * @param value The color of the label given in 6-digit hex notation with leading '#' sign (e.g. #FFAABB) or one of the [CSS color names](https://developer.mozilla.org/en-US/docs/Web/CSS/color_value#Color_keywords).
     */
    @JvmName("dmnowwskexdtscqd")
    public suspend fun color(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.color = mapped
    }

    /**
     * @param value The description of the label.
     */
    @JvmName("dlhoeqxvlpuxtfrn")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value The name of the label.
     */
    @JvmName("cvirklmppitvqofl")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value The name or id of the project to add the label to.
     */
    @JvmName("kesqijyeregncvtd")
    public suspend fun project(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.project = mapped
    }

    internal fun build(): ProjectLabelArgs = ProjectLabelArgs(
        color = color,
        description = description,
        name = name,
        project = project,
    )
}
