@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.gitlab.ProjectMilestoneArgs.builder
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * The `gitlab.ProjectMilestone` resource allows to manage the lifecycle of a project milestone.
 * **Upstream API**: [GitLab REST API docs](https://docs.gitlab.com/ee/api/milestones.html)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * // Create a project for the milestone to use
 * const example = new gitlab.Project("example", {
 *     name: "example",
 *     description: "An example project",
 *     namespaceId: exampleGitlabGroup.id,
 * });
 * const exampleProjectMilestone = new gitlab.ProjectMilestone("example", {
 *     project: example.id,
 *     title: "example",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * # Create a project for the milestone to use
 * example = gitlab.Project("example",
 *     name="example",
 *     description="An example project",
 *     namespace_id=example_gitlab_group["id"])
 * example_project_milestone = gitlab.ProjectMilestone("example",
 *     project=example.id,
 *     title="example")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     // Create a project for the milestone to use
 *     var example = new GitLab.Project("example", new()
 *     {
 *         Name = "example",
 *         Description = "An example project",
 *         NamespaceId = exampleGitlabGroup.Id,
 *     });
 *     var exampleProjectMilestone = new GitLab.ProjectMilestone("example", new()
 *     {
 *         Project = example.Id,
 *         Title = "example",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		// Create a project for the milestone to use
 * 		example, err := gitlab.NewProject(ctx, "example", &gitlab.ProjectArgs{
 * 			Name:        pulumi.String("example"),
 * 			Description: pulumi.String("An example project"),
 * 			NamespaceId: pulumi.Any(exampleGitlabGroup.Id),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = gitlab.NewProjectMilestone(ctx, "example", &gitlab.ProjectMilestoneArgs{
 * 			Project: example.ID(),
 * 			Title:   pulumi.String("example"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.Project;
 * import com.pulumi.gitlab.ProjectArgs;
 * import com.pulumi.gitlab.ProjectMilestone;
 * import com.pulumi.gitlab.ProjectMilestoneArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         // Create a project for the milestone to use
 *         var example = new Project("example", ProjectArgs.builder()
 *             .name("example")
 *             .description("An example project")
 *             .namespaceId(exampleGitlabGroup.id())
 *             .build());
 *         var exampleProjectMilestone = new ProjectMilestone("exampleProjectMilestone", ProjectMilestoneArgs.builder()
 *             .project(example.id())
 *             .title("example")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   # Create a project for the milestone to use
 *   example:
 *     type: gitlab:Project
 *     properties:
 *       name: example
 *       description: An example project
 *       namespaceId: ${exampleGitlabGroup.id}
 *   exampleProjectMilestone:
 *     type: gitlab:ProjectMilestone
 *     name: example
 *     properties:
 *       project: ${example.id}
 *       title: example
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Gitlab project milestone can be imported with a key composed of `<project>:<milestone_id>`, e.g.
 * ```sh
 * $ pulumi import gitlab:index/projectMilestone:ProjectMilestone example "12345:11"
 * ```
 * @property description The description of the milestone.
 * @property dueDate The due date of the milestone. Date time string in the format YYYY-MM-DD, for example 2016-03-11.
 * @property project The ID or URL-encoded path of the project owned by the authenticated user.
 * @property startDate The start date of the milestone. Date time string in the format YYYY-MM-DD, for example 2016-03-11.
 * @property state The state of the milestone. Valid values are: `active`, `closed`.
 * @property title The title of a milestone.
 */
public data class ProjectMilestoneArgs(
    public val description: Output<String>? = null,
    public val dueDate: Output<String>? = null,
    public val project: Output<String>? = null,
    public val startDate: Output<String>? = null,
    public val state: Output<String>? = null,
    public val title: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.gitlab.ProjectMilestoneArgs> {
    override fun toJava(): com.pulumi.gitlab.ProjectMilestoneArgs =
        com.pulumi.gitlab.ProjectMilestoneArgs.builder()
            .description(description?.applyValue({ args0 -> args0 }))
            .dueDate(dueDate?.applyValue({ args0 -> args0 }))
            .project(project?.applyValue({ args0 -> args0 }))
            .startDate(startDate?.applyValue({ args0 -> args0 }))
            .state(state?.applyValue({ args0 -> args0 }))
            .title(title?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [ProjectMilestoneArgs].
 */
@PulumiTagMarker
public class ProjectMilestoneArgsBuilder internal constructor() {
    private var description: Output<String>? = null

    private var dueDate: Output<String>? = null

    private var project: Output<String>? = null

    private var startDate: Output<String>? = null

    private var state: Output<String>? = null

    private var title: Output<String>? = null

    /**
     * @param value The description of the milestone.
     */
    @JvmName("eqxeaiqkgvvsrrre")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value The due date of the milestone. Date time string in the format YYYY-MM-DD, for example 2016-03-11.
     */
    @JvmName("fhxpoivimrwqsmss")
    public suspend fun dueDate(`value`: Output<String>) {
        this.dueDate = value
    }

    /**
     * @param value The ID or URL-encoded path of the project owned by the authenticated user.
     */
    @JvmName("artqgnjmejxvjxpl")
    public suspend fun project(`value`: Output<String>) {
        this.project = value
    }

    /**
     * @param value The start date of the milestone. Date time string in the format YYYY-MM-DD, for example 2016-03-11.
     */
    @JvmName("ihmdfqmcarrptqtc")
    public suspend fun startDate(`value`: Output<String>) {
        this.startDate = value
    }

    /**
     * @param value The state of the milestone. Valid values are: `active`, `closed`.
     */
    @JvmName("nqshpnljcaejvcxc")
    public suspend fun state(`value`: Output<String>) {
        this.state = value
    }

    /**
     * @param value The title of a milestone.
     */
    @JvmName("jjijfpeqahfwemug")
    public suspend fun title(`value`: Output<String>) {
        this.title = value
    }

    /**
     * @param value The description of the milestone.
     */
    @JvmName("tyoremvjmhhljyqm")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value The due date of the milestone. Date time string in the format YYYY-MM-DD, for example 2016-03-11.
     */
    @JvmName("yrvvdwqftlcrlseb")
    public suspend fun dueDate(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.dueDate = mapped
    }

    /**
     * @param value The ID or URL-encoded path of the project owned by the authenticated user.
     */
    @JvmName("wcsdkcrsefekujub")
    public suspend fun project(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.project = mapped
    }

    /**
     * @param value The start date of the milestone. Date time string in the format YYYY-MM-DD, for example 2016-03-11.
     */
    @JvmName("kydpoqdhppxjhymo")
    public suspend fun startDate(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.startDate = mapped
    }

    /**
     * @param value The state of the milestone. Valid values are: `active`, `closed`.
     */
    @JvmName("wsnhbsijkbjtpurw")
    public suspend fun state(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.state = mapped
    }

    /**
     * @param value The title of a milestone.
     */
    @JvmName("gjifotexwmdnmcgx")
    public suspend fun title(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.title = mapped
    }

    internal fun build(): ProjectMilestoneArgs = ProjectMilestoneArgs(
        description = description,
        dueDate = dueDate,
        project = project,
        startDate = startDate,
        state = state,
        title = title,
    )
}
