@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.gitlab.ProjectTagArgs.builder
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * The `gitlab.ProjectTag` resource allows to manage the lifecycle of a tag in a project.
 * **Upstream API**: [GitLab API docs](https://docs.gitlab.com/ee/api/tags.html)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * // Create a project for the tag to use
 * const example = new gitlab.Project("example", {
 *     name: "example",
 *     description: "An example project",
 *     namespaceId: exampleGitlabGroup.id,
 * });
 * const exampleProjectTag = new gitlab.ProjectTag("example", {
 *     name: "example",
 *     ref: "main",
 *     project: example.id,
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * # Create a project for the tag to use
 * example = gitlab.Project("example",
 *     name="example",
 *     description="An example project",
 *     namespace_id=example_gitlab_group["id"])
 * example_project_tag = gitlab.ProjectTag("example",
 *     name="example",
 *     ref="main",
 *     project=example.id)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     // Create a project for the tag to use
 *     var example = new GitLab.Project("example", new()
 *     {
 *         Name = "example",
 *         Description = "An example project",
 *         NamespaceId = exampleGitlabGroup.Id,
 *     });
 *     var exampleProjectTag = new GitLab.ProjectTag("example", new()
 *     {
 *         Name = "example",
 *         Ref = "main",
 *         Project = example.Id,
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		// Create a project for the tag to use
 * 		example, err := gitlab.NewProject(ctx, "example", &gitlab.ProjectArgs{
 * 			Name:        pulumi.String("example"),
 * 			Description: pulumi.String("An example project"),
 * 			NamespaceId: pulumi.Any(exampleGitlabGroup.Id),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = gitlab.NewProjectTag(ctx, "example", &gitlab.ProjectTagArgs{
 * 			Name:    pulumi.String("example"),
 * 			Ref:     pulumi.String("main"),
 * 			Project: example.ID(),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.Project;
 * import com.pulumi.gitlab.ProjectArgs;
 * import com.pulumi.gitlab.ProjectTag;
 * import com.pulumi.gitlab.ProjectTagArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         // Create a project for the tag to use
 *         var example = new Project("example", ProjectArgs.builder()
 *             .name("example")
 *             .description("An example project")
 *             .namespaceId(exampleGitlabGroup.id())
 *             .build());
 *         var exampleProjectTag = new ProjectTag("exampleProjectTag", ProjectTagArgs.builder()
 *             .name("example")
 *             .ref("main")
 *             .project(example.id())
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   # Create a project for the tag to use
 *   example:
 *     type: gitlab:Project
 *     properties:
 *       name: example
 *       description: An example project
 *       namespaceId: ${exampleGitlabGroup.id}
 *   exampleProjectTag:
 *     type: gitlab:ProjectTag
 *     name: example
 *     properties:
 *       name: example
 *       ref: main
 *       project: ${example.id}
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Gitlab project tags can be imported with a key composed of `<project_id>:<tag_name>`, e.g.
 * ```sh
 * $ pulumi import gitlab:index/projectTag:ProjectTag example "12345:develop"
 * ```
 * NOTE: the `ref` attribute won't be available for imported `gitlab_project_tag` resources.
 * @property message The message of the annotated tag.
 * @property name The name of a tag.
 * @property project The ID or URL-encoded path of the project owned by the authenticated user.
 * @property ref Create tag using commit SHA, another tag name, or branch name. This attribute is not available for imported resources.
 */
public data class ProjectTagArgs(
    public val message: Output<String>? = null,
    public val name: Output<String>? = null,
    public val project: Output<String>? = null,
    public val ref: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.gitlab.ProjectTagArgs> {
    override fun toJava(): com.pulumi.gitlab.ProjectTagArgs =
        com.pulumi.gitlab.ProjectTagArgs.builder()
            .message(message?.applyValue({ args0 -> args0 }))
            .name(name?.applyValue({ args0 -> args0 }))
            .project(project?.applyValue({ args0 -> args0 }))
            .ref(ref?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [ProjectTagArgs].
 */
@PulumiTagMarker
public class ProjectTagArgsBuilder internal constructor() {
    private var message: Output<String>? = null

    private var name: Output<String>? = null

    private var project: Output<String>? = null

    private var ref: Output<String>? = null

    /**
     * @param value The message of the annotated tag.
     */
    @JvmName("syjyqmxwkwxfnchw")
    public suspend fun message(`value`: Output<String>) {
        this.message = value
    }

    /**
     * @param value The name of a tag.
     */
    @JvmName("qkoxqdwtjfdjmwes")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value The ID or URL-encoded path of the project owned by the authenticated user.
     */
    @JvmName("prdnnvkhnloiiedl")
    public suspend fun project(`value`: Output<String>) {
        this.project = value
    }

    /**
     * @param value Create tag using commit SHA, another tag name, or branch name. This attribute is not available for imported resources.
     */
    @JvmName("inbmcvtlrpdymlyo")
    public suspend fun ref(`value`: Output<String>) {
        this.ref = value
    }

    /**
     * @param value The message of the annotated tag.
     */
    @JvmName("gxvahflmdllhclyd")
    public suspend fun message(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.message = mapped
    }

    /**
     * @param value The name of a tag.
     */
    @JvmName("vfxhnojhirjnvipw")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value The ID or URL-encoded path of the project owned by the authenticated user.
     */
    @JvmName("xgskbqdpnofeqrfo")
    public suspend fun project(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.project = mapped
    }

    /**
     * @param value Create tag using commit SHA, another tag name, or branch name. This attribute is not available for imported resources.
     */
    @JvmName("tbjfyemwgfxpmrvr")
    public suspend fun ref(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.ref = mapped
    }

    internal fun build(): ProjectTagArgs = ProjectTagArgs(
        message = message,
        name = name,
        project = project,
        ref = ref,
    )
}
