@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.gitlab.ReleaseLinkArgs.builder
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * The `gitlab.ReleaseLink` resource allows to manage the lifecycle of a release link.
 * **Upstream API**: [GitLab REST API docs](https://docs.gitlab.com/ee/api/releases/links.html)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * // Create a project
 * const example = new gitlab.Project("example", {
 *     name: "example",
 *     description: "An example project",
 * });
 * // Can create release link only to a tag associated with a release
 * const exampleReleaseLink = new gitlab.ReleaseLink("example", {
 *     project: example.id,
 *     tagName: "tag_name_associated_with_release",
 *     name: "test",
 *     url: "https://test/",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * # Create a project
 * example = gitlab.Project("example",
 *     name="example",
 *     description="An example project")
 * # Can create release link only to a tag associated with a release
 * example_release_link = gitlab.ReleaseLink("example",
 *     project=example.id,
 *     tag_name="tag_name_associated_with_release",
 *     name="test",
 *     url="https://test/")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     // Create a project
 *     var example = new GitLab.Project("example", new()
 *     {
 *         Name = "example",
 *         Description = "An example project",
 *     });
 *     // Can create release link only to a tag associated with a release
 *     var exampleReleaseLink = new GitLab.ReleaseLink("example", new()
 *     {
 *         Project = example.Id,
 *         TagName = "tag_name_associated_with_release",
 *         Name = "test",
 *         Url = "https://test/",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		// Create a project
 * 		example, err := gitlab.NewProject(ctx, "example", &gitlab.ProjectArgs{
 * 			Name:        pulumi.String("example"),
 * 			Description: pulumi.String("An example project"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Can create release link only to a tag associated with a release
 * 		_, err = gitlab.NewReleaseLink(ctx, "example", &gitlab.ReleaseLinkArgs{
 * 			Project: example.ID(),
 * 			TagName: pulumi.String("tag_name_associated_with_release"),
 * 			Name:    pulumi.String("test"),
 * 			Url:     pulumi.String("https://test/"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.Project;
 * import com.pulumi.gitlab.ProjectArgs;
 * import com.pulumi.gitlab.ReleaseLink;
 * import com.pulumi.gitlab.ReleaseLinkArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         // Create a project
 *         var example = new Project("example", ProjectArgs.builder()
 *             .name("example")
 *             .description("An example project")
 *             .build());
 *         // Can create release link only to a tag associated with a release
 *         var exampleReleaseLink = new ReleaseLink("exampleReleaseLink", ReleaseLinkArgs.builder()
 *             .project(example.id())
 *             .tagName("tag_name_associated_with_release")
 *             .name("test")
 *             .url("https://test/")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   # Create a project
 *   example:
 *     type: gitlab:Project
 *     properties:
 *       name: example
 *       description: An example project
 *   # Can create release link only to a tag associated with a release
 *   exampleReleaseLink:
 *     type: gitlab:ReleaseLink
 *     name: example
 *     properties:
 *       project: ${example.id}
 *       tagName: tag_name_associated_with_release
 *       name: test
 *       url: https://test/
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * Gitlab release link can be imported with a key composed of `<project>:<tag_name>:<link_id>`, e.g.
 * ```sh
 * $ pulumi import gitlab:index/releaseLink:ReleaseLink example "12345:test:2"
 * ```
 * @property filepath Relative path for a [Direct Asset link](https://docs.gitlab.com/ee/user/project/releases/index.html#permanent-links-to-release-assets).
 * @property linkType The type of the link. Valid values are `other`, `runbook`, `image`, `package`. Defaults to other.
 * @property name The name of the link. Link names must be unique within the release.
 * @property project The ID or [URL-encoded path of the project](https://docs.gitlab.com/ee/api/index.html#namespaced-path-encoding).
 * @property tagName The tag associated with the Release.
 * @property url The URL of the link. Link URLs must be unique within the release.
 */
public data class ReleaseLinkArgs(
    public val filepath: Output<String>? = null,
    public val linkType: Output<String>? = null,
    public val name: Output<String>? = null,
    public val project: Output<String>? = null,
    public val tagName: Output<String>? = null,
    public val url: Output<String>? = null,
) : ConvertibleToJava<com.pulumi.gitlab.ReleaseLinkArgs> {
    override fun toJava(): com.pulumi.gitlab.ReleaseLinkArgs =
        com.pulumi.gitlab.ReleaseLinkArgs.builder()
            .filepath(filepath?.applyValue({ args0 -> args0 }))
            .linkType(linkType?.applyValue({ args0 -> args0 }))
            .name(name?.applyValue({ args0 -> args0 }))
            .project(project?.applyValue({ args0 -> args0 }))
            .tagName(tagName?.applyValue({ args0 -> args0 }))
            .url(url?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [ReleaseLinkArgs].
 */
@PulumiTagMarker
public class ReleaseLinkArgsBuilder internal constructor() {
    private var filepath: Output<String>? = null

    private var linkType: Output<String>? = null

    private var name: Output<String>? = null

    private var project: Output<String>? = null

    private var tagName: Output<String>? = null

    private var url: Output<String>? = null

    /**
     * @param value Relative path for a [Direct Asset link](https://docs.gitlab.com/ee/user/project/releases/index.html#permanent-links-to-release-assets).
     */
    @JvmName("rqqduomjkggrrlna")
    public suspend fun filepath(`value`: Output<String>) {
        this.filepath = value
    }

    /**
     * @param value The type of the link. Valid values are `other`, `runbook`, `image`, `package`. Defaults to other.
     */
    @JvmName("oajtfkcdclyrxyba")
    public suspend fun linkType(`value`: Output<String>) {
        this.linkType = value
    }

    /**
     * @param value The name of the link. Link names must be unique within the release.
     */
    @JvmName("tglutssocykmvxnc")
    public suspend fun name(`value`: Output<String>) {
        this.name = value
    }

    /**
     * @param value The ID or [URL-encoded path of the project](https://docs.gitlab.com/ee/api/index.html#namespaced-path-encoding).
     */
    @JvmName("akxsfusyldvhhauh")
    public suspend fun project(`value`: Output<String>) {
        this.project = value
    }

    /**
     * @param value The tag associated with the Release.
     */
    @JvmName("butidonlyakxietw")
    public suspend fun tagName(`value`: Output<String>) {
        this.tagName = value
    }

    /**
     * @param value The URL of the link. Link URLs must be unique within the release.
     */
    @JvmName("adarkqtsrfdvvkmr")
    public suspend fun url(`value`: Output<String>) {
        this.url = value
    }

    /**
     * @param value Relative path for a [Direct Asset link](https://docs.gitlab.com/ee/user/project/releases/index.html#permanent-links-to-release-assets).
     */
    @JvmName("mhuophlufpbwxccb")
    public suspend fun filepath(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.filepath = mapped
    }

    /**
     * @param value The type of the link. Valid values are `other`, `runbook`, `image`, `package`. Defaults to other.
     */
    @JvmName("ykpbwdynkqwauqds")
    public suspend fun linkType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.linkType = mapped
    }

    /**
     * @param value The name of the link. Link names must be unique within the release.
     */
    @JvmName("ypmhcpevygqkgpsb")
    public suspend fun name(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.name = mapped
    }

    /**
     * @param value The ID or [URL-encoded path of the project](https://docs.gitlab.com/ee/api/index.html#namespaced-path-encoding).
     */
    @JvmName("idxbakyokmvvqesh")
    public suspend fun project(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.project = mapped
    }

    /**
     * @param value The tag associated with the Release.
     */
    @JvmName("wwhocxsyyaydxwse")
    public suspend fun tagName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tagName = mapped
    }

    /**
     * @param value The URL of the link. Link URLs must be unique within the release.
     */
    @JvmName("lncygohvcntpvhfj")
    public suspend fun url(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.url = mapped
    }

    internal fun build(): ReleaseLinkArgs = ReleaseLinkArgs(
        filepath = filepath,
        linkType = linkType,
        name = name,
        project = project,
        tagName = tagName,
        url = url,
    )
}
