@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.gitlab.UserGpgKeyArgs.builder
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * The `gitlab.UserGpgKey` resource allows to manage the lifecycle of a GPG key assigned to the current user or a specific user.
 * > Managing GPG keys for arbitrary users requires admin privileges.
 * **Upstream API**: [GitLab REST API docs](https://docs.gitlab.com/ee/api/users.html#get-a-specific-gpg-key)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * const example = gitlab.getUser({
 *     username: "example-user",
 * });
 * // Manages a GPG key for the specified user. An admin token is required if `user_id` is specified.
 * const exampleUserGpgKey = new gitlab.UserGpgKey("example", {
 *     userId: example.then(example => example.id),
 *     key: `-----BEGIN PGP PUBLIC KEY BLOCK-----
 * ...
 * -----END PGP PUBLIC KEY BLOCK-----`,
 * });
 * // Manages a GPG key for the current user
 * const exampleUser = new gitlab.UserGpgKey("example_user", {key: `-----BEGIN PGP PUBLIC KEY BLOCK-----
 * ...
 * -----END PGP PUBLIC KEY BLOCK-----`});
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * example = gitlab.get_user(username="example-user")
 * # Manages a GPG key for the specified user. An admin token is required if `user_id` is specified.
 * example_user_gpg_key = gitlab.UserGpgKey("example",
 *     user_id=example.id,
 *     key="""-----BEGIN PGP PUBLIC KEY BLOCK-----
 * ...
 * -----END PGP PUBLIC KEY BLOCK-----""")
 * # Manages a GPG key for the current user
 * example_user = gitlab.UserGpgKey("example_user", key="""-----BEGIN PGP PUBLIC KEY BLOCK-----
 * ...
 * -----END PGP PUBLIC KEY BLOCK-----""")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = GitLab.GetUser.Invoke(new()
 *     {
 *         Username = "example-user",
 *     });
 *     // Manages a GPG key for the specified user. An admin token is required if `user_id` is specified.
 *     var exampleUserGpgKey = new GitLab.UserGpgKey("example", new()
 *     {
 *         UserId = example.Apply(getUserResult => getUserResult.Id),
 *         Key = @"-----BEGIN PGP PUBLIC KEY BLOCK-----
 * ...
 * -----END PGP PUBLIC KEY BLOCK-----",
 *     });
 *     // Manages a GPG key for the current user
 *     var exampleUser = new GitLab.UserGpgKey("example_user", new()
 *     {
 *         Key = @"-----BEGIN PGP PUBLIC KEY BLOCK-----
 * ...
 * -----END PGP PUBLIC KEY BLOCK-----",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := gitlab.LookupUser(ctx, &gitlab.LookupUserArgs{
 * 			Username: pulumi.StringRef("example-user"),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Manages a GPG key for the specified user. An admin token is required if `user_id` is specified.
 * 		_, err = gitlab.NewUserGpgKey(ctx, "example", &gitlab.UserGpgKeyArgs{
 * 			UserId: pulumi.String(example.Id),
 * 			Key:    pulumi.String("-----BEGIN PGP PUBLIC KEY BLOCK-----\n...\n-----END PGP PUBLIC KEY BLOCK-----"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Manages a GPG key for the current user
 * 		_, err = gitlab.NewUserGpgKey(ctx, "example_user", &gitlab.UserGpgKeyArgs{
 * 			Key: pulumi.String("-----BEGIN PGP PUBLIC KEY BLOCK-----\n...\n-----END PGP PUBLIC KEY BLOCK-----"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.GitlabFunctions;
 * import com.pulumi.gitlab.inputs.GetUserArgs;
 * import com.pulumi.gitlab.UserGpgKey;
 * import com.pulumi.gitlab.UserGpgKeyArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var example = GitlabFunctions.getUser(GetUserArgs.builder()
 *             .username("example-user")
 *             .build());
 *         // Manages a GPG key for the specified user. An admin token is required if `user_id` is specified.
 *         var exampleUserGpgKey = new UserGpgKey("exampleUserGpgKey", UserGpgKeyArgs.builder()
 *             .userId(example.applyValue(getUserResult -> getUserResult.id()))
 *             .key("""
 * -----BEGIN PGP PUBLIC KEY BLOCK-----
 * ...
 * -----END PGP PUBLIC KEY BLOCK-----            """)
 *             .build());
 *         // Manages a GPG key for the current user
 *         var exampleUser = new UserGpgKey("exampleUser", UserGpgKeyArgs.builder()
 *             .key("""
 * -----BEGIN PGP PUBLIC KEY BLOCK-----
 * ...
 * -----END PGP PUBLIC KEY BLOCK-----            """)
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   # Manages a GPG key for the specified user. An admin token is required if `user_id` is specified.
 *   exampleUserGpgKey:
 *     type: gitlab:UserGpgKey
 *     name: example
 *     properties:
 *       userId: ${example.id}
 *       key: |-
 *         -----BEGIN PGP PUBLIC KEY BLOCK-----
 *         ...
 *         -----END PGP PUBLIC KEY BLOCK-----
 *   # Manages a GPG key for the current user
 *   exampleUser:
 *     type: gitlab:UserGpgKey
 *     name: example_user
 *     properties:
 *       key: |-
 *         -----BEGIN PGP PUBLIC KEY BLOCK-----
 *         ...
 *         -----END PGP PUBLIC KEY BLOCK-----
 * variables:
 *   example:
 *     fn::invoke:
 *       Function: gitlab:getUser
 *       Arguments:
 *         username: example-user
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * You can import a GPG key for a specific user using an id made up of `{user-id}:{key}`, e.g.
 * ```sh
 * $ pulumi import gitlab:index/userGpgKey:UserGpgKey example 42:1
 * ```
 * Alternatively, you can import a GPG key for the current user using an id made up of `{key}`, e.g.
 * ```sh
 * $ pulumi import gitlab:index/userGpgKey:UserGpgKey example_user 1
 * ```
 * @property key The armored GPG public key.
 * @property userId The ID of the user to add the GPG key to. If this field is omitted, this resource manages a GPG key for the current user. Otherwise, this resource manages a GPG key for the specified user, and an admin token is required.
 */
public data class UserGpgKeyArgs(
    public val key: Output<String>? = null,
    public val userId: Output<Int>? = null,
) : ConvertibleToJava<com.pulumi.gitlab.UserGpgKeyArgs> {
    override fun toJava(): com.pulumi.gitlab.UserGpgKeyArgs =
        com.pulumi.gitlab.UserGpgKeyArgs.builder()
            .key(key?.applyValue({ args0 -> args0 }))
            .userId(userId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [UserGpgKeyArgs].
 */
@PulumiTagMarker
public class UserGpgKeyArgsBuilder internal constructor() {
    private var key: Output<String>? = null

    private var userId: Output<Int>? = null

    /**
     * @param value The armored GPG public key.
     */
    @JvmName("prqayifkfbsynbje")
    public suspend fun key(`value`: Output<String>) {
        this.key = value
    }

    /**
     * @param value The ID of the user to add the GPG key to. If this field is omitted, this resource manages a GPG key for the current user. Otherwise, this resource manages a GPG key for the specified user, and an admin token is required.
     */
    @JvmName("vdofwpfqtifjvioh")
    public suspend fun userId(`value`: Output<Int>) {
        this.userId = value
    }

    /**
     * @param value The armored GPG public key.
     */
    @JvmName("qlulabaxeafrasns")
    public suspend fun key(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.key = mapped
    }

    /**
     * @param value The ID of the user to add the GPG key to. If this field is omitted, this resource manages a GPG key for the current user. Otherwise, this resource manages a GPG key for the specified user, and an admin token is required.
     */
    @JvmName("rnbrechpiwgyqnhf")
    public suspend fun userId(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.userId = mapped
    }

    internal fun build(): UserGpgKeyArgs = UserGpgKeyArgs(
        key = key,
        userId = userId,
    )
}
