@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.gitlab.UserRunnerArgs.builder
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlin.jvm.JvmName

/**
 * The `gitlab.UserRunner` resource allows creating a GitLab runner using the new [GitLab Runner Registration Flow](https://docs.gitlab.com/ee/ci/runners/new_creation_workflow.html).
 * **Upstream API**: [GitLab REST API docs](https://docs.gitlab.com/ee/api/users.html#create-a-runner)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * // Create a project runner
 * const projectRunner = new gitlab.UserRunner("project_runner", {
 *     runnerType: "project_type",
 *     projectId: 123456,
 *     description: "A runner created using a user access token instead of a registration token",
 *     tagLists: [
 *         "a-tag",
 *         "other-tag",
 *     ],
 *     untagged: true,
 * });
 * // Create a group runner
 * const groupRunner = new gitlab.UserRunner("group_runner", {
 *     runnerType: "group_type",
 *     groupId: 123456,
 * });
 * // Create a instance runner
 * const instanceRunner = new gitlab.UserRunner("instance_runner", {runnerType: "instance_type"});
 * const configToml = pulumi.interpolate`concurrent = 1
 * check_interval = 0
 * [session_server]
 *   session_timeout = 1800
 * [[runners]]
 *   name = "my_gitlab_runner"
 *   url = "https://example.gitlab.com"
 *   token = "${groupRunner.token}"
 *   executor = "docker"
 *   [runners&#46;custom_build_dir]
 *   [runners&#46;cache]
 *     [runners&#46;cache&#46;s3]
 *     [runners&#46;cache&#46;gcs]
 *     [runners&#46;cache&#46;azure]
 *   [runners&#46;docker]
 *     tls_verify = false
 *     image = "ubuntu"
 *     privileged = true
 *     disable_entrypoint_overwrite = false
 *     oom_kill_disable = false
 *     disable_cache = false
 *     volumes = ["/cache", "/certs/client"]
 *     shm_size = 0
 * `;
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * # Create a project runner
 * project_runner = gitlab.UserRunner("project_runner",
 *     runner_type="project_type",
 *     project_id=123456,
 *     description="A runner created using a user access token instead of a registration token",
 *     tag_lists=[
 *         "a-tag",
 *         "other-tag",
 *     ],
 *     untagged=True)
 * # Create a group runner
 * group_runner = gitlab.UserRunner("group_runner",
 *     runner_type="group_type",
 *     group_id=123456)
 * # Create a instance runner
 * instance_runner = gitlab.UserRunner("instance_runner", runner_type="instance_type")
 * config_toml = group_runner.token.apply(lambda token: f"""concurrent = 1
 * check_interval = 0
 * [session_server]
 *   session_timeout = 1800
 * [[runners]]
 *   name = "my_gitlab_runner"
 *   url = "https://example.gitlab.com"
 *   token = "{token}"
 *   executor = "docker"
 *   [runners&#46;custom_build_dir]
 *   [runners&#46;cache]
 *     [runners&#46;cache&#46;s3]
 *     [runners&#46;cache&#46;gcs]
 *     [runners&#46;cache&#46;azure]
 *   [runners&#46;docker]
 *     tls_verify = false
 *     image = "ubuntu"
 *     privileged = true
 *     disable_entrypoint_overwrite = false
 *     oom_kill_disable = false
 *     disable_cache = false
 *     volumes = ["/cache", "/certs/client"]
 *     shm_size = 0
 * """)
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     // Create a project runner
 *     var projectRunner = new GitLab.UserRunner("project_runner", new()
 *     {
 *         RunnerType = "project_type",
 *         ProjectId = 123456,
 *         Description = "A runner created using a user access token instead of a registration token",
 *         TagLists = new[]
 *         {
 *             "a-tag",
 *             "other-tag",
 *         },
 *         Untagged = true,
 *     });
 *     // Create a group runner
 *     var groupRunner = new GitLab.UserRunner("group_runner", new()
 *     {
 *         RunnerType = "group_type",
 *         GroupId = 123456,
 *     });
 *     // Create a instance runner
 *     var instanceRunner = new GitLab.UserRunner("instance_runner", new()
 *     {
 *         RunnerType = "instance_type",
 *     });
 *     var configToml = groupRunner.Token.Apply(token => @$"concurrent = 1
 * check_interval = 0
 * [session_server]
 *   session_timeout = 1800
 * [[runners]]
 *   name = ""my_gitlab_runner""
 *   url = ""https://example.gitlab.com""
 *   token = ""{token}""
 *   executor = ""docker""
 *   [runners&#46;custom_build_dir]
 *   [runners&#46;cache]
 *     [runners&#46;cache&#46;s3]
 *     [runners&#46;cache&#46;gcs]
 *     [runners&#46;cache&#46;azure]
 *   [runners&#46;docker]
 *     tls_verify = false
 *     image = ""ubuntu""
 *     privileged = true
 *     disable_entrypoint_overwrite = false
 *     oom_kill_disable = false
 *     disable_cache = false
 *     volumes = [""/cache"", ""/certs/client""]
 *     shm_size = 0
 * ");
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"fmt"
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		// Create a project runner
 * 		_, err := gitlab.NewUserRunner(ctx, "project_runner", &gitlab.UserRunnerArgs{
 * 			RunnerType:  pulumi.String("project_type"),
 * 			ProjectId:   pulumi.Int(123456),
 * 			Description: pulumi.String("A runner created using a user access token instead of a registration token"),
 * 			TagLists: pulumi.StringArray{
 * 				pulumi.String("a-tag"),
 * 				pulumi.String("other-tag"),
 * 			},
 * 			Untagged: pulumi.Bool(true),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Create a group runner
 * 		groupRunner, err := gitlab.NewUserRunner(ctx, "group_runner", &gitlab.UserRunnerArgs{
 * 			RunnerType: pulumi.String("group_type"),
 * 			GroupId:    pulumi.Int(123456),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		// Create a instance runner
 * 		_, err = gitlab.NewUserRunner(ctx, "instance_runner", &gitlab.UserRunnerArgs{
 * 			RunnerType: pulumi.String("instance_type"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_ = groupRunner.Token.ApplyT(func(token string) (string, error) {
 * 			return fmt.Sprintf(`concurrent = 1
 * check_interval = 0
 * [session_server]
 *   session_timeout = 1800
 * [[runners]]
 *   name = "my_gitlab_runner"
 *   url = "https://example.gitlab.com"
 *   token = "%v"
 *   executor = "docker"
 *   [runners&#46;custom_build_dir]
 *   [runners&#46;cache]
 *     [runners&#46;cache&#46;s3]
 *     [runners&#46;cache&#46;gcs]
 *     [runners&#46;cache&#46;azure]
 *   [runners&#46;docker]
 *     tls_verify = false
 *     image = "ubuntu"
 *     privileged = true
 *     disable_entrypoint_overwrite = false
 *     oom_kill_disable = false
 *     disable_cache = false
 *     volumes = ["/cache", "/certs/client"]
 *     shm_size = 0
 * `, token), nil
 * 		}).(pulumi.StringOutput)
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.UserRunner;
 * import com.pulumi.gitlab.UserRunnerArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         // Create a project runner
 *         var projectRunner = new UserRunner("projectRunner", UserRunnerArgs.builder()
 *             .runnerType("project_type")
 *             .projectId(123456)
 *             .description("A runner created using a user access token instead of a registration token")
 *             .tagLists(
 *                 "a-tag",
 *                 "other-tag")
 *             .untagged(true)
 *             .build());
 *         // Create a group runner
 *         var groupRunner = new UserRunner("groupRunner", UserRunnerArgs.builder()
 *             .runnerType("group_type")
 *             .groupId(123456)
 *             .build());
 *         // Create a instance runner
 *         var instanceRunner = new UserRunner("instanceRunner", UserRunnerArgs.builder()
 *             .runnerType("instance_type")
 *             .build());
 *         final var configToml = groupRunner.token().applyValue(token -> """
 * concurrent = 1
 * check_interval = 0
 * [session_server]
 *   session_timeout = 1800
 * [[runners]]
 *   name = "my_gitlab_runner"
 *   url = "https://example.gitlab.com"
 *   token = "%s"
 *   executor = "docker"
 *   [runners&#46;custom_build_dir]
 *   [runners&#46;cache]
 *     [runners&#46;cache&#46;s3]
 *     [runners&#46;cache&#46;gcs]
 *     [runners&#46;cache&#46;azure]
 *   [runners&#46;docker]
 *     tls_verify = false
 *     image = "ubuntu"
 *     privileged = true
 *     disable_entrypoint_overwrite = false
 *     oom_kill_disable = false
 *     disable_cache = false
 *     volumes = ["/cache", "/certs/client"]
 *     shm_size = 0
 * ", token));
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   # Create a project runner
 *   projectRunner:
 *     type: gitlab:UserRunner
 *     name: project_runner
 *     properties:
 *       runnerType: project_type
 *       projectId: 123456
 *       description: A runner created using a user access token instead of a registration token
 *       tagLists:
 *         - a-tag
 *         - other-tag
 *       untagged: true
 *   # Create a group runner
 *   groupRunner:
 *     type: gitlab:UserRunner
 *     name: group_runner
 *     properties:
 *       runnerType: group_type
 *       groupId: 123456 # populate other attributes...
 *   # Create a instance runner
 *   instanceRunner:
 *     type: gitlab:UserRunner
 *     name: instance_runner
 *     properties:
 *       runnerType: instance_type
 * variables:
 *   configToml: |
 *     concurrent = 1
 *     check_interval = 0
 *     [session_server]
 *       session_timeout = 1800
 *     [[runners]]
 *       name = "my_gitlab_runner"
 *       url = "https://example.gitlab.com"
 *       token = "${groupRunner.token}"
 *       executor = "docker"
 *       [runners&#46;custom_build_dir]
 *       [runners&#46;cache]
 *         [runners&#46;cache&#46;s3]
 *         [runners&#46;cache&#46;gcs]
 *         [runners&#46;cache&#46;azure]
 *       [runners&#46;docker]
 *         tls_verify = false
 *         image = "ubuntu"
 *         privileged = true
 *         disable_entrypoint_overwrite = false
 *         oom_kill_disable = false
 *         disable_cache = false
 *         volumes = ["/cache", "/certs/client"]
 *         shm_size = 0
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * You can import a gitlab runner using its ID
 * Note: Importing a runner will not provide access to the `token` attribute
 * ```sh
 * $ pulumi import gitlab:index/userRunner:UserRunner example 12345
 * ```
 * @property accessLevel The access level of the runner. Valid values are: `not_protected`, `ref_protected`.
 * @property description Description of the runner.
 * @property groupId The ID of the group that the runner is created in. Required if runner*type is group*type.
 * @property locked Specifies if the runner should be locked for the current project.
 * @property maximumTimeout Maximum timeout that limits the amount of time (in seconds) that runners can run jobs. Must be at least 600 (10 minutes).
 * @property paused Specifies if the runner should ignore new jobs.
 * @property projectId The ID of the project that the runner is created in. Required if runner*type is project*type.
 * @property runnerType The scope of the runner. Valid values are: `instance_type`, `group_type`, `project_type`.
 * @property tagLists A list of runner tags.
 * @property untagged Specifies if the runner should handle untagged jobs.
 */
public data class UserRunnerArgs(
    public val accessLevel: Output<String>? = null,
    public val description: Output<String>? = null,
    public val groupId: Output<Int>? = null,
    public val locked: Output<Boolean>? = null,
    public val maximumTimeout: Output<Int>? = null,
    public val paused: Output<Boolean>? = null,
    public val projectId: Output<Int>? = null,
    public val runnerType: Output<String>? = null,
    public val tagLists: Output<List<String>>? = null,
    public val untagged: Output<Boolean>? = null,
) : ConvertibleToJava<com.pulumi.gitlab.UserRunnerArgs> {
    override fun toJava(): com.pulumi.gitlab.UserRunnerArgs =
        com.pulumi.gitlab.UserRunnerArgs.builder()
            .accessLevel(accessLevel?.applyValue({ args0 -> args0 }))
            .description(description?.applyValue({ args0 -> args0 }))
            .groupId(groupId?.applyValue({ args0 -> args0 }))
            .locked(locked?.applyValue({ args0 -> args0 }))
            .maximumTimeout(maximumTimeout?.applyValue({ args0 -> args0 }))
            .paused(paused?.applyValue({ args0 -> args0 }))
            .projectId(projectId?.applyValue({ args0 -> args0 }))
            .runnerType(runnerType?.applyValue({ args0 -> args0 }))
            .tagLists(tagLists?.applyValue({ args0 -> args0.map({ args0 -> args0 }) }))
            .untagged(untagged?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [UserRunnerArgs].
 */
@PulumiTagMarker
public class UserRunnerArgsBuilder internal constructor() {
    private var accessLevel: Output<String>? = null

    private var description: Output<String>? = null

    private var groupId: Output<Int>? = null

    private var locked: Output<Boolean>? = null

    private var maximumTimeout: Output<Int>? = null

    private var paused: Output<Boolean>? = null

    private var projectId: Output<Int>? = null

    private var runnerType: Output<String>? = null

    private var tagLists: Output<List<String>>? = null

    private var untagged: Output<Boolean>? = null

    /**
     * @param value The access level of the runner. Valid values are: `not_protected`, `ref_protected`.
     */
    @JvmName("tqgswcckllhuenix")
    public suspend fun accessLevel(`value`: Output<String>) {
        this.accessLevel = value
    }

    /**
     * @param value Description of the runner.
     */
    @JvmName("sumosgmoqsbcwsch")
    public suspend fun description(`value`: Output<String>) {
        this.description = value
    }

    /**
     * @param value The ID of the group that the runner is created in. Required if runner*type is group*type.
     */
    @JvmName("yuwvlkokbhjastxi")
    public suspend fun groupId(`value`: Output<Int>) {
        this.groupId = value
    }

    /**
     * @param value Specifies if the runner should be locked for the current project.
     */
    @JvmName("pklcsocujuhgvvxj")
    public suspend fun locked(`value`: Output<Boolean>) {
        this.locked = value
    }

    /**
     * @param value Maximum timeout that limits the amount of time (in seconds) that runners can run jobs. Must be at least 600 (10 minutes).
     */
    @JvmName("xtehsokeucstobev")
    public suspend fun maximumTimeout(`value`: Output<Int>) {
        this.maximumTimeout = value
    }

    /**
     * @param value Specifies if the runner should ignore new jobs.
     */
    @JvmName("kvsrrjkqmllaoohg")
    public suspend fun paused(`value`: Output<Boolean>) {
        this.paused = value
    }

    /**
     * @param value The ID of the project that the runner is created in. Required if runner*type is project*type.
     */
    @JvmName("ukeilekpmaogokub")
    public suspend fun projectId(`value`: Output<Int>) {
        this.projectId = value
    }

    /**
     * @param value The scope of the runner. Valid values are: `instance_type`, `group_type`, `project_type`.
     */
    @JvmName("ukgdjwynulsauerd")
    public suspend fun runnerType(`value`: Output<String>) {
        this.runnerType = value
    }

    /**
     * @param value A list of runner tags.
     */
    @JvmName("whlnqxvxmcqvpsap")
    public suspend fun tagLists(`value`: Output<List<String>>) {
        this.tagLists = value
    }

    @JvmName("ycarcuvewueovowp")
    public suspend fun tagLists(vararg values: Output<String>) {
        this.tagLists = Output.all(values.asList())
    }

    /**
     * @param values A list of runner tags.
     */
    @JvmName("bwaacpmyhmxshskv")
    public suspend fun tagLists(values: List<Output<String>>) {
        this.tagLists = Output.all(values)
    }

    /**
     * @param value Specifies if the runner should handle untagged jobs.
     */
    @JvmName("jsmolkmipawstahs")
    public suspend fun untagged(`value`: Output<Boolean>) {
        this.untagged = value
    }

    /**
     * @param value The access level of the runner. Valid values are: `not_protected`, `ref_protected`.
     */
    @JvmName("evjgdnnrfuydwddr")
    public suspend fun accessLevel(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.accessLevel = mapped
    }

    /**
     * @param value Description of the runner.
     */
    @JvmName("iojbkqfrynneblch")
    public suspend fun description(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.description = mapped
    }

    /**
     * @param value The ID of the group that the runner is created in. Required if runner*type is group*type.
     */
    @JvmName("vlcqjelujtpcsimm")
    public suspend fun groupId(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.groupId = mapped
    }

    /**
     * @param value Specifies if the runner should be locked for the current project.
     */
    @JvmName("wgdlprsmafmgfluy")
    public suspend fun locked(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.locked = mapped
    }

    /**
     * @param value Maximum timeout that limits the amount of time (in seconds) that runners can run jobs. Must be at least 600 (10 minutes).
     */
    @JvmName("tirrmnvtbfcwqwki")
    public suspend fun maximumTimeout(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maximumTimeout = mapped
    }

    /**
     * @param value Specifies if the runner should ignore new jobs.
     */
    @JvmName("snirrqmxewlccbqf")
    public suspend fun paused(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.paused = mapped
    }

    /**
     * @param value The ID of the project that the runner is created in. Required if runner*type is project*type.
     */
    @JvmName("obtrxuvmknnxxepr")
    public suspend fun projectId(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.projectId = mapped
    }

    /**
     * @param value The scope of the runner. Valid values are: `instance_type`, `group_type`, `project_type`.
     */
    @JvmName("tssauriubiokqwvx")
    public suspend fun runnerType(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.runnerType = mapped
    }

    /**
     * @param value A list of runner tags.
     */
    @JvmName("skobkhfwftedpurd")
    public suspend fun tagLists(`value`: List<String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tagLists = mapped
    }

    /**
     * @param values A list of runner tags.
     */
    @JvmName("dlvtyuxqixbionvc")
    public suspend fun tagLists(vararg values: String) {
        val toBeMapped = values.toList()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tagLists = mapped
    }

    /**
     * @param value Specifies if the runner should handle untagged jobs.
     */
    @JvmName("qvtqwwpfsscrrbma")
    public suspend fun untagged(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.untagged = mapped
    }

    internal fun build(): UserRunnerArgs = UserRunnerArgs(
        accessLevel = accessLevel,
        description = description,
        groupId = groupId,
        locked = locked,
        maximumTimeout = maximumTimeout,
        paused = paused,
        projectId = projectId,
        runnerType = runnerType,
        tagLists = tagLists,
        untagged = untagged,
    )
}
