@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.gitlab.kotlin

import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.gitlab.UserSshKeyArgs.builder
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.jvm.JvmName

/**
 * The `gitlab.UserSshKey` resource allows to manage the lifecycle of an SSH key assigned to a user.
 * **Upstream API**: [GitLab API docs](https://docs.gitlab.com/ee/api/users.html#single-ssh-key)
 * ## Example Usage
 * <!--Start PulumiCodeChooser -->
 * ```typescript
 * import * as pulumi from "@pulumi/pulumi";
 * import * as gitlab from "@pulumi/gitlab";
 * const example = gitlab.getUser({
 *     username: "example-user",
 * });
 * const exampleUserSshKey = new gitlab.UserSshKey("example", {
 *     userId: example.then(example => example.id),
 *     title: "example-key",
 *     key: "ssh-ed25519 AAAA...",
 *     expiresAt: "2016-01-21T00:00:00.000Z",
 * });
 * ```
 * ```python
 * import pulumi
 * import pulumi_gitlab as gitlab
 * example = gitlab.get_user(username="example-user")
 * example_user_ssh_key = gitlab.UserSshKey("example",
 *     user_id=example.id,
 *     title="example-key",
 *     key="ssh-ed25519 AAAA...",
 *     expires_at="2016-01-21T00:00:00.000Z")
 * ```
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using GitLab = Pulumi.GitLab;
 * return await Deployment.RunAsync(() =>
 * {
 *     var example = GitLab.GetUser.Invoke(new()
 *     {
 *         Username = "example-user",
 *     });
 *     var exampleUserSshKey = new GitLab.UserSshKey("example", new()
 *     {
 *         UserId = example.Apply(getUserResult => getUserResult.Id),
 *         Title = "example-key",
 *         Key = "ssh-ed25519 AAAA...",
 *         ExpiresAt = "2016-01-21T00:00:00.000Z",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-gitlab/sdk/v8/go/gitlab"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		example, err := gitlab.LookupUser(ctx, &gitlab.LookupUserArgs{
 * 			Username: pulumi.StringRef("example-user"),
 * 		}, nil)
 * 		if err != nil {
 * 			return err
 * 		}
 * 		_, err = gitlab.NewUserSshKey(ctx, "example", &gitlab.UserSshKeyArgs{
 * 			UserId:    pulumi.String(example.Id),
 * 			Title:     pulumi.String("example-key"),
 * 			Key:       pulumi.String("ssh-ed25519 AAAA..."),
 * 			ExpiresAt: pulumi.String("2016-01-21T00:00:00.000Z"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.gitlab.GitlabFunctions;
 * import com.pulumi.gitlab.inputs.GetUserArgs;
 * import com.pulumi.gitlab.UserSshKey;
 * import com.pulumi.gitlab.UserSshKeyArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         final var example = GitlabFunctions.getUser(GetUserArgs.builder()
 *             .username("example-user")
 *             .build());
 *         var exampleUserSshKey = new UserSshKey("exampleUserSshKey", UserSshKeyArgs.builder()
 *             .userId(example.applyValue(getUserResult -> getUserResult.id()))
 *             .title("example-key")
 *             .key("ssh-ed25519 AAAA...")
 *             .expiresAt("2016-01-21T00:00:00.000Z")
 *             .build());
 *     }
 * }
 * ```
 * ```yaml
 * resources:
 *   exampleUserSshKey:
 *     type: gitlab:UserSshKey
 *     name: example
 *     properties:
 *       userId: ${example.id}
 *       title: example-key
 *       key: ssh-ed25519 AAAA...
 *       expiresAt: 2016-01-21T00:00:00.000Z
 * variables:
 *   example:
 *     fn::invoke:
 *       Function: gitlab:getUser
 *       Arguments:
 *         username: example-user
 * ```
 * <!--End PulumiCodeChooser -->
 * ## Import
 * You can import a user ssh key using an id made up of `{user-id}:{key}`, e.g.
 * ```sh
 * $ pulumi import gitlab:index/userSshKey:UserSshKey example 42:1
 * ```
 * @property expiresAt The expiration date of the SSH key in ISO 8601 format (YYYY-MM-DDTHH:MM:SSZ)
 * @property key The ssh key. The SSH key `comment` (trailing part) is optional and ignored for diffing, because GitLab overrides it with the username and GitLab hostname.
 * @property title The title of the ssh key.
 * @property userId The ID or username of the user. If this field is omitted, this resource manages a SSH key for the current user. Otherwise, this resource manages a SSH key for the specified user, and an admin token is required.
 */
public data class UserSshKeyArgs(
    public val expiresAt: Output<String>? = null,
    public val key: Output<String>? = null,
    public val title: Output<String>? = null,
    public val userId: Output<Int>? = null,
) : ConvertibleToJava<com.pulumi.gitlab.UserSshKeyArgs> {
    override fun toJava(): com.pulumi.gitlab.UserSshKeyArgs =
        com.pulumi.gitlab.UserSshKeyArgs.builder()
            .expiresAt(expiresAt?.applyValue({ args0 -> args0 }))
            .key(key?.applyValue({ args0 -> args0 }))
            .title(title?.applyValue({ args0 -> args0 }))
            .userId(userId?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [UserSshKeyArgs].
 */
@PulumiTagMarker
public class UserSshKeyArgsBuilder internal constructor() {
    private var expiresAt: Output<String>? = null

    private var key: Output<String>? = null

    private var title: Output<String>? = null

    private var userId: Output<Int>? = null

    /**
     * @param value The expiration date of the SSH key in ISO 8601 format (YYYY-MM-DDTHH:MM:SSZ)
     */
    @JvmName("hnfwkhjqgumlcgxs")
    public suspend fun expiresAt(`value`: Output<String>) {
        this.expiresAt = value
    }

    /**
     * @param value The ssh key. The SSH key `comment` (trailing part) is optional and ignored for diffing, because GitLab overrides it with the username and GitLab hostname.
     */
    @JvmName("cnwmfvyfsvkbivxo")
    public suspend fun key(`value`: Output<String>) {
        this.key = value
    }

    /**
     * @param value The title of the ssh key.
     */
    @JvmName("xvbubyjaglcoquue")
    public suspend fun title(`value`: Output<String>) {
        this.title = value
    }

    /**
     * @param value The ID or username of the user. If this field is omitted, this resource manages a SSH key for the current user. Otherwise, this resource manages a SSH key for the specified user, and an admin token is required.
     */
    @JvmName("tpttxmeaflearbpj")
    public suspend fun userId(`value`: Output<Int>) {
        this.userId = value
    }

    /**
     * @param value The expiration date of the SSH key in ISO 8601 format (YYYY-MM-DDTHH:MM:SSZ)
     */
    @JvmName("oxjaqxjrgsivgfem")
    public suspend fun expiresAt(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.expiresAt = mapped
    }

    /**
     * @param value The ssh key. The SSH key `comment` (trailing part) is optional and ignored for diffing, because GitLab overrides it with the username and GitLab hostname.
     */
    @JvmName("hpmtucwoxhrnlmyp")
    public suspend fun key(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.key = mapped
    }

    /**
     * @param value The title of the ssh key.
     */
    @JvmName("domvfryqxjoepdfc")
    public suspend fun title(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.title = mapped
    }

    /**
     * @param value The ID or username of the user. If this field is omitted, this resource manages a SSH key for the current user. Otherwise, this resource manages a SSH key for the specified user, and an admin token is required.
     */
    @JvmName("vgavhupoydwlkjbb")
    public suspend fun userId(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.userId = mapped
    }

    internal fun build(): UserSshKeyArgs = UserSshKeyArgs(
        expiresAt = expiresAt,
        key = key,
        title = title,
        userId = userId,
    )
}
