/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.faulttolerance.core.retry;

import io.smallrye.faulttolerance.core.Completer;
import io.smallrye.faulttolerance.core.FailureContext;
import io.smallrye.faulttolerance.core.FaultToleranceContext;
import io.smallrye.faulttolerance.core.FaultToleranceStrategy;
import io.smallrye.faulttolerance.core.Future;
import io.smallrye.faulttolerance.core.retry.AsyncDelay;
import io.smallrye.faulttolerance.core.retry.RetryEvents;
import io.smallrye.faulttolerance.core.retry.RetryLogger;
import io.smallrye.faulttolerance.core.retry.SyncDelay;
import io.smallrye.faulttolerance.core.retry.SyncDelayAsAsync;
import io.smallrye.faulttolerance.core.stopwatch.RunningStopwatch;
import io.smallrye.faulttolerance.core.stopwatch.Stopwatch;
import io.smallrye.faulttolerance.core.util.ExceptionDecision;
import io.smallrye.faulttolerance.core.util.Preconditions;
import io.smallrye.faulttolerance.core.util.ResultDecision;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.eclipse.microprofile.faulttolerance.exceptions.FaultToleranceException;

public class Retry<V>
implements FaultToleranceStrategy<V> {
    private final FaultToleranceStrategy<V> delegate;
    private final String description;
    private final ResultDecision resultDecision;
    private final ExceptionDecision exceptionDecision;
    private final long maxRetries;
    private final long maxTotalDurationInMillis;
    private final Supplier<SyncDelay> syncDelayBetweenRetries;
    private final Supplier<AsyncDelay> asyncDelayBetweenRetries;
    private final Stopwatch stopwatch;
    private final Consumer<FailureContext> beforeRetry;

    public Retry(FaultToleranceStrategy<V> delegate, String description, ResultDecision resultDecision, ExceptionDecision exceptionDecision, long maxRetries, long maxTotalDurationInMillis, Supplier<SyncDelay> syncDelayBetweenRetries, Supplier<AsyncDelay> asyncDelayBetweenRetries, Stopwatch stopwatch, Consumer<FailureContext> beforeRetry) {
        this.delegate = Preconditions.checkNotNull(delegate, "Retry delegate must be set");
        this.description = Preconditions.checkNotNull(description, "Retry description must be set");
        this.resultDecision = Preconditions.checkNotNull(resultDecision, "Result decision must be set");
        this.exceptionDecision = Preconditions.checkNotNull(exceptionDecision, "Exception decision must be set");
        this.maxRetries = maxRetries < 0L ? Long.MAX_VALUE : maxRetries;
        this.maxTotalDurationInMillis = maxTotalDurationInMillis <= 0L ? Long.MAX_VALUE : maxTotalDurationInMillis;
        this.syncDelayBetweenRetries = Preconditions.checkNotNull(syncDelayBetweenRetries, "Synchronous delay must be set");
        this.asyncDelayBetweenRetries = Preconditions.checkNotNull(asyncDelayBetweenRetries, "Asynchronous delay must be set");
        this.stopwatch = Preconditions.checkNotNull(stopwatch, "Stopwatch must be set");
        this.beforeRetry = beforeRetry;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Future<V> apply(FaultToleranceContext<V> ctx) {
        RetryLogger.LOG.trace("Retry started");
        try {
            AsyncDelay delay = ctx.isAsync() ? this.asyncDelayBetweenRetries.get() : new SyncDelayAsAsync(this.syncDelayBetweenRetries.get());
            RunningStopwatch runningStopwatch = this.stopwatch.start();
            Future<V> future = this.doRetry(ctx, 0, delay, runningStopwatch, null);
            return future;
        }
        finally {
            RetryLogger.LOG.trace("Retry finished");
        }
    }

    private Future<V> doRetry(FaultToleranceContext<V> ctx, int attempt, AsyncDelay delay, RunningStopwatch stopwatch, Throwable lastFailure) {
        if (attempt == 0) {
            return this.afterDelay(ctx, attempt, delay, stopwatch, lastFailure);
        }
        if ((long)attempt <= this.maxRetries) {
            if (stopwatch.elapsedTimeInMillis() >= this.maxTotalDurationInMillis) {
                ctx.fireEvent(RetryEvents.Finished.MAX_DURATION_REACHED);
                if (lastFailure != null) {
                    return Future.ofError(lastFailure);
                }
                return Future.ofError((Throwable)new FaultToleranceException(this.description + " reached max retry duration"));
            }
            RetryLogger.LOG.debugf("%s invocation failed, retrying (%d/%d)", this.description, attempt, this.maxRetries);
            ctx.fireEvent(RetryEvents.Retried.INSTANCE);
            Completer result = Completer.create();
            try {
                delay.after(lastFailure, () -> this.afterDelay(ctx, attempt, delay, stopwatch, lastFailure).thenComplete(result), ctx.get(Executor.class));
            }
            catch (Exception e) {
                if (ctx.isSync() && Thread.interrupted()) {
                    result.completeWithError(new InterruptedException());
                }
                result.completeWithError(e);
            }
            return result.future();
        }
        ctx.fireEvent(RetryEvents.Finished.MAX_RETRIES_REACHED);
        if (lastFailure != null) {
            return Future.ofError(lastFailure);
        }
        return Future.ofError((Throwable)new FaultToleranceException(this.description + " reached max retries"));
    }

    private Future<V> afterDelay(FaultToleranceContext<V> ctx, int attempt, AsyncDelay delay, RunningStopwatch stopwatch, Throwable lastFailure) {
        if (stopwatch.elapsedTimeInMillis() >= this.maxTotalDurationInMillis) {
            ctx.fireEvent(RetryEvents.Finished.MAX_DURATION_REACHED);
            if (lastFailure != null) {
                return Future.ofError(lastFailure);
            }
            return Future.ofError((Throwable)new FaultToleranceException(this.description + " reached max retry duration"));
        }
        if (this.beforeRetry != null && attempt > 0) {
            try {
                this.beforeRetry.accept(new FailureContext(lastFailure, ctx));
            }
            catch (Exception e) {
                RetryLogger.LOG.warn("Before retry action has thrown an exception", e);
            }
        }
        Completer result = Completer.create();
        try {
            this.delegate.apply(ctx).then((value, error) -> {
                if (ctx.isSync()) {
                    if (error instanceof InterruptedException) {
                        ctx.fireEvent(RetryEvents.Finished.EXCEPTION_NOT_RETRYABLE);
                        result.completeWithError((Throwable)error);
                        return;
                    }
                    if (Thread.interrupted()) {
                        ctx.fireEvent(RetryEvents.Finished.EXCEPTION_NOT_RETRYABLE);
                        result.completeWithError(new InterruptedException());
                        return;
                    }
                }
                if (error == null) {
                    if (this.resultDecision.isConsideredExpected(value)) {
                        ctx.fireEvent(RetryEvents.Finished.VALUE_RETURNED);
                        result.complete(value);
                    } else {
                        this.doRetry(ctx, attempt + 1, delay, stopwatch, (Throwable)error).thenComplete(result);
                    }
                } else if (this.exceptionDecision.isConsideredExpected((Throwable)error)) {
                    ctx.fireEvent(RetryEvents.Finished.EXCEPTION_NOT_RETRYABLE);
                    result.completeWithError((Throwable)error);
                } else {
                    this.doRetry(ctx, attempt + 1, delay, stopwatch, (Throwable)error).thenComplete(result);
                }
            });
        }
        catch (Throwable e) {
            if (this.exceptionDecision.isConsideredExpected(e)) {
                ctx.fireEvent(RetryEvents.Finished.EXCEPTION_NOT_RETRYABLE);
                result.completeWithError(e);
            }
            this.doRetry(ctx, attempt + 1, delay, stopwatch, e).thenComplete(result);
        }
        return result.future();
    }
}

