/**
 * Copyright (c) 2019, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 */
import { AxiosError, AxiosRequestConfig, AxiosResponse } from "axios";
import { AxiosHttpClientInstance, HttpClient } from "../models";
/**
 * An Axios Http client to perform Http requests.
 *
 * @remarks
 * Typescript doesn't support static functions in interfaces. Therefore,
 * a decorator i.e `staticDecorator` was written to add static support.
 * Follow {@link https://github.com/Microsoft/TypeScript/issues/13462}
 * for more info.
 *
 * @example
 * Example usage.
 * ```
 * import { AxiosHttpClient } from "@wso2is/http";
 *
 * const httpClient = AxiosHttpClient.getInstance();
 * httpClient.init(true, onRequestStart, onRequestSuccess, onRequestError, onRequestFinish);
 * ```
 */
export declare class AxiosHttpClient implements HttpClient<AxiosRequestConfig, AxiosResponse, AxiosError> {
    private static axiosInstance;
    private static clientInstance;
    private static isHandlerEnabled;
    private requestStartCallback;
    private requestSuccessCallback;
    private requestErrorCallback;
    private requestFinishCallback;
    private static readonly DEFAULT_HANDLER_DISABLE_TIMEOUT;
    /**
     * Private constructor to avoid object instantiation from outside
     * the class.
     *
     * @hideconstructor
     */
    private constructor();
    /**
     * Returns an aggregated instance of type `AxiosInstance` of `AxiosHttpClient`.
     *
     * @return {any}
     */
    static getInstance(): AxiosHttpClientInstance;
    /**
     * Intercepts all the requests.
     * If the `isHandlerEnabled` flag is set to true, fires the `requestStartCallback`
     * and retrieves the access token from the server and attaches it to the request.
     * Else, just returns the original request.
     *
     * @param {AxiosRequestConfig} request - Original request.
     * @return {AxiosRequestConfig}
     */
    requestHandler(request: AxiosRequestConfig): AxiosRequestConfig;
    /**
     * Handles response errors.
     * If the `isHandlerEnabled` flag is set to true, fires the `requestErrorCallback`
     * and the `requestFinishCallback` functions. Else, just returns the original error.
     *
     * @param {AxiosError} error - Original error.
     * @return {AxiosError}
     */
    errorHandler(error: AxiosError): AxiosError;
    /**
     * Handles response success.
     * If the `isHandlerEnabled` flag is set to true, fires the `requestSuccessCallback`
     * and the `requestFinishCallback` functions. Else, just returns the original response.
     *
     * @param {AxiosResponse} response - Original response.
     * @return {AxiosResponse}
     */
    successHandler(response: AxiosResponse): AxiosResponse;
    /**
     * Initializes the Http client.
     *
     * @param isHandlerEnabled - Flag to toggle handler enablement.
     * @param requestStartCallback - Callback function to be triggered on request start.
     * @param requestSuccessCallback - Callback function to be triggered on request success.
     * @param requestErrorCallback - Callback function to be triggered on request error.
     * @param requestFinishCallback - Callback function to be triggered on request error.
     */
    init(isHandlerEnabled: boolean, requestStartCallback: () => void, requestSuccessCallback: (response: AxiosResponse) => void, requestErrorCallback: (error: AxiosError) => void, requestFinishCallback: () => void): void;
    /**
     * Enables the handler.
     */
    enableHandler(): void;
    /**
     * Disables the handler.
     */
    disableHandler(): void;
    /**
     * Disables the handler for a given period of time.
     *
     * @param {number} timeout - Timeout in milliseconds.
     */
    disableHandlerWithTimeout(timeout?: number): void;
}
