/**
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { AlertLevels } from "@wso2is/core/models";
import { ErrorIcon, InfoIcon, SuccessIcon, WarningIcon } from "@wso2is/theme";
import classNames from "classnames";
import React, { useEffect, useRef } from "react";
import NotificationSystem from "react-notification-system";
import { GenericIcon } from "../icon";
/**
 * Icons for different alert states.
 * @type {{success: any; warning: any; error: any; info: any}}
 */
const AlertIcons = {
    error: ErrorIcon,
    info: InfoIcon,
    success: SuccessIcon,
    warning: WarningIcon
};
/**
 * Alert component to show success, error, warning and info notifications on the front end dashboards.
 *
 * @param {AlertPropsInterface} props - Props injected in to the alert component.
 * @return {JSX.Element}
 */
export const Alert = (props) => {
    const { absolute, alert, alertsPosition, alertSystem, dismissible, dismissInterval, onAlertSystemInitialize, withIcon } = props;
    const classes = classNames({
        absolute
    }, "");
    const alertRef = useRef(null);
    useEffect(() => {
        onAlertSystemInitialize(alertRef.current);
    }, []);
    /**
     * Triggered when a new alert is available on the redux store.
     */
    useEffect(() => {
        if (!alertSystem || !alert) {
            return;
        }
        let icon = null;
        switch (alert.level) {
            case AlertLevels.SUCCESS: {
                icon = AlertIcons.success;
                break;
            }
            case AlertLevels.WARNING: {
                icon = AlertIcons.warning;
                break;
            }
            case AlertLevels.ERROR: {
                icon = AlertIcons.error;
                break;
            }
            case AlertLevels.INFO: {
                icon = AlertIcons.info;
                break;
            }
            default:
                break;
        }
        alertSystem.addNotification({
            autoDismiss: dismissInterval,
            dismissible,
            level: alert.level,
            message: (React.createElement("div", { className: "alert-message" },
                React.createElement("div", { className: "header bold-text" }, alert.message),
                React.createElement("div", { className: "description" }, alert.description))),
            position: alertsPosition,
            title: withIcon
                ? (React.createElement(GenericIcon, { icon: icon, colored: true, floated: "left", transparent: true, size: "mini", inline: true, spaced: "right", relaxed: true }))
                : null
        });
    }, [alert]);
    return (React.createElement("div", { className: `alert-wrapper ${classes}` },
        React.createElement(NotificationSystem, { ref: alertRef })));
};
/**
 * Prop types for the Alert component.
 */
Alert.defaultProps = {
    absolute: false,
    alertsPosition: "br",
    dismissInterval: 5,
    dismissible: true,
    withIcon: true
};
//# sourceMappingURL=alert.js.map