/**
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { getGravatarImage } from "@wso2is/core/api";
import { resolveUserDisplayName, resolveUsername } from "@wso2is/core/helpers";
import classNames from "classnames";
import React from "react";
import { Link } from "react-router-dom";
import { Container, Divider, Dropdown, Icon, Item, Menu, Placeholder, Responsive } from "semantic-ui-react";
import { UserAvatar } from "../avatar";
/**
 * Header component.
 *
 * @param {HeaderPropsInterface} props - Props injected to the component.
 * @return {JSX.Element}
 */
export const Header = (props) => {
    const { brand, brandLink, basicProfileInfo, className, fixed, fluid, isProfileInfoLoading, linkedAccounts, profileInfo, userDropdownInfoAction, showSidePanelToggle, showUserDropdown, onLinkedAccountSwitch, onSidePanelToggleClick, userDropdownIcon, userDropdownLinks } = props;
    const classes = classNames("app-header", {
        ["fluid-header"]: fluid
    }, className);
    const trigger = (React.createElement("span", { className: "user-dropdown-trigger" },
        React.createElement("div", { className: "username" }, isProfileInfoLoading
            ? (React.createElement(Placeholder, null,
                React.createElement(Placeholder.Line, null)))
            : resolveUserDisplayName(profileInfo, basicProfileInfo)),
        React.createElement(UserAvatar, { isLoading: isProfileInfoLoading, authState: basicProfileInfo, profileInfo: profileInfo, size: "mini" })));
    /**
     * Stops the dropdown from closing on click.
     *
     * @param { React.SyntheticEvent<HTMLElement> } e - Click event.
     */
    const handleUserDropdownClick = (e) => {
        e.stopPropagation();
    };
    /**
     * Handles the account switch click event.
     *
     * @param { LinkedAccountInterface } account - Target account.
     */
    const handleLinkedAccountSwitch = (account) => {
        onLinkedAccountSwitch(account);
    };
    const resolveAuthenticatedUserEmail = () => {
        if (!profileInfo || !profileInfo.emails || !profileInfo.emails.length || profileInfo.emails.length < 1) {
            return null;
        }
        if (typeof profileInfo.emails[0] === "string") {
            return profileInfo.emails[0];
        }
        else if (typeof profileInfo.emails[0] === "object") {
            return profileInfo.emails[0].value;
        }
        return null;
    };
    return (React.createElement(Menu, { id: "app-header", className: classes, fixed: fixed, borderless: true },
        React.createElement(Container, { fluid: fluid },
            showSidePanelToggle
                ? (React.createElement(Responsive, { as: Menu.Item, maxWidth: 767 },
                    React.createElement(Icon, { name: "bars", size: "large", onClick: onSidePanelToggleClick, link: true })))
                : null,
            brand && (React.createElement(Menu.Item, { as: Link, to: brandLink, header: true }, brand)),
            (React.createElement(Menu.Menu, { position: "right" }, showUserDropdown && (React.createElement(Dropdown, { item: true, trigger: trigger, floating: true, icon: userDropdownIcon, className: "user-dropdown" },
                React.createElement(Dropdown.Menu, { onClick: handleUserDropdownClick },
                    React.createElement(Item.Group, { className: "authenticated-user", unstackable: true },
                        React.createElement(Item, { className: "header", key: `logged-in-user-${profileInfo.userName}` },
                            React.createElement(UserAvatar, { authState: basicProfileInfo, isLoading: isProfileInfoLoading, size: "tiny" }),
                            React.createElement(Item.Content, { verticalAlign: "middle" },
                                React.createElement(Item.Description, null,
                                    React.createElement("div", { className: "name" }, isProfileInfoLoading
                                        ? React.createElement(Placeholder, null,
                                            React.createElement(Placeholder.Line, null))
                                        : resolveUserDisplayName(profileInfo, basicProfileInfo)),
                                    (profileInfo.emails !== undefined
                                        && profileInfo.emails !== null) && (React.createElement("div", { className: "email" }, isProfileInfoLoading
                                        ? React.createElement(Placeholder, null,
                                            React.createElement(Placeholder.Line, null))
                                        : resolveAuthenticatedUserEmail())),
                                    userDropdownInfoAction && (React.createElement(React.Fragment, null,
                                        React.createElement(Divider, { hidden: true }),
                                        userDropdownInfoAction)))))),
                    React.createElement(Dropdown.Divider, null),
                    (linkedAccounts && linkedAccounts.length && linkedAccounts.length > 0)
                        ? (React.createElement(Item.Group, { className: "linked-accounts-list", unstackable: true }, linkedAccounts.map((association, index) => (React.createElement(Item, { className: "linked-account", key: `${association.userId}-${index}`, onClick: () => handleLinkedAccountSwitch(association) },
                            React.createElement(UserAvatar, { bordered: true, avatar: true, size: "little", image: getGravatarImage(association.email), name: association.username }),
                            React.createElement(Item.Content, { verticalAlign: "middle" },
                                React.createElement(Item.Description, null,
                                    React.createElement("div", { className: "name" }, resolveUsername(association.username, association.userStoreDomain)),
                                    React.createElement("div", { className: "email" }, association.tenantDomain))))))))
                        : null,
                    (userDropdownLinks
                        && userDropdownLinks.length
                        && userDropdownLinks.length > 0)
                        ? userDropdownLinks.map((link, index) => (React.createElement(Dropdown.Item, { key: index, className: "action-panel" },
                            React.createElement(Link, { className: "action-button", to: link.to }, link.name))))
                        : null))))))));
};
/**
 * Default prop types for the header component.
 */
Header.defaultProps = {
    fixed: "top",
    fluid: false,
    onLinkedAccountSwitch: () => null,
    onSidePanelToggleClick: () => null,
    showSidePanelToggle: true,
    showUserDropdown: true,
    userDropdownIcon: null
};
//# sourceMappingURL=header.js.map