"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
Object.defineProperty(exports, "__esModule", { value: true });
var fs = require("fs");
var path = require("path");
var ts = require("typescript");
var buildFilter_1 = require("./buildFilter");
// We'll use the currentDirectoryName to trim parent fileNames
var currentDirectoryPath = process.cwd();
var currentDirectoryParts = currentDirectoryPath.split(path.sep);
var currentDirectoryName = currentDirectoryParts[currentDirectoryParts.length - 1];
exports.defaultParserOpts = {};
var defaultOptions = {
    jsx: ts.JsxEmit.React,
    module: ts.ModuleKind.CommonJS,
    target: ts.ScriptTarget.Latest
};
/**
 * Parses a file with default TS options
 * @param filePath component file that should be parsed
 */
function parse(filePathOrPaths, parserOpts) {
    if (parserOpts === void 0) { parserOpts = exports.defaultParserOpts; }
    return withCompilerOptions(defaultOptions, parserOpts).parse(filePathOrPaths);
}
exports.parse = parse;
/**
 * Constructs a parser for a default configuration.
 */
function withDefaultConfig(parserOpts) {
    if (parserOpts === void 0) { parserOpts = exports.defaultParserOpts; }
    return withCompilerOptions(defaultOptions, parserOpts);
}
exports.withDefaultConfig = withDefaultConfig;
/**
 * Constructs a parser for a specified tsconfig file.
 */
function withCustomConfig(tsconfigPath, parserOpts) {
    var basePath = path.dirname(tsconfigPath);
    var _a = ts.readConfigFile(tsconfigPath, function (filename) {
        return fs.readFileSync(filename, 'utf8');
    }), config = _a.config, error = _a.error;
    if (error !== undefined) {
        // tslint:disable-next-line: max-line-length
        var errorText = "Cannot load custom tsconfig.json from provided path: " + tsconfigPath + ", with error code: " + error.code + ", message: " + error.messageText;
        throw new Error(errorText);
    }
    var _b = ts.parseJsonConfigFileContent(config, ts.sys, basePath, {}, tsconfigPath), options = _b.options, errors = _b.errors;
    if (errors && errors.length) {
        throw errors[0];
    }
    return withCompilerOptions(options, parserOpts);
}
exports.withCustomConfig = withCustomConfig;
/**
 * Constructs a parser for a specified set of TS compiler options.
 */
function withCompilerOptions(compilerOptions, parserOpts) {
    if (parserOpts === void 0) { parserOpts = exports.defaultParserOpts; }
    return {
        parse: function (filePathOrPaths) {
            return parseWithProgramProvider(filePathOrPaths, compilerOptions, parserOpts);
        },
        parseWithProgramProvider: function (filePathOrPaths, programProvider) {
            return parseWithProgramProvider(filePathOrPaths, compilerOptions, parserOpts, programProvider);
        }
    };
}
exports.withCompilerOptions = withCompilerOptions;
var defaultJSDoc = {
    description: '',
    fullComment: '',
    tags: {}
};
var Parser = /** @class */ (function () {
    function Parser(program, opts) {
        var savePropValueAsString = opts.savePropValueAsString, shouldExtractLiteralValuesFromEnum = opts.shouldExtractLiteralValuesFromEnum;
        this.checker = program.getTypeChecker();
        this.propFilter = buildFilter_1.buildFilter(opts);
        this.shouldExtractLiteralValuesFromEnum = Boolean(shouldExtractLiteralValuesFromEnum);
        this.savePropValueAsString = Boolean(savePropValueAsString);
    }
    Parser.prototype.getComponentInfo = function (exp, source, componentNameResolver) {
        if (componentNameResolver === void 0) { componentNameResolver = function () { return undefined; }; }
        if (!!exp.declarations && exp.declarations.length === 0) {
            return null;
        }
        var type = this.checker.getTypeOfSymbolAtLocation(exp, exp.valueDeclaration || exp.declarations[0]);
        var commentSource = exp;
        var typeSymbol = type.symbol || type.aliasSymbol;
        if (!exp.valueDeclaration) {
            if (!typeSymbol) {
                return null;
            }
            exp = typeSymbol;
            var expName = exp.getName();
            if (expName === 'StatelessComponent' ||
                expName === 'Stateless' ||
                expName === 'StyledComponentClass' ||
                expName === 'StyledComponent' ||
                expName === 'FunctionComponent') {
                commentSource = this.checker.getAliasedSymbol(commentSource);
            }
            else {
                commentSource = exp;
            }
        }
        // Skip over PropTypes that are exported
        if (typeSymbol &&
            (typeSymbol.getEscapedName() === 'Requireable' ||
                typeSymbol.getEscapedName() === 'Validator')) {
            return null;
        }
        var propsType = this.extractPropsFromTypeIfStatelessComponent(type) ||
            this.extractPropsFromTypeIfStatefulComponent(type);
        var resolvedComponentName = componentNameResolver(exp, source);
        var displayName = resolvedComponentName || computeComponentName(exp, source);
        var description = this.findDocComment(commentSource).fullComment;
        var methods = this.getMethodsInfo(type);
        if (propsType) {
            var defaultProps = this.extractDefaultPropsFromComponent(exp, source);
            var props = this.getPropsInfo(propsType, defaultProps);
            for (var _i = 0, _a = Object.keys(props); _i < _a.length; _i++) {
                var propName = _a[_i];
                var prop = props[propName];
                var component = { name: displayName };
                if (!this.propFilter(prop, component)) {
                    delete props[propName];
                }
            }
            return {
                description: description,
                displayName: displayName,
                methods: methods,
                props: props
            };
        }
        else if (description && displayName) {
            return {
                description: description,
                displayName: displayName,
                methods: methods,
                props: {}
            };
        }
        return null;
    };
    Parser.prototype.extractPropsFromTypeIfStatelessComponent = function (type) {
        var callSignatures = type.getCallSignatures();
        if (callSignatures.length) {
            // Could be a stateless component.  Is a function, so the props object we're interested
            // in is the (only) parameter.
            for (var _i = 0, callSignatures_1 = callSignatures; _i < callSignatures_1.length; _i++) {
                var sig = callSignatures_1[_i];
                var params = sig.getParameters();
                if (params.length === 0) {
                    continue;
                }
                // Maybe we could check return type instead,
                // but not sure if Element, ReactElement<T> are all possible values
                var propsParam = params[0];
                if (propsParam.name === 'props' || params.length === 1) {
                    return propsParam;
                }
            }
        }
        return null;
    };
    Parser.prototype.extractPropsFromTypeIfStatefulComponent = function (type) {
        var constructSignatures = type.getConstructSignatures();
        if (constructSignatures.length) {
            // React.Component. Is a class, so the props object we're interested
            // in is the type of 'props' property of the object constructed by the class.
            for (var _i = 0, constructSignatures_1 = constructSignatures; _i < constructSignatures_1.length; _i++) {
                var sig = constructSignatures_1[_i];
                var instanceType = sig.getReturnType();
                var props = instanceType.getProperty('props');
                if (props) {
                    return props;
                }
            }
        }
        return null;
    };
    Parser.prototype.extractMembersFromType = function (type) {
        var _this = this;
        var methodSymbols = [];
        /**
         * Need to loop over properties first so we capture any
         * static methods. static methods aren't captured in type.symbol.members
         */
        type.getProperties().forEach(function (property) {
            // Only add members, don't add non-member properties
            if (_this.getCallSignature(property)) {
                methodSymbols.push(property);
            }
        });
        if (type.symbol && type.symbol.members) {
            type.symbol.members.forEach(function (member) {
                methodSymbols.push(member);
            });
        }
        return methodSymbols;
    };
    Parser.prototype.getMethodsInfo = function (type) {
        var _this = this;
        var members = this.extractMembersFromType(type);
        var methods = [];
        members.forEach(function (member) {
            if (!_this.isTaggedPublic(member)) {
                return;
            }
            var name = member.getName();
            var docblock = _this.getFullJsDocComment(member).fullComment;
            var callSignature = _this.getCallSignature(member);
            var params = _this.getParameterInfo(callSignature);
            var description = ts.displayPartsToString(member.getDocumentationComment(_this.checker));
            var returnType = _this.checker.typeToString(callSignature.getReturnType());
            var returnDescription = _this.getReturnDescription(member);
            var modifiers = _this.getModifiers(member);
            methods.push({
                description: description,
                docblock: docblock,
                modifiers: modifiers,
                name: name,
                params: params,
                returns: returnDescription
                    ? {
                        description: returnDescription,
                        type: returnType
                    }
                    : null
            });
        });
        return methods;
    };
    Parser.prototype.getModifiers = function (member) {
        var modifiers = [];
        var flags = ts.getCombinedModifierFlags(member.valueDeclaration);
        var isStatic = (flags & ts.ModifierFlags.Static) !== 0; // tslint:disable-line no-bitwise
        if (isStatic) {
            modifiers.push('static');
        }
        return modifiers;
    };
    Parser.prototype.getParameterInfo = function (callSignature) {
        var _this = this;
        return callSignature.parameters.map(function (param) {
            var paramType = _this.checker.getTypeOfSymbolAtLocation(param, param.valueDeclaration);
            var paramDeclaration = _this.checker.symbolToParameterDeclaration(param);
            var isOptionalParam = !!(paramDeclaration && paramDeclaration.questionToken);
            return {
                description: ts.displayPartsToString(param.getDocumentationComment(_this.checker)) || null,
                name: param.getName() + (isOptionalParam ? '?' : ''),
                type: { name: _this.checker.typeToString(paramType) }
            };
        });
    };
    Parser.prototype.getCallSignature = function (symbol) {
        var symbolType = this.checker.getTypeOfSymbolAtLocation(symbol, symbol.valueDeclaration);
        return symbolType.getCallSignatures()[0];
    };
    Parser.prototype.isTaggedPublic = function (symbol) {
        var jsDocTags = symbol.getJsDocTags();
        var isPublic = Boolean(jsDocTags.find(function (tag) { return tag.name === 'public'; }));
        return isPublic;
    };
    Parser.prototype.getReturnDescription = function (symbol) {
        var tags = symbol.getJsDocTags();
        var returnTag = tags.find(function (tag) { return tag.name === 'returns'; });
        if (!returnTag) {
            return null;
        }
        return returnTag.text || null;
    };
    Parser.prototype.getDocgenType = function (propType) {
        var propTypeString = this.checker.typeToString(propType);
        if (this.shouldExtractLiteralValuesFromEnum &&
            propType.isUnion() &&
            propType.types.every(function (type) { return type.isStringLiteral(); })) {
            return {
                name: 'enum',
                raw: propTypeString,
                value: propType.types
                    .map(function (type) { return ({
                    value: type.isStringLiteral() ? "\"" + type.value + "\"" : undefined
                }); })
                    .filter(Boolean)
            };
        }
        return { name: propTypeString };
    };
    Parser.prototype.getPropsInfo = function (propsObj, defaultProps) {
        var _this = this;
        if (defaultProps === void 0) { defaultProps = {}; }
        if (!propsObj.valueDeclaration) {
            return {};
        }
        var propsType = this.checker.getTypeOfSymbolAtLocation(propsObj, propsObj.valueDeclaration);
        var propertiesOfProps = propsType.getProperties();
        if (!propertiesOfProps.length && propsType.isUnionOrIntersection()) {
            propertiesOfProps = propsType.types.reduce(function (acc, type) { return acc.concat(type.getProperties()); }, []);
        }
        var result = {};
        propertiesOfProps.forEach(function (prop) {
            var propName = prop.getName();
            // Find type of prop by looking in context of the props object itself.
            var propType = _this.checker.getTypeOfSymbolAtLocation(prop, propsObj.valueDeclaration);
            // tslint:disable-next-line:no-bitwise
            var isOptional = (prop.getFlags() & ts.SymbolFlags.Optional) !== 0;
            var jsDocComment = _this.findDocComment(prop);
            var hasCodeBasedDefault = defaultProps[propName] !== undefined;
            var defaultValue = null;
            if (hasCodeBasedDefault) {
                defaultValue = { value: defaultProps[propName] };
            }
            else if (jsDocComment.tags.default) {
                defaultValue = { value: jsDocComment.tags.default };
            }
            var parent = getParentType(prop);
            result[propName] = {
                defaultValue: defaultValue,
                description: jsDocComment.fullComment,
                name: propName,
                parent: parent,
                required: !isOptional && !hasCodeBasedDefault,
                type: _this.getDocgenType(propType)
            };
        });
        return result;
    };
    Parser.prototype.findDocComment = function (symbol) {
        var _this = this;
        var comment = this.getFullJsDocComment(symbol);
        if (comment.fullComment || comment.tags.default) {
            return comment;
        }
        var rootSymbols = this.checker.getRootSymbols(symbol);
        var commentsOnRootSymbols = rootSymbols
            .filter(function (x) { return x !== symbol; })
            .map(function (x) { return _this.getFullJsDocComment(x); })
            .filter(function (x) { return !!x.fullComment || !!comment.tags.default; });
        if (commentsOnRootSymbols.length) {
            return commentsOnRootSymbols[0];
        }
        return defaultJSDoc;
    };
    /**
     * Extracts a full JsDoc comment from a symbol, even
     * though TypeScript has broken down the JsDoc comment into plain
     * text and JsDoc tags.
     */
    Parser.prototype.getFullJsDocComment = function (symbol) {
        // in some cases this can be undefined (Pick<Type, 'prop1'|'prop2'>)
        if (symbol.getDocumentationComment === undefined) {
            return defaultJSDoc;
        }
        var mainComment = ts.displayPartsToString(symbol.getDocumentationComment(this.checker));
        if (mainComment) {
            mainComment = mainComment.replace('\r\n', '\n');
        }
        var tags = symbol.getJsDocTags() || [];
        var tagComments = [];
        var tagMap = {};
        tags.forEach(function (tag) {
            var trimmedText = (tag.text || '').trim();
            var currentValue = tagMap[tag.name];
            tagMap[tag.name] = currentValue
                ? currentValue + '\n' + trimmedText
                : trimmedText;
            if (tag.name !== 'default') {
                tagComments.push(formatTag(tag));
            }
        });
        return {
            description: mainComment,
            fullComment: (mainComment + '\n' + tagComments.join('\n')).trim(),
            tags: tagMap
        };
    };
    Parser.prototype.getFunctionStatement = function (statement) {
        if (ts.isFunctionDeclaration(statement)) {
            return statement;
        }
        if (ts.isVariableStatement(statement)) {
            var initializer = statement.declarationList &&
                statement.declarationList.declarations[0].initializer;
            if (initializer &&
                (ts.isArrowFunction(initializer) ||
                    ts.isFunctionExpression(initializer))) {
                return initializer;
            }
        }
    };
    Parser.prototype.extractDefaultPropsFromComponent = function (symbol, source) {
        var _this = this;
        var possibleStatements = source.statements
            // ensure that name property is available
            .filter(function (stmt) { return !!stmt.name; })
            .filter(function (stmt) {
            return _this.checker.getSymbolAtLocation(stmt.name) === symbol;
        });
        if (!possibleStatements.length) {
            // if no class declaration is found, try to find a
            // expression statement used in a React.StatelessComponent
            possibleStatements = source.statements.filter(function (stmt) { return ts.isExpressionStatement(stmt) || ts.isVariableStatement(stmt); });
        }
        return possibleStatements.reduce(function (res, statement) {
            if (statementIsClassDeclaration(statement) && statement.members.length) {
                var possibleDefaultProps = statement.members.filter(function (member) {
                    return member.name && getPropertyName(member.name) === 'defaultProps';
                });
                if (!possibleDefaultProps.length) {
                    return res;
                }
                var defaultProps = possibleDefaultProps[0];
                var initializer = defaultProps.initializer;
                var properties = initializer.properties;
                while (ts.isIdentifier(initializer)) {
                    var defaultPropsReference = _this.checker.getSymbolAtLocation(initializer);
                    if (defaultPropsReference) {
                        var declarations = defaultPropsReference.getDeclarations();
                        if (declarations) {
                            initializer = declarations[0]
                                .initializer;
                            properties = initializer
                                .properties;
                        }
                    }
                }
                var propMap = {};
                if (properties) {
                    propMap = _this.getPropMap(properties);
                }
                return __assign({}, res, propMap);
            }
            else if (statementIsStatelessWithDefaultProps(statement)) {
                var propMap_1 = {};
                statement.getChildren().forEach(function (child) {
                    var right = child.right;
                    if (right) {
                        var properties = right.properties;
                        if (properties) {
                            propMap_1 = _this.getPropMap(properties);
                        }
                    }
                });
                return __assign({}, res, propMap_1);
            }
            var functionStatement = _this.getFunctionStatement(statement);
            // Extracting default values from props destructuring
            if (functionStatement && functionStatement.parameters.length) {
                var name = functionStatement.parameters[0].name;
                if (ts.isObjectBindingPattern(name)) {
                    return __assign({}, res, _this.getPropMap(name.elements));
                }
            }
            return res;
        }, {});
    };
    Parser.prototype.getLiteralValueFromPropertyAssignment = function (property) {
        var initializer = property.initializer;
        // Shorthand properties, so inflect their actual value
        if (!initializer) {
            if (ts.isShorthandPropertyAssignment(property)) {
                var symbol = this.checker.getShorthandAssignmentValueSymbol(property);
                var decl = symbol && symbol.valueDeclaration;
                if (decl && decl.initializer) {
                    initializer = decl.initializer;
                }
            }
        }
        if (!initializer) {
            return undefined;
        }
        // Literal values
        switch (initializer.kind) {
            case ts.SyntaxKind.FalseKeyword:
                return this.savePropValueAsString ? 'false' : false;
            case ts.SyntaxKind.TrueKeyword:
                return this.savePropValueAsString ? 'true' : true;
            case ts.SyntaxKind.StringLiteral:
                return initializer.text.trim();
            case ts.SyntaxKind.PrefixUnaryExpression:
                return this.savePropValueAsString
                    ? initializer.getFullText().trim()
                    : Number(initializer.getFullText());
            case ts.SyntaxKind.NumericLiteral:
                return this.savePropValueAsString
                    ? "" + initializer.text
                    : Number(initializer.text);
            case ts.SyntaxKind.NullKeyword:
                return this.savePropValueAsString ? 'null' : null;
            case ts.SyntaxKind.Identifier:
                // can potentially find other identifiers in the source and map those in the future
                return initializer.text === 'undefined'
                    ? 'undefined'
                    : null;
            case ts.SyntaxKind.PropertyAccessExpression:
            case ts.SyntaxKind.ObjectLiteralExpression:
            default:
                try {
                    return initializer.getText();
                }
                catch (e) {
                    return null;
                }
        }
    };
    Parser.prototype.getPropMap = function (properties) {
        var _this = this;
        var propMap = properties.reduce(function (acc, property) {
            if (ts.isSpreadAssignment(property) || !property.name) {
                return acc;
            }
            var literalValue = _this.getLiteralValueFromPropertyAssignment(property);
            var propertyName = getPropertyName(property.name);
            if ((typeof literalValue === 'string' ||
                typeof literalValue === 'number' ||
                typeof literalValue === 'boolean' ||
                literalValue === null) &&
                propertyName !== null) {
                acc[propertyName] = literalValue;
            }
            return acc;
        }, {});
        return propMap;
    };
    return Parser;
}());
exports.Parser = Parser;
function statementIsClassDeclaration(statement) {
    return !!statement.members;
}
function statementIsStatelessWithDefaultProps(statement) {
    var children = statement.getChildren();
    for (var _i = 0, children_1 = children; _i < children_1.length; _i++) {
        var child = children_1[_i];
        var left = child.left;
        if (left) {
            var name = left.name;
            if (name && name.escapedText === 'defaultProps') {
                return true;
            }
        }
    }
    return false;
}
function getPropertyName(name) {
    switch (name.kind) {
        case ts.SyntaxKind.NumericLiteral:
        case ts.SyntaxKind.StringLiteral:
        case ts.SyntaxKind.Identifier:
            return name.text;
        case ts.SyntaxKind.ComputedPropertyName:
            return name.getText();
        default:
            return null;
    }
}
function formatTag(tag) {
    var result = '@' + tag.name;
    if (tag.text) {
        result += ' ' + tag.text;
    }
    return result;
}
function getTextValueOfClassMember(classDeclaration, memberName) {
    var classDeclarationMembers = classDeclaration.members || [];
    var textValue = (classDeclarationMembers &&
        classDeclarationMembers
            .filter(function (member) { return ts.isPropertyDeclaration(member); })
            .filter(function (member) {
            var name = ts.getNameOfDeclaration(member);
            return name && name.text === memberName;
        })
            .map(function (member) {
            var property = member;
            return (property.initializer && property.initializer.text);
        }))[0];
    return textValue || '';
}
function getTextValueOfFunctionProperty(exp, source, propertyName) {
    var textValue = source.statements
        .filter(function (statement) { return ts.isExpressionStatement(statement); })
        .filter(function (statement) {
        var expr = statement
            .expression;
        return (expr.left &&
            expr.left.name &&
            expr.left.name.escapedText ===
                propertyName);
    })
        .filter(function (statement) {
        var expr = statement
            .expression;
        return (expr.left.expression
            .escapedText === exp.getName());
    })
        .filter(function (statement) {
        return ts.isStringLiteral(statement
            .expression.right);
    })
        .map(function (statement) {
        return statement
            .expression.right.text;
    })[0];
    return textValue || '';
}
function computeComponentName(exp, source) {
    var exportName = exp.getName();
    var statelessDisplayName = getTextValueOfFunctionProperty(exp, source, 'displayName');
    var statefulDisplayName = exp.valueDeclaration &&
        ts.isClassDeclaration(exp.valueDeclaration) &&
        getTextValueOfClassMember(exp.valueDeclaration, 'displayName');
    if (statelessDisplayName || statefulDisplayName) {
        return statelessDisplayName || statefulDisplayName || '';
    }
    if (exportName === 'default' ||
        exportName === '__function' ||
        exportName === 'Stateless' ||
        exportName === 'StyledComponentClass' ||
        exportName === 'StyledComponent' ||
        exportName === 'FunctionComponent' ||
        exportName === 'StatelessComponent') {
        return getDefaultExportForFile(source);
    }
    else {
        return exportName;
    }
}
// Default export for a file: named after file
function getDefaultExportForFile(source) {
    var name = path.basename(source.fileName, path.extname(source.fileName));
    var filename = name === 'index' ? path.basename(path.dirname(source.fileName)) : name;
    // JS identifiers must starts with a letter, and contain letters and/or numbers
    // So, you could not take filename as is
    var identifier = filename
        .replace(/^[^A-Z]*/gi, '')
        .replace(/[^A-Z0-9]*/gi, '');
    return identifier.length ? identifier : 'DefaultName';
}
exports.getDefaultExportForFile = getDefaultExportForFile;
function getParentType(prop) {
    var declarations = prop.getDeclarations();
    if (declarations == null || declarations.length === 0) {
        return undefined;
    }
    // Props can be declared only in one place
    var parent = declarations[0].parent;
    if (!isInterfaceOrTypeAliasDeclaration(parent)) {
        return undefined;
    }
    var parentName = parent.name.text;
    var fileName = parent.getSourceFile().fileName;
    var fileNameParts = fileName.split('/');
    var trimmedFileNameParts = fileNameParts.slice();
    while (trimmedFileNameParts.length) {
        if (trimmedFileNameParts[0] === currentDirectoryName) {
            break;
        }
        trimmedFileNameParts.splice(0, 1);
    }
    var trimmedFileName;
    if (trimmedFileNameParts.length) {
        trimmedFileName = trimmedFileNameParts.join('/');
    }
    else {
        trimmedFileName = fileName;
    }
    return {
        fileName: trimmedFileName,
        name: parentName
    };
}
function isInterfaceOrTypeAliasDeclaration(node) {
    return (node.kind === ts.SyntaxKind.InterfaceDeclaration ||
        node.kind === ts.SyntaxKind.TypeAliasDeclaration);
}
function parseWithProgramProvider(filePathOrPaths, compilerOptions, parserOpts, programProvider) {
    var filePaths = Array.isArray(filePathOrPaths)
        ? filePathOrPaths
        : [filePathOrPaths];
    var program = programProvider
        ? programProvider()
        : ts.createProgram(filePaths, compilerOptions);
    var parser = new Parser(program, parserOpts);
    var checker = program.getTypeChecker();
    return filePaths
        .map(function (filePath) { return program.getSourceFile(filePath); })
        .filter(function (sourceFile) {
        return typeof sourceFile !== 'undefined';
    })
        .reduce(function (docs, sourceFile) {
        var moduleSymbol = checker.getSymbolAtLocation(sourceFile);
        if (!moduleSymbol) {
            return docs;
        }
        Array.prototype.push.apply(docs, checker
            .getExportsOfModule(moduleSymbol)
            .map(function (exp) {
            return parser.getComponentInfo(exp, sourceFile, parserOpts.componentNameResolver);
        })
            .filter(function (comp) { return comp !== null; })
            .filter(function (comp, index, comps) {
            return comps
                .slice(index + 1)
                .every(function (innerComp) { return innerComp.displayName !== comp.displayName; });
        }));
        return docs;
    }, []);
}
//# sourceMappingURL=parser.js.map