/**
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { AuthenticateSessionUtil, SignInUtil } from "@wso2is/authentication";
import { AxiosHttpClient } from "@wso2is/http";
import _ from "lodash";
import { CommonServiceResourcesEndpoints } from "../configs";
import { HTTPRequestHeaders } from "../helpers";
import { AcceptHeaderValues, ContentTypeHeaderValues, HttpMethods } from "../models";
import { ContextUtils } from "../utils";
import axios from "axios";
/**
 * Get an http client instance.
 *
 * @type {AxiosHttpClientInstance}
 */
const httpClient = AxiosHttpClient.getInstance();
/**
 * Retrieve the user information of the currently authenticated user.
 *
 * @return {Promise<ProfileInfoInterface>} User information as a Promise.
 */
export const getUserInfo = () => {
    const requestConfig = {
        headers: HTTPRequestHeaders(ContextUtils.getRuntimeConfig().clientHost, null, ContentTypeHeaderValues.APP_JSON),
        method: HttpMethods.GET,
        url: CommonServiceResourcesEndpoints(ContextUtils.getRuntimeConfig().serverHost).me
    };
    return httpClient.request(requestConfig)
        .then((response) => {
        if (response.status !== 200) {
            return Promise.reject("Failed get user info.");
        }
        return Promise.resolve(response.data);
    })
        .catch((error) => {
        return Promise.reject(error);
    });
};
/**
 *  Get Gravatar image using the email address.
 *
 * @param email - Email address.
 * @return {Promise<string>} Valid Gravatar URL as a Promise.
 */
export const getGravatarImage = (email) => {
    const requestConfig = {
        method: HttpMethods.GET,
        url: SignInUtil.getGravatar(email)
    };
    return axios.request(requestConfig)
        .then(() => {
        return Promise.resolve(requestConfig.url.split("?")[0]);
    })
        .catch((error) => {
        return Promise.reject(error);
    })
        .finally(() => {
        // Re-enable the handler.
        httpClient.enableHandler();
    });
};
/**
 * Retrieve the user profile details of the currently authenticated user.
 *
 * @param {() => void} onSCIMDisabled - Callback to be fired if SCIM is disabled for the user store.
 * @returns {Promise<ProfileInfoInterface>} Profile information as a Promise.
 */
export const getProfileInfo = (onSCIMDisabled) => {
    const orgKey = "urn:ietf:params:scim:schemas:extension:enterprise:2.0:User";
    const requestConfig = {
        headers: HTTPRequestHeaders(ContextUtils.getRuntimeConfig().clientHost, AcceptHeaderValues.APP_JSON, ContentTypeHeaderValues.APP_SCIM),
        method: HttpMethods.GET,
        url: CommonServiceResourcesEndpoints(ContextUtils.getRuntimeConfig().serverHost).me
    };
    return httpClient.request(requestConfig)
        .then((response) => __awaiter(void 0, void 0, void 0, function* () {
        let gravatar = "";
        if (response.status !== 200) {
            return Promise.reject("Failed get user profile info.");
        }
        if (_.isEmpty(response.data.userImage) && !response.data.profileUrl) {
            try {
                gravatar = yield getGravatarImage(typeof response.data.emails[0] === "string"
                    ? response.data.emails[0]
                    : response.data.emails[0].value);
            }
            catch (error) {
                gravatar = "";
            }
        }
        const profileImage = response.data.profileUrl ? response.data.profileUrl : gravatar;
        const profileResponse = Object.assign({ emails: response.data.emails || "", name: response.data.name || { givenName: "", familyName: "" }, organisation: response.data[orgKey] ? response.data[orgKey].organization : "", phoneNumbers: response.data.phoneNumbers || [], profileUrl: response.data.profileUrl || "", responseStatus: response.status || null, roles: response.data.roles || [], userImage: response.data.userImage || profileImage, userName: response.data.userName || "" }, response.data);
        return Promise.resolve(profileResponse);
    }))
        .catch((error) => {
        // Check if the API responds with a `500` error, if it does,
        // navigate the user to the login error page.
        if (error.response
            && error.response.data
            && error.response.data.status
            && error.response.data.status === "500") {
            // Fire `onSCIMDisabled` callback which will navigate the
            // user to the corresponding error page.
            onSCIMDisabled();
        }
        return Promise.reject(error);
    });
};
/**
 * Update the required details of the user profile.
 *
 * @param {object} info - Information that needs to ber updated.
 * @return {Promise<ProfileInfoInterface>} Updated profile info as a Promise.
 */
export const updateProfileInfo = (info) => {
    const requestConfig = {
        data: info,
        headers: HTTPRequestHeaders(ContextUtils.getRuntimeConfig().clientHost, null),
        method: HttpMethods.PATCH,
        url: CommonServiceResourcesEndpoints(ContextUtils.getRuntimeConfig().serverHost).me
    };
    return httpClient.request(requestConfig)
        .then((response) => {
        if (response.status !== 200) {
            return Promise.reject("Failed update user profile info.");
        }
        return Promise.resolve(response.data);
    })
        .catch((error) => {
        return Promise.reject(error);
    });
};
/**
 * Retrieve the profile schemas of the user claims of the currently authenticated user.
 *
 * @return {Promise<ProfileSchemaInterface[]>} Array of profile schemas as a Promise.
 */
export const getProfileSchemas = () => {
    const requestConfig = {
        headers: HTTPRequestHeaders(ContextUtils.getRuntimeConfig().clientHost, null, ContentTypeHeaderValues.APP_JSON),
        method: HttpMethods.GET,
        url: CommonServiceResourcesEndpoints(ContextUtils.getRuntimeConfig().serverHost).profileSchemas
    };
    return httpClient.request(requestConfig)
        .then((response) => {
        if (response.status !== 200) {
            return Promise.reject(new Error("Failed get user schemas"));
        }
        return Promise.resolve(response.data[0].attributes);
    })
        .catch((error) => {
        return Promise.reject(error);
    });
};
/**
 * Switches the logged in user's account to one of the linked accounts
 * associated to the corresponding user.
 *
 * @param {LinkedAccountInterface} account - The target account.
 * @param {string[]} scopes - Required scopes array.
 * @param {string} clientID - Client ID.
 * @param {string} clientHost - Client Host URL.
 * @return {Promise<any>}
 */
export const switchAccount = (account, scopes, clientID, clientHost) => {
    const requestParams = {
        "client_id": clientID,
        "scope": scopes,
        "tenant-domain": account.tenantDomain,
        "username": account.username,
        "userstore-domain": account.userStoreDomain
    };
    return SignInUtil.sendAccountSwitchRequest(requestParams, clientHost)
        .then((response) => {
        AuthenticateSessionUtil.initUserSession(response, SignInUtil.getAuthenticatedUser(response.idToken));
        return Promise.resolve(response);
    })
        .catch((error) => {
        return Promise.reject(error);
    });
};
//# sourceMappingURL=profile.js.map