/**
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 */
import _ from "lodash";
import { hasRequiredScopes } from "../helpers";
/**
 * Utility class for application routes related operations.
 */
export class RouteUtils {
    /**
     * Private constructor to avoid object instantiation from outside
     * the class.
     *
     * @hideconstructor
     */
    // eslint-disable-next-line @typescript-eslint/no-empty-function
    constructor() { }
    /**
     * Filters the set of enabled routes based on the app config.
     *
     * @param {RouteInterface[] | ChildRouteInterface[]} routes - Routes to evaluate.
     * @param {T} featureConfig - Feature config.
     *
     * @return {RouteInterface[] | ChildRouteInterface[]} Filtered routes.
     */
    static filterEnabledRoutes(routes, featureConfig) {
        // Filters features based on scope requirements.
        const filter = (routeArr) => {
            return routeArr.filter((route) => {
                var _a;
                if (route.children) {
                    route.children = filter(route.children);
                }
                let feature = null;
                for (const [key, value] of Object.entries(featureConfig)) {
                    if (key === route.id) {
                        feature = value;
                        break;
                    }
                }
                if (!feature) {
                    return true;
                }
                return hasRequiredScopes(feature, (_a = feature === null || feature === void 0 ? void 0 : feature.scopes) === null || _a === void 0 ? void 0 : _a.read);
            });
        };
        // Remove any redundant routes.
        const sanitize = (routeArr) => {
            return routeArr.filter((route) => {
                if (_.isEmpty(route.children) && !route.path) {
                    return false;
                }
                if (!_.isEmpty(route.children) && !route.path) {
                    const isFurtherNested = route.children.some((item) => item.children);
                    if (isFurtherNested) {
                        route.children = sanitize(route.children);
                    }
                    else {
                        return route.children.some((item) => item.showOnSidePanel);
                    }
                }
                if (route.children) {
                    route.children = sanitize(route.children);
                }
                return true;
            });
        };
        return sanitize(filter(routes));
    }
}
//# sourceMappingURL=route-utils.js.map