/**
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { CodeIcon, DummyUser } from "@wso2is/theme";
import classNames from "classnames";
import React from "react";
import { Image, Placeholder } from "semantic-ui-react";
/**
 * Avatar component.
 *
 * @param {AvatarPropsInterface} props - Props passed in to the Avatar component.
 * @return {JSX.Element}
 * @constructor
 */
export const Avatar = (props) => {
    const { avatar, avatarInitialsLimit, avatarType, bordered, children, className, floated, image, inline, isLoading, label, name, onClick, onMouseOver, onMouseOut, relaxed, size, spaced, style, transparent } = props;
    const relaxLevel = (relaxed && relaxed === true) ? "" : relaxed;
    const classes = classNames({
        bordered,
        [`floated-${floated}`]: floated,
        inline,
        relaxed,
        [`${size}`]: size,
        [`spaced-${spaced}`]: spaced,
        transparent,
        [`${avatarType === "user" ? "user-avatar" : "app-avatar"}`]: avatar,
        [`${relaxLevel}`]: relaxLevel,
    }, className);
    // If loading, show the placeholder.
    if (isLoading) {
        return (React.createElement(Image, { className: `${avatarType === "user" ? "user-image" : "app-image"} ${classes}`, bordered: bordered, floated: floated, circular: avatarType === "user", rounded: avatarType === "app", style: style },
            React.createElement(Placeholder, null,
                React.createElement(Placeholder.Image, { square: true }))));
    }
    /**
     * Generates the initials for the avatar. If the name
     * contains two or more words, two letter initial will
     * be generated using the first two words of the name.
     * i.e For the name "Brion Silva", "BS" will be generated.
     * If the name only has one word, then only a single initial
     * will be generated. i.e For "Brion", "B" will be generated.
     *
     * @return {string}
     */
    const generateInitials = () => {
        // App avatar only requires one letter.
        if (avatarType === "app") {
            return name.charAt(0).toUpperCase();
        }
        const nameParts = name.split(" ");
        if (avatarInitialsLimit === 2 && nameParts.length >= 2) {
            return (nameParts[0].charAt(0) + nameParts[1].charAt(0)).toUpperCase();
        }
        return name.charAt(0).toUpperCase();
    };
    if (React.isValidElement(image)) {
        return (React.createElement(Image, { className: `${avatarType === "user" ? "user-image" : "app-image"} ${classes}`, bordered: bordered, floated: floated, circular: avatarType === "user", rounded: avatarType === "app", style: style, onClick: onClick, onMouseOver: onMouseOver, onMouseOut: onMouseOut },
            React.createElement("div", { className: "wrapper" }, image)));
    }
    if (image) {
        return (React.createElement(React.Fragment, null,
            React.createElement(Image, { className: `${avatarType === "user" ? "user-image" : "app-image"} ${classes}`, bordered: bordered, floated: floated, circular: avatarType === "user", rounded: avatarType === "app", style: style, onClick: onClick, onMouseOver: onMouseOver, onMouseOut: onMouseOut },
                React.createElement("div", { className: "wrapper" },
                    label
                        ? (React.createElement("div", { className: "custom-label" },
                            React.createElement(Image, { avatar: true, circular: true, size: "mini", src: label })))
                        : null,
                    children,
                    React.createElement("img", { className: "inner-image", alt: "avatar", src: image })))));
    }
    if (avatar && name) {
        return (React.createElement(Image, { className: `${avatarType === "user" ? "user-image" : "app-image"} ${classes}`, bordered: bordered, floated: floated, verticalAlign: "middle", circular: avatarType === "user", rounded: avatarType === "app", centered: true, style: style, onClick: onClick, onMouseOver: onMouseOver, onMouseOut: onMouseOut },
            children,
            React.createElement("span", { className: "initials" }, generateInitials())));
    }
    return (React.createElement(Image, { className: `${avatarType === "user" ? "user-image" : "app-image"} ${classes}`, bordered: bordered, floated: floated, verticalAlign: "middle", circular: avatarType === "user", rounded: avatarType === "app", centered: true, style: style, onClick: onClick, onMouseOver: onMouseOver, onMouseOut: onMouseOut },
        React.createElement("div", { className: "wrapper" },
            children,
            React.createElement("img", { className: "inner-image", alt: "avatar", src: avatarType === "user" ? DummyUser : CodeIcon.default }))));
};
/**
 * Default prop types for the Avatar component.
 */
Avatar.defaultProps = {
    avatar: false,
    avatarInitialsLimit: 1,
    avatarType: "user",
    bordered: true,
    className: "",
    inline: false,
    isLoading: false,
    label: null,
    onClick: null,
    onMouseOut: null,
    onMouseOver: null,
    relaxed: false,
    size: "mini",
    spaced: null,
    style: {},
    transparent: false
};
//# sourceMappingURL=avatar.js.map