/**
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import React from "react";
import classNames from "classnames";
/**
 * Generic component to render icons.
 *
 * @param {GenericIconProps} props - Props injected to the component.
 * @return {JSX.Element}
 */
export const GenericIcon = (props) => {
    const { bordered, className, colored, defaultIcon, floated, hoverable, icon, inline, link, linkType, onClick, relaxed, rounded, size, spaced, style, square, transparent, twoTone, verticalAlign } = props;
    const relaxLevel = (relaxed && relaxed === true) ? "" : relaxed;
    const classes = classNames({
        "bordered": bordered,
        "colored": colored,
        "default": defaultIcon,
        [`floated-${floated}`]: floated,
        hoverable,
        "inline": inline,
        link,
        [`link-${linkType}`]: linkType,
        "relaxed": relaxed,
        "rounded": rounded,
        [(size === "default") ? "default-size" : size]: size,
        [`spaced-${spaced}`]: spaced,
        "square": square,
        "transparent": transparent,
        "two-tone": twoTone,
        [`${relaxLevel}`]: relaxLevel,
        [`vertical-aligned-${verticalAlign}`]: verticalAlign
    }, className);
    /**
     * Constructs the icon.
     * TODO: Add a default icon if the an error occurs rather than returning null.
     *
     * @return {HTMLElement | SVGElement | JSX.Element}
     */
    const constructContent = () => {
        if (!icon) {
            return null;
        }
        try {
            // Check if the icon is an SVG element
            if (icon instanceof SVGElement) {
                return icon;
            }
            // Check if the icon is a module and has `ReactComponent` property.
            // Important when used with SVG's imported with `@svgr/webpack`.
            if (Object.prototype.hasOwnProperty.call(icon, "ReactComponent")
                && typeof icon.ReactComponent === "function") {
                return React.createElement(icon.ReactComponent, null);
            }
            // Check is icon is a component.
            if (typeof icon === "function") {
                return icon;
            }
            // Check is icon is a component.
            if (typeof icon === "object") {
                return icon;
            }
            // Check if icon passed in is a string. Can be a URL or a base64 encoded.
            if (typeof icon === "string") {
                return React.createElement("img", { src: icon, className: "icon", alt: "icon" });
            }
        }
        catch (e) {
            return null;
        }
    };
    return (React.createElement("div", { className: `theme-icon ${classes}`, style: style, onClick: onClick }, constructContent()));
};
/**
 * Default proptypes for the Generic Icon component.
 */
GenericIcon.defaultProps = {
    bordered: false,
    className: "",
    defaultIcon: false,
    floated: null,
    inline: false,
    relaxed: false,
    rounded: false,
    size: "auto",
    spaced: null,
    square: false,
    style: {},
    transparent: false,
    twoTone: false
};
//# sourceMappingURL=generic-icon.js.map