/**
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import React, { useState, useEffect } from "react";
import _ from "lodash";
import { TransitionGroup, CSSTransition } from "react-transition-group";
/**
 * A Component which will return a tree view for a given set of node data.
 *
 * @param props Props to create a tree view component
 */
export const TreeView = (props) => {
    const [treeData, setTreeData] = useState();
    const [lastCheckToggledNodeIndex, setLastCheckToggledNodeIndex] = useState();
    const { data } = props;
    useEffect(() => {
        setTreeData(_.cloneDeep(data));
    }, [data]);
    /**
     * Util method to handle tree update function.
     *
     * @param updatedData object with updated tree data
     */
    const handleUpdate = (updatedData) => {
        const { depth, onUpdateCb } = props;
        onUpdateCb(updatedData, depth);
    };
    /**
     * Util method to handle tree node check.
     *
     * @param node Tree node
     * @param e Event Object
     */
    const handleCheckToggle = (node, e) => {
        const { onCheckToggleCb, depth } = props;
        const data = _.cloneDeep(treeData);
        const currentNode = _.find(data, node);
        const currentNodeIndex = data.indexOf(currentNode);
        const toggledNodes = [];
        if (e.shiftKey && !_.isNil(lastCheckToggledNodeIndex)) {
            const rangeStart = Math.min(currentNodeIndex, lastCheckToggledNodeIndex);
            const rangeEnd = Math.max(currentNodeIndex, lastCheckToggledNodeIndex);
            const nodeRange = data.slice(rangeStart, rangeEnd + 1);
            nodeRange.forEach((node) => {
                node.isChecked = e.target.checked;
                toggledNodes.push(node);
            });
        }
        else {
            currentNode.isChecked = e.target.checked;
            toggledNodes.push(currentNode);
        }
        onCheckToggleCb(toggledNodes, depth);
        setLastCheckToggledNodeIndex(currentNodeIndex);
        handleUpdate(data);
    };
    /**
     * Util method to handle tree node delete.
     *
     * @param node Tree node
     */
    const handleDelete = (node) => {
        const { onDeleteCb, depth } = props;
        const data = _.cloneDeep(treeData);
        const newData = data.filter((nodeItem) => {
            return !_.isEqual(node, nodeItem);
        });
        onDeleteCb(node, newData, depth) && handleUpdate(newData);
    };
    /**
     * Util method to handle expanding of a selected tree node.
     *
     * @param node Expanded tree node
     */
    const handleExpandToggle = (node) => {
        const { onExpandToggleCb, depth } = props;
        const data = _.cloneDeep(treeData);
        const currentNode = _.find(data, node);
        currentNode.isExpanded = !currentNode.isExpanded;
        if (onExpandToggleCb) {
            onExpandToggleCb(currentNode, depth);
        }
        handleUpdate(data);
    };
    /**
     * Util method to print a checkbox for a given tree node.
     *
     * @param node Tree node to print the check box
     */
    const printCheckbox = (node) => {
        const { isCheckable, keywordLabel, depth } = props;
        const nodeText = _.get(node, keywordLabel, "");
        if (isCheckable(node, depth)) {
            return (React.createElement("label", { htmlFor: node.id, className: "tree-label" },
                React.createElement("input", { type: "checkbox", name: node[keywordLabel], className: "invisible", onClick: (e) => {
                        handleCheckToggle(node, e);
                    }, checked: !!node.isChecked, id: node.id }),
                React.createElement("div", { className: "checkbox " + (node.isPartiallyChecked ? "indeterminate" : "") },
                    React.createElement("svg", { width: "17px", height: "17px", viewBox: "0 0 20 20" },
                        React.createElement("path", { d: "M3,1 L17,1 L17,1 C18.1045695,1 19,1.8954305 19,3 L19,17 L19,17 C19,\n                                18.1045695 18.1045695,19 17,19 L3,19 L3,19 C1.8954305,19 1,18.1045695 1,17 L1,3 L1,\n                                3 C1,1.8954305 1.8954305,1 3,1 Z" }),
                        React.createElement("polyline", { className: "tick", points: "4 11 8 15 16 6" }),
                        React.createElement("polyline", { className: "dash", points: "5 10 15 10 20" }))),
                React.createElement("span", null, nodeText)));
        }
    };
    /**
     * Util method to print a delete button for a given tree node.
     *
     * @param node Tree node to print delete button
     */
    const printDeleteButton = (node) => {
        const { isDeletable, depth, deleteElement } = props;
        if (isDeletable(node, depth)) {
            return (React.createElement("div", { className: "delete-btn", onClick: () => {
                    handleDelete(node);
                } }, deleteElement));
        }
    };
    /**
     * Util method to print the expand button of a given tree node.
     *
     * @param node Tree node to draw the expand button
     */
    const printExpandButton = (node) => {
        const className = node.isExpanded ? "" : "active";
        const { isExpandable, depth } = props;
        if (isExpandable(node, depth)) {
            return (React.createElement("div", { className: "tree-arrow-wrap", onClick: () => {
                    handleExpandToggle(node);
                } },
                React.createElement("span", { className: `tree-arrow ${className}` },
                    React.createElement("span", null),
                    React.createElement("span", null))));
        }
        else {
            return (React.createElement("div", { className: "tree-arrow-wrap" },
                React.createElement("span", { className: "tree-arrow" },
                    React.createElement("span", null),
                    React.createElement("span", null))));
        }
    };
    /**
     * Util method to print a text where no children is available.
     */
    const printNoChildrenMessage = () => {
        const { transitionExitTimeout, noChildrenAvailableMessage } = props;
        const noChildrenTransitionProps = {
            classNames: "treeview-no-children-transition",
            key: "treeview-no-children",
            style: {
                transitionDuration: `${transitionExitTimeout}ms`,
                transitionDelay: `${transitionExitTimeout}ms`
            },
            timeout: {
                enter: transitionExitTimeout
            },
            exit: false
        };
        return (React.createElement(CSSTransition, Object.assign({}, noChildrenTransitionProps),
            React.createElement("div", { className: "treeview-no-children" },
                React.createElement("div", { className: "treeview-no-children-content" }, noChildrenAvailableMessage))));
    };
    const printNodes = (nodeArray) => {
        const { keywordKey, transitionEnterTimeout, transitionExitTimeout, getStyleClassCb } = props;
        const nodeTransitionProps = {
            classNames: "treeview-node-transition",
            style: {
                transitionDuration: `${transitionEnterTimeout}ms`
            },
            timeout: {
                enter: transitionEnterTimeout,
                exit: transitionExitTimeout
            }
        };
        return (React.createElement(TransitionGroup, null, _.isEmpty(nodeArray) ? printNoChildrenMessage() : nodeArray.map((node, index) => {
            return (React.createElement(CSSTransition, Object.assign({}, nodeTransitionProps, { key: node[keywordKey] || index }),
                React.createElement("div", { className: "treeview-node" + getStyleClassCb(node) },
                    React.createElement("div", { className: `treeview-node-content ${!node.children
                            || node.children.length == 0 ? "no-child" : ""}` },
                        node.children && node.children.length != 0 ? printExpandButton(node) : "",
                        printCheckbox(node),
                        printDeleteButton(node)),
                    printChildren(node))));
        })));
    };
    const printChildren = (node) => {
        if (!node.isExpanded) {
            return null;
        }
        const { keywordChildren, keywordChildrenLoading, depth } = props;
        const isChildrenLoading = _.get(node, keywordChildrenLoading, false);
        let childrenElement;
        if (isChildrenLoading) {
            childrenElement = _.get(props, "loadingElement");
        }
        else {
            childrenElement = (React.createElement(TreeView, Object.assign({}, props, { data: node[keywordChildren] || [], depth: depth + 1, onUpdateCb: onChildrenUpdateCb.bind(this) })));
        }
        return (React.createElement("div", { className: "treeview-children-container" }, childrenElement));
        function onChildrenUpdateCb(updatedData) {
            const data = _.cloneDeep(treeData);
            const currentNode = _.find(data, node);
            currentNode[keywordChildren] = updatedData;
            handleUpdate(data);
        }
    };
    return (React.createElement("div", { className: "treeview" }, printNodes(treeData)));
};
TreeView.defaultProps = {
    depth: 0,
    deleteElement: React.createElement("div", null, "(X)"),
    keywordChildren: "children",
    keywordChildrenLoading: "isChildrenLoading",
    keywordLabel: "name",
    keywordKey: "id",
    loadingElement: React.createElement("div", null, "loading..."),
    noChildrenAvailableMessage: "No data found",
    transitionEnterTimeout: 1200,
    transitionExitTimeout: 1200,
    getStyleClassCb: ( /* node, depth */) => { return ""; },
    isCheckable: ( /* node, depth */) => { return true; },
    isDeletable: ( /* node, depth */) => { return true; },
    isExpandable: ( /* node, depth */) => { return true; },
};
//# sourceMappingURL=tree-view.js.map