"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var chai_1 = require("chai");
var fs = require("fs");
var path = require("path");
var ts = require("typescript");
var parser_1 = require("../parser");
var testUtils_1 = require("./testUtils");
describe('parser', function () {
    var children = { type: 'ReactNode', required: false, description: '' };
    it('should parse simple react class component', function () {
        testUtils_1.check('Column', {
            Column: {
                prop1: { type: 'string', required: false },
                prop2: { type: 'number' },
                prop3: { type: '() => void' },
                prop4: { type: '"option1" | "option2" | "option3"' }
            }
        });
    });
    it('should parse simple react class component with console.log inside', function () {
        testUtils_1.check('ColumnWithLog', {
            Column: {
                prop1: { type: 'string', required: false },
                prop2: { type: 'number' },
                prop3: { type: '() => void' },
                prop4: { type: '"option1" | "option2" | "option3"' }
            }
        });
    });
    it('should parse simple react class component as default export', function () {
        testUtils_1.check('ColumnWithDefaultExport', {
            Column: {
                prop1: { type: 'string', required: false },
                prop2: { type: 'number' },
                prop3: { type: '() => void' },
                prop4: { type: '"option1" | "option2" | "option3"' }
            }
        });
    });
    it('should parse mulitple files', function () {
        var result = parser_1.parse([
            testUtils_1.fixturePath('Column'),
            testUtils_1.fixturePath('ColumnWithDefaultExportOnly')
        ]);
        testUtils_1.checkComponent(result, {
            Column: {},
            ColumnWithDefaultExportOnly: {}
        }, false);
    });
    it('should parse simple react class component as default export only', function () {
        testUtils_1.check('ColumnWithDefaultExportOnly', {
            ColumnWithDefaultExportOnly: {
                prop1: { type: 'string', required: false },
                prop2: { type: 'number' },
                prop3: { type: '() => void' },
                prop4: { type: '"option1" | "option2" | "option3"' }
            }
        });
    });
    it('should parse simple react class component as default anonymous export', function () {
        testUtils_1.check('ColumnWithDefaultAnonymousExportOnly', {
            ColumnWithDefaultAnonymousExportOnly: {
                prop1: { type: 'string', required: false },
                prop2: { type: 'number' },
                prop3: { type: '() => void' },
                prop4: { type: '"option1" | "option2" | "option3"' }
            }
        });
    });
    it('should parse simple react class component with state', function () {
        testUtils_1.check('AppMenu', {
            AppMenu: {
                menu: { type: 'any' }
            }
        });
    });
    it('should parse simple react class component with picked properties', function () {
        // we are not able to get correct descriptions for prop1,prop2
        testUtils_1.check('ColumnWithPick', {
            Column: {
                prop1: { type: 'string', required: false, description: '' },
                prop2: { type: 'number', description: '' },
                propx: { type: 'number' }
            }
        });
    });
    it('should parse component with props with external type', function () {
        testUtils_1.check('ColumnWithPropsWithExternalType', {
            ColumnWithPropsWithExternalType: {
                prop1: { type: 'string', required: false },
                prop2: { type: 'number' },
                prop3: { type: 'MyExternalType' }
            }
        });
    });
    it('should parse HOCs', function () {
        testUtils_1.check('ColumnHigherOrderComponent', {
            ColumnExternalHigherOrderComponent: {
                prop1: { type: 'string' }
            },
            ColumnHigherOrderComponent1: {
                prop1: { type: 'string' }
            },
            ColumnHigherOrderComponent2: {
                prop1: { type: 'string' }
            },
            RowExternalHigherOrderComponent: {
                prop1: { type: 'string' }
            },
            RowHigherOrderComponent1: {
                prop1: { type: 'string' }
            },
            RowHigherOrderComponent2: {
                prop1: { type: 'string' }
            }
        });
    });
    it('should parse component with inherited properties HtmlAttributes<any>', function () {
        testUtils_1.check('ColumnWithHtmlAttributes', {
            Column: {
                // tslint:disable:object-literal-sort-keys
                prop1: { type: 'string', required: false },
                prop2: { type: 'number' },
                // HtmlAttributes
                defaultChecked: {
                    type: 'boolean',
                    required: false,
                    description: ''
                }
                // ...
                // tslint:enable:object-literal-sort-keys
            }
        }, false);
    });
    it('should parse component without exported props interface', function () {
        testUtils_1.check('ColumnWithoutExportedProps', {
            Column: {
                prop1: { type: 'string', required: false },
                prop2: { type: 'number' }
            }
        });
    });
    it('should parse functional component exported as const', function () {
        testUtils_1.check('ConstExport', {
            Row: {
                prop1: { type: 'string', required: false },
                prop2: { type: 'number' }
            },
            // TODO: this wasn't there before, i would guess that that's correct
            test: {}
        }, false);
    });
    it('should parse react component with properties defined in external file', function () {
        testUtils_1.check('ExternalPropsComponent', {
            ExternalPropsComponent: {
                prop1: { type: 'string' }
            }
        });
    });
    it('should parse react component with properties extended from an external .tsx file', function () {
        testUtils_1.check('ExtendsExternalPropsComponent', {
            ExtendsExternalPropsComponent: {
                prop1: { type: 'number', required: false, description: 'prop1' },
                prop2: { type: 'string', required: false, description: 'prop2' }
            }
        });
    });
    it('should parse react component with properties defined as type', function () {
        testUtils_1.check('FlippableImage', {
            FlippableImage: {
                isFlippedX: { type: 'boolean', required: false },
                isFlippedY: { type: 'boolean', required: false }
            }
        }, false);
    });
    it('should parse react component with const definitions', function () {
        testUtils_1.check('InlineConst', {
            MyComponent: {
                foo: { type: 'any' }
            }
        });
    });
    it('should parse react component that exports a prop type const', function () {
        testUtils_1.check('ExportsPropTypeShape', {
            ExportsPropTypes: {
                foo: { type: 'any' }
            }
        });
    });
    it('should parse react component that exports a prop type thats imported', function () {
        testUtils_1.check('ExportsPropTypeImport', {
            ExportsPropTypes: {
                foo: { type: 'any' }
            }
        });
    });
    // see issue #132 (https://github.com/styleguidist/react-docgen-typescript/issues/132)
    it('should determine the parent fileName relative to the project directory', function () {
        testUtils_1.check('ExportsPropTypeImport', {
            ExportsPropTypes: {
                foo: {
                    parent: {
                        fileName: 'react-docgen-typescript/src/__tests__/data/ExportsPropTypeImport.tsx',
                        name: 'ExportsPropTypesProps'
                    },
                    type: 'any'
                }
            }
        }, true);
    });
    describe('component with default props', function () {
        var expectation = {
            ComponentWithDefaultProps: {
                sampleDefaultFromJSDoc: {
                    defaultValue: 'hello',
                    description: 'sample with default value',
                    required: true,
                    type: '"hello" | "goodbye"'
                },
                sampleFalse: {
                    defaultValue: false,
                    required: false,
                    type: 'boolean'
                },
                sampleNull: { type: 'null', required: false, defaultValue: null },
                sampleNumber: { type: 'number', required: false, defaultValue: -1 },
                sampleObject: {
                    defaultValue: "{ a: '1', b: 2, c: true, d: false, e: undefined, f: null, g: { a: '1' } }",
                    required: false,
                    type: '{ [key: string]: any; }'
                },
                sampleString: {
                    defaultValue: 'hello',
                    required: false,
                    type: 'string'
                },
                sampleTrue: { type: 'boolean', required: false, defaultValue: true },
                sampleUndefined: {
                    defaultValue: 'undefined',
                    required: false,
                    type: 'any'
                }
            }
        };
        it('should parse defined props', function () {
            testUtils_1.check('ComponentWithDefaultProps', expectation);
        });
        it('should parse referenced props', function () {
            testUtils_1.check('ComponentWithReferencedDefaultProps', expectation);
        });
    });
    it('should parse react PureComponent', function () {
        testUtils_1.check('PureRow', {
            Row: {
                prop1: { type: 'string', required: false },
                prop2: { type: 'number' }
            }
        });
    });
    it('should parse react PureComponent - regression test', function () {
        testUtils_1.check('Regression_v0_0_12', {
            Zoomable: {
                originX: { type: 'number' },
                originY: { type: 'number' },
                scaleFactor: { type: 'number' }
            }
        }, false);
    });
    it('should parse react functional component', function () {
        testUtils_1.check('Row', {
            Row: {
                prop1: { type: 'string', required: false },
                prop2: { type: 'number' }
            }
        });
    });
    it('should parse react stateless component', function () {
        testUtils_1.check('Stateless', {
            Stateless: {
                myProp: { type: 'string' }
            }
        });
    });
    it('should parse react stateless component with intersection props', function () {
        testUtils_1.check('StatelessIntersectionProps', {
            StatelessIntersectionProps: {
                moreProp: { type: 'number' },
                myProp: { type: 'string' }
            }
        });
    });
    it('should parse react stateless component with external intersection props', function () {
        testUtils_1.check('StatelessIntersectionExternalProps', {
            StatelessIntersectionExternalProps: {
                myProp: { type: 'string' },
                prop1: { type: 'string', required: false }
            }
        });
    });
    it('should parse react stateless component with generic intersection props', function () {
        testUtils_1.check('StatelessIntersectionGenericProps', {
            StatelessIntersectionGenericProps: {
                myProp: { type: 'string' }
            }
        });
    });
    it('should parse react stateful component with intersection props', function () {
        testUtils_1.check('StatefulIntersectionProps', {
            StatefulIntersectionProps: {
                moreProp: { type: 'number' },
                myProp: { type: 'string' }
            }
        });
    });
    it('should parse react stateful component with external intersection props', function () {
        testUtils_1.check('StatefulIntersectionExternalProps', {
            StatefulIntersectionExternalProps: {
                myProp: { type: 'string' },
                prop1: { type: 'string', required: false }
            }
        });
    });
    it('should parse react stateful component (wrapped in HOC) with intersection props', function () {
        testUtils_1.check('HOCIntersectionProps', {
            HOCIntersectionProps: {
                injected: { type: 'boolean' },
                myProp: { type: 'string' }
            }
        });
    });
    describe('stateless component with default props', function () {
        var expectation = {
            StatelessWithDefaultProps: {
                sampleDefaultFromJSDoc: {
                    defaultValue: 'hello',
                    description: 'sample with default value',
                    required: true,
                    type: '"hello" | "goodbye"'
                },
                sampleEnum: {
                    defaultValue: 'enumSample.HELLO',
                    required: false,
                    type: 'enumSample'
                },
                sampleFalse: {
                    defaultValue: false,
                    required: false,
                    type: 'boolean'
                },
                sampleNull: { type: 'null', required: false, defaultValue: null },
                sampleNumber: { type: 'number', required: false, defaultValue: -1 },
                sampleObject: {
                    defaultValue: "{ a: '1', b: 2, c: true, d: false, e: undefined, f: null, g: { a: '1' } }",
                    required: false,
                    type: '{ [key: string]: any; }'
                },
                sampleString: {
                    defaultValue: 'hello',
                    required: false,
                    type: 'string'
                },
                sampleTrue: { type: 'boolean', required: false, defaultValue: true },
                sampleUndefined: {
                    defaultValue: undefined,
                    required: false,
                    type: 'any'
                }
            }
        };
        it('should parse defined props', function () {
            testUtils_1.check('StatelessWithDefaultProps', expectation);
        });
        it('should parse props with shorthands', function () {
            testUtils_1.check('StatelessShorthandDefaultProps', {
                StatelessShorthandDefaultProps: {
                    onCallback: {
                        defaultValue: null,
                        description: 'onCallback description',
                        required: false,
                        type: '() => void'
                    },
                    regularProp: {
                        defaultValue: 'foo',
                        description: 'regularProp description',
                        required: false,
                        type: 'string'
                    },
                    shorthandProp: {
                        defaultValue: 123,
                        description: 'shorthandProp description',
                        required: false,
                        type: 'number'
                    }
                }
            });
        });
        it('supports destructuring', function () {
            testUtils_1.check('StatelessWithDestructuredProps', expectation);
        });
        it('supports destructuring for arrow functions', function () {
            testUtils_1.check('StatelessWithDestructuredPropsArrow', expectation);
        });
        it('supports typescript 3.0 style defaulted props', function () {
            testUtils_1.check('StatelessWithDefaultPropsTypescript3', expectation);
        });
    });
    it('should parse jsdocs with the @default tag and no description', function () {
        testUtils_1.check('StatelessWithDefaultOnlyJsDoc', {
            StatelessWithDefaultOnlyJsDoc: {
                myProp: { defaultValue: 'hello', description: '', type: 'string' }
            }
        });
    });
    it('should parse functional component component defined as function', function () {
        testUtils_1.check('FunctionDeclaration', {
            Jumbotron: {
                prop1: { type: 'string', required: true }
            }
        });
    });
    it('should parse functional component component defined as const', function () {
        testUtils_1.check('FunctionalComponentAsConst', {
            Jumbotron: {
                prop1: { type: 'string', required: true }
            }
        });
    });
    it('should parse React.SFC component defined as const', function () {
        testUtils_1.check('ReactSFCAsConst', {
            Jumbotron: {
                prop1: { type: 'string', required: true }
            }
        });
    });
    it('should parse functional component component defined as function as default export', function () {
        testUtils_1.check('FunctionDeclarationAsDefaultExport', {
            Jumbotron: {
                prop1: { type: 'string', required: true }
            }
        });
    });
    it('should parse functional component component defined as const as default export', function () {
        testUtils_1.check('FunctionalComponentAsConstAsDefaultExport', {
            // in this case the component name is taken from the file name
            FunctionalComponentAsConstAsDefaultExport: {
                prop1: { type: 'string', required: true }
            }
        }, true, 'Jumbotron description');
    });
    it('should parse React.SFC component defined as const as default export', function () {
        testUtils_1.check('ReactSFCAsConstAsDefaultExport', {
            // in this case the component name is taken from the file name
            ReactSFCAsConstAsDefaultExport: {
                prop1: { type: 'string', required: true }
            }
        }, true, 'Jumbotron description');
    });
    it('should parse functional component component defined as const as named export', function () {
        testUtils_1.check('FunctionalComponentAsConstAsNamedExport', {
            // in this case the component name is taken from the file name
            FunctionalComponentAsConstAsNamedExport: {
                prop1: { type: 'string', required: true }
            }
        }, true, 'Jumbotron description');
    });
    it('should parse React.SFC component defined as const as named export', function () {
        testUtils_1.check('ReactSFCAsConstAsNamedExport', {
            // in this case the component name is taken from the file name
            ReactSFCAsConstAsNamedExport: {
                prop1: { type: 'string', required: true }
            }
        }, true, 'Jumbotron description');
    });
    describe('displayName', function () {
        it('should be taken from stateless component `displayName` property (using named export)', function () {
            var parsed = parser_1.parse(testUtils_1.fixturePath('StatelessDisplayName'))[0];
            chai_1.assert.equal(parsed.displayName, 'StatelessDisplayName');
        });
        it('should be taken from stateful component `displayName` property (using named export)', function () {
            var parsed = parser_1.parse(testUtils_1.fixturePath('StatefulDisplayName'))[0];
            chai_1.assert.equal(parsed.displayName, 'StatefulDisplayName');
        });
        it('should be taken from stateless component `displayName` property (using default export)', function () {
            var parsed = parser_1.parse(testUtils_1.fixturePath('StatelessDisplayNameDefaultExport'))[0];
            chai_1.assert.equal(parsed.displayName, 'StatelessDisplayNameDefaultExport');
        });
        it('should be taken from stateful component `displayName` property (using default export)', function () {
            var parsed = parser_1.parse(testUtils_1.fixturePath('StatefulDisplayNameDefaultExport'))[0];
            chai_1.assert.equal(parsed.displayName, 'StatefulDisplayNameDefaultExport');
        });
        it('should be taken from named export when default export is an HOC', function () {
            var parsed = parser_1.parse(testUtils_1.fixturePath('StatelessDisplayNameHOC'))[0];
            chai_1.assert.equal(parsed.displayName, 'StatelessDisplayName');
        });
        it('should be taken from named export when default export is an HOC', function () {
            var parsed = parser_1.parse(testUtils_1.fixturePath('StatefulDisplayNameHOC'))[0];
            chai_1.assert.equal(parsed.displayName, 'StatefulDisplayName');
        });
        it('should be taken from stateless component folder name if file name is "index"', function () {
            var parsed = parser_1.parse(testUtils_1.fixturePath('StatelessDisplayNameFolder/index'))[0];
            chai_1.assert.equal(parsed.displayName, 'StatelessDisplayNameFolder');
        });
        it('should be taken from stateful component folder name if file name is "index"', function () {
            var parsed = parser_1.parse(testUtils_1.fixturePath('StatefulDisplayNameFolder/index'))[0];
            chai_1.assert.equal(parsed.displayName, 'StatefulDisplayNameFolder');
        });
    });
    describe('Parser options', function () {
        describe('Property filtering', function () {
            describe('children', function () {
                it('should ignore property "children" if not explicitly documented', function () {
                    testUtils_1.check('Column', {
                        Column: {
                            prop1: { type: 'string', required: false },
                            prop2: { type: 'number' },
                            prop3: { type: '() => void' },
                            prop4: { type: '"option1" | "option2" | "option3"' }
                        }
                    }, true);
                });
                it('should not ignore any property that is documented explicitly', function () {
                    testUtils_1.check('ColumnWithAnnotatedChildren', {
                        Column: {
                            children: {
                                description: 'children description',
                                required: false,
                                type: 'ReactNode'
                            },
                            prop1: { type: 'string', required: false },
                            prop2: { type: 'number' },
                            prop3: { type: '() => void' },
                            prop4: { type: '"option1" | "option2" | "option3"' }
                        }
                    }, true);
                });
            });
            describe('propsFilter method', function () {
                it('should apply filter function and filter components accordingly', function () {
                    var propFilter = function (prop, component) {
                        return prop.name !== 'prop1';
                    };
                    testUtils_1.check('Column', {
                        Column: {
                            prop2: { type: 'number' },
                            prop3: { type: '() => void' },
                            prop4: { type: '"option1" | "option2" | "option3"' }
                        }
                    }, true, undefined, { propFilter: propFilter });
                });
                it('should apply filter function and filter components accordingly', function () {
                    var propFilter = function (prop, component) {
                        if (component.name === 'Column') {
                            return prop.name !== 'prop1';
                        }
                        return true;
                    };
                    testUtils_1.check('Column', {
                        Column: {
                            prop2: { type: 'number' },
                            prop3: { type: '() => void' },
                            prop4: { type: '"option1" | "option2" | "option3"' }
                        }
                    }, true, undefined, { propFilter: propFilter });
                    testUtils_1.check('AppMenu', {
                        AppMenu: {
                            menu: { type: 'any' }
                        }
                    }, true, undefined, { propFilter: propFilter });
                });
                it('should allow filtering by parent interface', function () {
                    var propFilter = function (prop, component) {
                        if (prop.parent == null) {
                            return true;
                        }
                        return (prop.parent.fileName.indexOf('@types/react') < 0 &&
                            prop.parent.name !== 'HTMLAttributes');
                    };
                    testUtils_1.check('ColumnWithHtmlAttributes', {
                        Column: {
                            prop1: { type: 'string', required: false },
                            prop2: { type: 'number' }
                        }
                    }, true, undefined, { propFilter: propFilter });
                });
            });
            describe('skipPropsWithName', function () {
                it('should skip a single property in skipPropsWithName', function () {
                    var propFilter = { skipPropsWithName: 'prop1' };
                    testUtils_1.check('Column', {
                        Column: {
                            prop2: { type: 'number' },
                            prop3: { type: '() => void' },
                            prop4: { type: '"option1" | "option2" | "option3"' }
                        }
                    }, true, undefined, { propFilter: propFilter });
                });
                it('should skip multiple properties in skipPropsWithName', function () {
                    var propFilter = { skipPropsWithName: ['prop1', 'prop2'] };
                    testUtils_1.check('Column', {
                        Column: {
                            prop3: { type: '() => void' },
                            prop4: { type: '"option1" | "option2" | "option3"' }
                        }
                    }, true, undefined, { propFilter: propFilter });
                });
            });
            describe('skipPropsWithoutDoc', function () {
                it('should skip a properties without documentation', function () {
                    var propFilter = { skipPropsWithoutDoc: false };
                    testUtils_1.check('ColumnWithUndocumentedProps', {
                        Column: {
                            prop1: { type: 'string', required: false },
                            prop2: { type: 'number' }
                        }
                    }, true, undefined, { propFilter: propFilter });
                });
            });
        });
        describe('Extracting literal values from enums', function () {
            it('extracts literal values from enum', function () {
                testUtils_1.check('ExtractLiteralValuesFromEnum', {
                    ExtractLiteralValuesFromEnum: {
                        sampleBoolean: { type: 'boolean' },
                        sampleComplexUnion: { type: 'number | "string1" | "string2"' },
                        sampleEnum: {
                            raw: 'sampleEnum',
                            type: 'enum',
                            value: [
                                { value: '"one"' },
                                { value: '"two"' },
                                { value: '"three"' }
                            ]
                        },
                        sampleString: { type: 'string' },
                        sampleStringUnion: {
                            raw: '"string1" | "string2"',
                            type: 'enum',
                            value: [{ value: '"string1"' }, { value: '"string2"' }]
                        }
                    }
                }, true, null, {
                    shouldExtractLiteralValuesFromEnum: true
                });
            });
        });
        describe('Returning not string default props ', function () {
            it('returns not string defaultProps', function () {
                testUtils_1.check('StatelessWithDefaultPropsAsString', {
                    StatelessWithDefaultPropsAsString: {
                        sampleFalse: {
                            defaultValue: 'false',
                            required: false,
                            type: 'boolean'
                        },
                        sampleNull: {
                            defaultValue: 'null',
                            required: false,
                            type: 'null'
                        },
                        sampleNumber: {
                            defaultValue: '1',
                            required: false,
                            type: 'number'
                        },
                        sampleNumberWithPrefix: {
                            defaultValue: '-1',
                            required: false,
                            type: 'number'
                        },
                        sampleTrue: {
                            defaultValue: 'true',
                            required: false,
                            type: 'boolean'
                        },
                        sampleUndefined: {
                            defaultValue: 'undefined',
                            required: false,
                            type: 'undefined'
                        }
                    }
                }, true, null, {
                    savePropValueAsString: true
                });
            });
        });
    });
    describe('withCustomConfig', function () {
        it('should accept tsconfigs that typescript accepts', function () {
            chai_1.assert.ok(parser_1.withCustomConfig(
            // need to navigate to root because tests run on compiled tests
            // and tsc does not include json files
            path.join(__dirname, '../../src/__tests__/data/tsconfig.json'), {}));
        });
    });
    describe('parseWithProgramProvider', function () {
        it('should accept existing ts.Program instance', function () {
            var programProviderInvoked = false;
            // mimic a third party library providing a ts.Program instance.
            var programProvider = function () {
                // need to navigate to root because tests run on compiled tests
                // and tsc does not include json files
                var tsconfigPath = path.join(__dirname, '../../src/__tests__/data/tsconfig.json');
                var basePath = path.dirname(tsconfigPath);
                var _a = ts.readConfigFile(tsconfigPath, function (filename) {
                    return fs.readFileSync(filename, 'utf8');
                }), config = _a.config, error = _a.error;
                chai_1.assert.isUndefined(error);
                var _b = ts.parseJsonConfigFileContent(config, ts.sys, basePath, {}, tsconfigPath), options = _b.options, errors = _b.errors;
                chai_1.assert.lengthOf(errors, 0);
                programProviderInvoked = true;
                return ts.createProgram([testUtils_1.fixturePath('Column')], options);
            };
            var result = parser_1.withDefaultConfig().parseWithProgramProvider([testUtils_1.fixturePath('Column')], programProvider);
            testUtils_1.checkComponent(result, {
                Column: {}
            }, false);
            chai_1.assert.isTrue(programProviderInvoked);
        });
    });
    describe('componentNameResolver', function () {
        it('should override default behavior', function () {
            var parsed = parser_1.parse(testUtils_1.fixturePath('StatelessDisplayNameStyledComponent'), {
                componentNameResolver: function (exp, source) {
                    return exp.getName() === 'StyledComponentClass' &&
                        parser_1.getDefaultExportForFile(source);
                }
            })[0];
            chai_1.assert.equal(parsed.displayName, 'StatelessDisplayNameStyledComponent');
        });
        it('should fallback to default behavior without a match', function () {
            var parsed = parser_1.parse(testUtils_1.fixturePath('StatelessDisplayNameStyledComponent'), {
                componentNameResolver: function () { return false; }
            })[0];
            chai_1.assert.equal(parsed.displayName, 'StatelessDisplayNameStyledComponent');
        });
    });
    describe('methods', function () {
        it('should properly parse methods', function () {
            var parsed = parser_1.parse(testUtils_1.fixturePath('ColumnWithMethods'))[0];
            var methods = parsed.methods;
            var myCoolMethod = methods[0];
            chai_1.assert.equal(myCoolMethod.description, 'My super cool method');
            chai_1.assert.equal(myCoolMethod.docblock, 'My super cool method\n@param myParam Documentation for parameter 1\n@public\n@returns The answer to the universe' // tslint:disable-line max-line-length
            );
            chai_1.assert.deepEqual(myCoolMethod.modifiers, []);
            chai_1.assert.equal(myCoolMethod.name, 'myCoolMethod');
            chai_1.assert.deepEqual(myCoolMethod.params, [
                {
                    description: 'Documentation for parameter 1',
                    name: 'myParam',
                    type: { name: 'number' }
                },
                {
                    description: null,
                    name: 'mySecondParam?',
                    type: { name: 'string' }
                }
            ]);
            chai_1.assert.deepEqual(myCoolMethod.returns, {
                description: 'The answer to the universe',
                type: 'number'
            });
        });
        it('should properly parse static methods', function () {
            var parsed = parser_1.parse(testUtils_1.fixturePath('ColumnWithStaticMethods'))[0];
            var methods = parsed.methods;
            chai_1.assert.equal(methods[0].name, 'myStaticMethod');
            chai_1.assert.deepEqual(methods[0].modifiers, ['static']);
        });
        it('should handle method with no information', function () {
            var parsed = parser_1.parse(testUtils_1.fixturePath('ColumnWithMethods'))[0];
            var methods = parsed.methods;
            chai_1.assert.equal(methods[1].name, 'myBasicMethod');
        });
        it('should handle arrow function', function () {
            var parsed = parser_1.parse(testUtils_1.fixturePath('ColumnWithMethods'))[0];
            var methods = parsed.methods;
            chai_1.assert.equal(methods[2].name, 'myArrowFunction');
        });
        it('should not parse functions not marked with @public', function () {
            var parsed = parser_1.parse(testUtils_1.fixturePath('ColumnWithMethods'))[0];
            var methods = parsed.methods;
            chai_1.assert.equal(Boolean(methods.find(function (method) { return method.name === 'myPrivateFunction'; })), false);
        });
    });
    describe('getDefaultExportForFile', function () {
        it('should filter out forbidden symbols', function () {
            var result = parser_1.getDefaultExportForFile({
                fileName: 'a-b'
            });
            chai_1.assert.equal(result, 'ab');
        });
        it('should remove leading non-letters', function () {
            var result = parser_1.getDefaultExportForFile({
                fileName: '---123aba'
            });
            chai_1.assert.equal(result, 'aba');
        });
        it('should preserve numbers in the middle', function () {
            var result = parser_1.getDefaultExportForFile({
                fileName: '1Body2Text3'
            });
            chai_1.assert.equal(result, 'Body2Text3');
        });
        it('should not return empty string', function () {
            var result = parser_1.getDefaultExportForFile({
                fileName: '---123'
            });
            chai_1.assert.equal(result.length > 0, true);
        });
    });
    describe('issues tests', function () {
        it('188', function () {
            testUtils_1.check('Issue188', {
                Header: {
                    content: { type: 'string', required: true, description: '' }
                }
            }, true, '');
        });
    });
});
//# sourceMappingURL=parser.js.map