/*
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import { TestableComponentInterface } from "@wso2is/core/models";
import { CrossIcon } from "@wso2is/theme";
import classNames from "classnames";
import React, {
    ChangeEvent,
    FunctionComponent,
    MutableRefObject,
    PropsWithChildren,
    ReactElement,
    useEffect,
    useRef,
    useState
} from "react";
import { Button, Divider, Icon, Input, InputProps, Popup, PopupProps } from "semantic-ui-react";
import { GenericIcon } from "../icon";
import { Heading } from "../typography";

/**
 *
 * Proptypes for the advanced search component.
 */
export interface AdvancedSearchPropsInterface extends TestableComponentInterface {
    /**
     * Text alignment.
     */
    aligned?: "left" | "right" | "center";
    /**
     * Additional CSS classes.
     */
    className?: string;
    /**
     * Clear button tooltip label.
     */
    clearButtonPopupLabel?: string;
    /**
     * Clear icon override.
     */
    clearIcon?: any;
    /**
     * Search strategy ex: name sw.
     */
    defaultSearchStrategy: string;
    /**
     * Dropdown appearing position.
     */
    dropdownPosition?: PopupProps["position"];
    /**
     * Dropdown trigger icon label.
     */
    dropdownTriggerPopupLabel?: string;
    /**
     * Search query from outside.
     */
    externalSearchQuery?: string;
    /**
     * Hint action keyboard keys.
     */
    hintActionKeys?: string;
    /**
     * Hint label.
     */
    hintLabel?: string;
    /**
     * Search input size.
     */
    inputSize?: InputProps["size"];
    /**
     * Callback for external search query clear.
     */
    onExternalSearchQueryClear?: () => void;
    /**
     * Callback for search query submit.
     * @param {boolean} processQuery - process flag.
     * @param {string} query - Search query.
     */
    onSearchQuerySubmit: (processQuery: boolean, query: string) => void;
    /**
     * input placeholder.
     */
    placeholder?: string;
    /**
     * Reset trigger.
     */
    resetSubmittedState?: () => void;
    /**
     * Dropdown heading.
     */
    searchOptionsHeader?: string;
    /**
     * Is form submitted.
     */
    submitted?: boolean;
}

/**
 * Advanced search component.
 *
 * @param {React.PropsWithChildren<AdvancedSearchPropsInterface>} props - Props injected to the component.
 *
 * @return {React.ReactElement}
 */
export const AdvancedSearch: FunctionComponent<PropsWithChildren<AdvancedSearchPropsInterface>> = (
    props: PropsWithChildren<AdvancedSearchPropsInterface>
): ReactElement => {

    const {
        aligned,
        className,
        children,
        clearButtonPopupLabel,
        defaultSearchStrategy,
        dropdownPosition,
        dropdownTriggerPopupLabel,
        externalSearchQuery,
        hintActionKeys,
        hintLabel,
        clearIcon,
        inputSize,
        onExternalSearchQueryClear,
        onSearchQuerySubmit,
        placeholder,
        resetSubmittedState,
        searchOptionsHeader,
        submitted,
        [ "data-testid" ]: testId
    } = props;

    const searchInputRef: MutableRefObject<HTMLDivElement> = useRef();

    const [ internalSearchQuery, setInternalSearchQuery ] = useState<string>("");
    const [ showSearchFieldHint, setShowSearchFieldHint ] = useState<boolean>(false);
    const [ isDropdownVisible, setIsDropdownVisible ] = useState<boolean>(false);

    /**
     * useEffect hook to handle `internalSearchQuery` change.
     */
    useEffect(() => {
        if (!internalSearchQuery) {
            setShowSearchFieldHint(false);
        }
        if (internalSearchQuery && !isDropdownVisible && (externalSearchQuery !== internalSearchQuery)) {
            setShowSearchFieldHint(true);
        }
    }, [ internalSearchQuery ]);

    /**
     * useEffect hook to handle `externalSearchQuery` changes.
     */
    useEffect(() => {
        setInternalSearchQuery(externalSearchQuery);
    }, [ externalSearchQuery ]);

    /**
     * useEffect hook to handle `submitted` prop changes.
     */
    useEffect(() => {
        if (submitted) {
            setIsDropdownVisible(false);
            resetSubmittedState();
        }
    }, [ submitted ]);

    /**
     * Wrapper `div` style classes.
     */
    const wrapperClasses = classNames({
        [ `aligned-${ aligned }` ]: aligned
    }, className);

    /**
     * Search field style classes.
     */
    const searchFieldClasses = classNames({
        active: internalSearchQuery
    }, className);

    /**
     * Search field hint style classes.
     */
    const searchFieldHintClasses = classNames({
        active: showSearchFieldHint
    }, className);

    /**
     * Handles the search input field `onChange` event.
     *
     * @param {React.ChangeEvent<HTMLInputElement>} e - Input change event.
     */
    const handleSearchQueryChange = (e: ChangeEvent<HTMLInputElement>): void => {
        const { value } = e.target;

        setInternalSearchQuery(value);
    };

    /**
     * Handles the show options dropdown `onClick` event.
     */
    const handleShowOptionsClick = (): void => {
        setIsDropdownVisible(!isDropdownVisible);
    };

    /**
     * Handles the clear query button `onClick` event.
     */
    const handleQueryClearClick = (): void => {
        setInternalSearchQuery("");
        onSearchQuerySubmit(false, null);
        onExternalSearchQueryClear();
    };

    /**
     * Handles search query submit by keyboard events.
     *
     * @param {KeyboardEvent} e - Keyboard event.
     */
    const handleSearchQuerySubmit = (e: KeyboardEvent): void => {
        const { key, shiftKey } = e;
        let query = "";

        // If only enter key is pressed perform the default filter strategy.
        if (!shiftKey && key === "Enter") {
            query = `${ defaultSearchStrategy } ${ internalSearchQuery }`;
            onSearchQuerySubmit(false, query);
            setShowSearchFieldHint(false);
        }
        // If both `Enter` key and `Shift` key are pressed, treat the input
        // as a query and perform the search.
        if (shiftKey && key === "Enter") {
            query = internalSearchQuery;
            onSearchQuerySubmit(true, query);
            setShowSearchFieldHint(false);
        }
    };

    /**
     * Handles the search field blur.
     */
    const handleSearchFieldBlur = (): void => {
        setShowSearchFieldHint(false);
    };

    /**
     * Handles the dropdown close event.
     */
    const handleSearchDropdownClose = (): void => {
        setIsDropdownVisible(false);
    };

    return (
        <div className={ `advanced-search-wrapper ${ wrapperClasses }` }>
            <div ref={ searchInputRef }>
                <Input
                    data-testid={ `${ testId }_input` }
                    action={ (
                        <>
                            {
                                internalSearchQuery
                                    ? (
                                        <Popup
                                            disabled={ !clearButtonPopupLabel }
                                            trigger={
                                                (
                                                    <Button
                                                        data-testid={ `${ testId }_clear_button` }
                                                        basic
                                                        compact
                                                        className="input-add-on"
                                                        onClick={ handleQueryClearClick }
                                                    >
                                                        <GenericIcon
                                                            size="nano"
                                                            defaultIcon
                                                            transparent
                                                            icon={ clearIcon ? clearIcon : CrossIcon }
                                                        />
                                                    </Button>
                                                )
                                            }
                                            position="top center"
                                            content={ clearButtonPopupLabel }
                                            inverted={ true }
                                        />
                                    )
                                    : null
                            }
                            <Popup
                                disabled={ !dropdownTriggerPopupLabel || isDropdownVisible }
                                trigger={
                                    (
                                        <Button
                                            data-testid={ `${ testId }_options_button` }
                                            basic
                                            compact className="input-add-on"
                                            onClick={ handleShowOptionsClick }
                                        >
                                            <Icon name="caret down"/>
                                        </Button>
                                    )
                                }
                                position="top center"
                                content={ dropdownTriggerPopupLabel }
                                inverted={ true }
                            />
                        </>
                    ) }
                    className={ `advanced-search with-add-on ${ searchFieldClasses }` }
                    size={ inputSize }
                    icon="search"
                    iconPosition="left"
                    placeholder={ placeholder }
                    value={ internalSearchQuery }
                    onBlur={ handleSearchFieldBlur }
                    onChange={ handleSearchQueryChange }
                    onKeyDown={ handleSearchQuerySubmit }
                />
            </div>
            <div className={ `search-query-hint ${ searchFieldHintClasses }` }>
                <div className="query">{ hintLabel }</div>
                <div className="short-cut"><Icon name="keyboard outline"/>{ " " }{ hintActionKeys }</div>
            </div>
            <Popup
                context={ searchInputRef }
                content={ (
                    <div className="search-filters-dropdown">
                        <Heading as="h6" bold="500" compact>{ searchOptionsHeader }</Heading>
                        <Divider className="compact" />
                        <div className="form-wrapper">
                            { children }
                        </div>
                    </div>
                ) }
                on="click"
                position={ dropdownPosition }
                eventsEnabled={ true }
                open={ isDropdownVisible }
                onClose={ handleSearchDropdownClose }
                closeOnPortalMouseLeave={ false }
                hoverable
                pinned
            />
        </div>
    );
};

/**
 * Default proptypes for the Advanced search component.
 */
AdvancedSearch.defaultProps = {
    aligned: "left",
    className: null,
    clearButtonPopupLabel: null,
    dropdownPosition: "bottom left",
    dropdownTriggerPopupLabel: null,
    externalSearchQuery: "",
    hintActionKeys: "Enter",
    hintLabel: "Search for",
    onExternalSearchQueryClear: null,
    placeholder: null,
    searchOptionsHeader: "Advanced Search",
    submitted: false
};
