/**
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import React, { Children, cloneElement, PropsWithChildren, useEffect, useState } from "react";
import classNames from "classnames";
import { Divider } from "semantic-ui-react";
import { Step } from "./step";

/**
 * Steps component prop types.
 */
export interface StepsPropsInterface {
    /**
     * Currently active step.
     */
    current?: number;
    /**
     * Additional classes.
     */
    className?: string;
    /**
     * Header to display above the steps.
     */
    header?: string;
    /**
     * Initially active step.
     */
    initial?: number;
    /**
     * Custom styles object
     */
    style?: object;
}

/**
 * Steps component.
 *
 * @param {React.PropsWithChildren<StepsPropsInterface>} props - Props injected to the component.
 *
 * @return {JSX.Element}
 */
export const Steps: React.FunctionComponent<PropsWithChildren<StepsPropsInterface>> & StepsSubComponentsInterface = (
    props: PropsWithChildren<StepsPropsInterface>
): JSX.Element => {

    const {
        children,
        className,
        current,
        header,
        initial,
        style
    } = props;

    const [ filteredChildren, setFilteredChildren ] = useState<any>([]);

    const classes = classNames(
        "steps",
        className
    );

    useEffect(() => {
        setFilteredChildren(React.Children.toArray(children).filter((child) => !!child));
    }, []);

    return (
        <div className={ classes } style={ style }>
            { header && (
                <>
                    <div className="header">{ header }</div>
                    <Divider hidden/>
                </>
            ) }
            {
                Children.map(filteredChildren, (child, index) => {
                    if (!child) {
                        return null;
                    }

                    const stepNumber = initial + index;

                    const childProps = {
                        stepNumber: `${ stepNumber + 1 }`,
                        ...child.props
                    };

                    if (!child.props.status) {
                        if (stepNumber === current) {
                            childProps.status = "active";
                        } else if (stepNumber < current) {
                            childProps.status = "completed";
                        } else {
                            childProps.status = undefined;
                        }
                    }

                    childProps.active = stepNumber === current;

                    return cloneElement(child, childProps);
                })
            }
        </div>
    );
};

/**
 * Interface for the steps sub components.
 */
export interface StepsSubComponentsInterface {
    Group: typeof Steps;
    Step: typeof Step;
}

Steps.Group = Steps;
Steps.Step = Step;

/**
 * Wizard component default props.
 */
Steps.defaultProps = {
    current: 0,
    initial: 0
};
