/**
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 */
/**
 * Utility class for local storage operations.
 */
export class LocalStorageUtils {
    /**
     * Private constructor to avoid object instantiation from outside
     * the class.
     *
     * @hideconstructor
     */
    // eslint-disable-next-line @typescript-eslint/no-empty-function
    constructor() { }
    /**
     * Set a value in local storage.
     *
     * @param {string} key - Key to set the value under.
     * @param {string} value - Value to be set.
     */
    static setValueInLocalStorage(key, value) {
        localStorage.setItem(key, value);
    }
    /**
     * Get a value from local storage.
     *
     * @param {string} key - Key to retrieve the value.
     * @returns {string} Value or null.
     */
    static getValueFromLocalStorage(key) {
        return localStorage.getItem(key);
    }
}
/**
 * Utility class for session storage operations.
 */
export class SessionStorageUtils {
    /**
     * Private constructor to avoid object instantiation from outside
     * the class.
     *
     * @hideconstructor
     */
    // eslint-disable-next-line @typescript-eslint/no-empty-function
    constructor() { }
    /**
     * Clear the session storage completely.
     */
    static clearSessionStorage() {
        sessionStorage.clear();
    }
    /**
     * Clear a specific item from the session storage.
     *
     * @param {string} key - Key of the item to be removed.
     */
    static clearItemFromSessionStorage(key) {
        sessionStorage.removeItem(key);
    }
    /**
     * Set a specific item to the session storage.
     *
     * @param {string} key - Key of the item to be set.
     * @param {string} value - Value of the item to be set.
     */
    static setItemToSessionStorage(key, value) {
        sessionStorage.setItem(key, value);
    }
    /**
     * Get a specific item from the session storage.
     *
     * @param {string} key - Key of the item to be retrieved.
     */
    static getItemFromSessionStorage(key) {
        return sessionStorage.getItem(key);
    }
}
/**
 * Utility class for cookie storage operations.
 */
export class CookieStorageUtils {
    /**
     * Private constructor to avoid object instantiation from outside
     * the class.
     *
     * @hideconstructor
     */
    constructor() { }
    /**
     * Set a cookie.
     *
     * @param {string} cookieString - Cookie to set.
     */
    static setItem(cookieString) {
        document.cookie = cookieString;
    }
    /**
     * Get the value of a specific cookie.
     *
     * @param {string} name - Name of the cookie to be retrieved.
     */
    static getItem(name) {
        const match = document.cookie.match(new RegExp("(^| )" + name + "=([^;]+)"));
        if (match) {
            return match[2];
        }
        return undefined;
    }
}
//# sourceMappingURL=storage-utils.js.map