/**
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 */
import { URLComponentsInterface } from "../models";
/**
 * Utility class for URL operations and validations.
 */
export declare class URLUtils {
    /**
     * Private constructor to avoid object instantiation from outside
     * the class.
     *
     * @hideconstructor
     */
    private constructor();
    /**
     * Checks if the passed in url is a valid Http URL.
     *
     * @param {string} url - URL to evaluate.
     *
     * @return {boolean} True if the url is a http url.
     */
    static isHttpUrl(url: string): boolean;
    /**
     * Checks if the passed in url is a valid Https URL.
     *
     * @param {string} url - URL to evaluate.
     *
     * @return {boolean} True if the url is a https url.
     */
    static isHttpsUrl(url: string): boolean;
    /**
     * Checks if the passed in url starts with HTTP/HTTPS.
     *
     * @param {string} url - URL to evaluate.
     *
     * @return {boolean} True if the url is a https/http url.
     */
    static isHttpsOrHttpUrl(url: string): boolean;
    /**
     * Checks if the passed in url is a valid data URL.
     *
     * @param {string} url - URL to evaluate.
     *
     * @return {boolean} True if the url is a data url.
     */
    static isDataUrl(url: string): boolean;
    /**
     * Checks if the the provided URL is a valid mobile deep link.
     *
     * @param {string} url - The URL to evaluate.
     *
     * @return {boolean} True if the URL is a mobile deep link.
     */
    static isMobileDeepLink(url: string): boolean;
    /**
     * Splits a given string url into <scheme>://<host> This function does
     * not handle individual ports or paths related to the url. Instead it
     * only returns the protocol, host, and combined origin of the url.
     *
     * Please refer specification for other part implementations of the url:
     * https://www.ietf.org/rfc/rfc2396.txt
     *
     * @param url {string} a valid url string.
     * @returns URLComponentsInterface
     * @throws Invalid URL | undefined
     */
    static urlComponents(url: string): URLComponentsInterface;
    static isURLValid(url: string, checkForSanity?: boolean): boolean;
    /**
     * Checks whether a given url is a SSL enabled protocol or not.
     * @param url {string}
     */
    static isHTTPS(url: string): boolean;
    /**
     * Checks whether a given URL is a valid origin or not.
     * It checks whether theres any pathname, search, or
     * search parameters are present.
     *
     * Will return {@code true} if matches the schema <proto>://<host>
     *
     * @param url {string} any url
     */
    static isAValidOriginUrl(url: string): boolean;
}
