/**
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
import { CommonUtils } from "@wso2is/core/utils";
import classNames from "classnames";
import JSBeautify from "js-beautify";
import { JSHINT } from "jshint/dist/jshint";
import React, { useEffect, useState } from "react";
import { UnControlled as CodeMirror } from "react-codemirror2";
import "codemirror/addon/lint/lint";
import "codemirror/addon/lint/javascript-lint";
import "codemirror/addon/hint/sql-hint";
import "codemirror/mode/javascript/javascript";
import "codemirror/mode/sql/sql";
import "codemirror/mode/xml/xml";
import "codemirror/mode/htmlmixed/htmlmixed";
import "codemirror/addon/edit/closebrackets";
import "codemirror/addon/edit/matchbrackets";
import "codemirror/addon/hint/show-hint";
import "codemirror/addon/hint/javascript-hint";
import "codemirror/lib/codemirror.css";
import "codemirror/theme/material.css";
import "codemirror/addon/lint/lint.css";
import "codemirror/addon/hint/show-hint.css";
import { Modal } from "semantic-ui-react";
import { ReactComponent as CheckIcon } from "../../assets/images/icons/check-icon.svg";
import { ReactComponent as CopyIcon } from "../../assets/images/icons/copy-icon.svg";
import { ReactComponent as MaximizeIcon } from "../../assets/images/icons/maximize-icon.svg";
import { ReactComponent as MinimizeIcon } from "../../assets/images/icons/minimize-icon.svg";
import { GenericIcon } from "../icon";
import { Tooltip } from "../typography";
window.JSHINT = JSHINT;
/**
 * Code editor component.
 *
 * @param {CodeEditorProps} props - Props injected to the danger zone component.
 *
 * @return {React.ReactElement}
 */
export const CodeEditor = (props) => {
    const { allowFullScreen, beautify, className, controlledFullScreenMode, getThemeFromEnvironment, height, language, lineWrapping, lint, onFullScreenToggle, options, oneLiner, readOnly, showLineNumbers, smart, sourceCode, tabSize, theme, translations, triggerFullScreen, withClipboardCopy, ["data-testid"]: testId } = props, rest = __rest(props, ["allowFullScreen", "beautify", "className", "controlledFullScreenMode", "getThemeFromEnvironment", "height", "language", "lineWrapping", "lint", "onFullScreenToggle", "options", "oneLiner", "readOnly", "showLineNumbers", "smart", "sourceCode", "tabSize", "theme", "translations", "triggerFullScreen", "withClipboardCopy", "data-testid"]);
    const [editorInstance, setEditorInstance] = useState(undefined);
    const [copyToClipboardIcon, setCopyToClipboardIcon] = useState(CopyIcon);
    const [fullScreenToggleIcon, setFullScreenToggleIcon] = useState(MaximizeIcon);
    const [isEditorFullScreen, setIsEditorFullScreen] = useState(false);
    const [darkMode, setDarkMode] = useState(false);
    /**
     * Set the internal full screen state based on the externally provided state.
     */
    useEffect(() => {
        if (triggerFullScreen === undefined) {
            return;
        }
        if (triggerFullScreen) {
            setFullScreenToggleIcon(MinimizeIcon);
        }
        else {
            setFullScreenToggleIcon(MaximizeIcon);
        }
        setIsEditorFullScreen(triggerFullScreen);
        onFullScreenToggle(triggerFullScreen);
    }, [triggerFullScreen]);
    /**
     * Gets the browser color scheme so that the color scheme of the textarea can be decided.
     */
    useEffect(() => {
        // If `getThemeFromEnvironment` is false or `matchMedia` is not supported, fallback to dark theme.
        if (!getThemeFromEnvironment || !window.matchMedia) {
            setDarkMode(true);
            return;
        }
        if (window.matchMedia("(prefers-color-scheme:dark)").matches) {
            setDarkMode(true);
        }
        const callback = (e) => {
            if (e.matches) {
                setDarkMode(true);
                return;
            }
            setDarkMode(false);
        };
        try {
            window.matchMedia("(prefers-color-scheme:dark)")
                .addEventListener("change", (e) => callback(e));
        }
        catch (error) {
            try {
                // Older versions of Safari doesn't support `addEventListener`.
                window.matchMedia("(prefers-color-scheme:dark)")
                    .addListener((e) => callback(e));
            }
            catch (error) {
                // Fallback to dark if everything fails.
                setDarkMode(true);
            }
        }
        // Housekeeping. Remove the listeners.
        return () => {
            try {
                window.matchMedia("(prefers-color-scheme:dark)")
                    .removeEventListener("change", (e) => callback(e));
            }
            catch (parentError) {
                try {
                    // Older versions of Safari doesn't support `addEventListener`.
                    window.matchMedia("(prefers-color-scheme:dark)")
                        .removeListener((e) => callback(e));
                }
                catch (error) {
                    // TODO: Rather than silently failing, add debug logs here.
                }
            }
        };
    }, [getThemeFromEnvironment]);
    /**
     * Resolves the language mode.
     *
     * @param {string} language - Selected language.
     * @return {object} Resolved mode.
     */
    const resolveMode = (language) => {
        if (!language) {
            throw new Error("Please define a language.");
        }
        return {
            json: language === "json",
            name: (language === "json" || language === "typescript") ? "javascript" : language,
            statementIndent: 4,
            typescript: language === "typescript"
        };
    };
    /**
     * Resolves the editor theme.
     *
     * @return {"material" | "default"} Resolved mode.
     */
    const resolveTheme = () => {
        if (getThemeFromEnvironment) {
            return (darkMode ? "material" : "default");
        }
        if (!(theme === "dark" || theme === "light")) {
            throw new Error("Please select a supported theme. Only `dark` and `light` are supported at the moment.");
        }
        return (theme === "dark" ? "material" : "default");
    };
    /**
     * Beautifies the source code.
     *
     * @return {string} Beautified source code snippet.
     */
    const beautifyCode = () => {
        let code = sourceCode;
        if (code instanceof Array) {
            code = code.join("");
        }
        if (language === "javascript") {
            return JSBeautify(code, { indent_size: tabSize, space_in_empty_paren: true });
        }
        return code;
    };
    /**
     * Handles clipboard copy event internally.
     */
    const handleCopyToClipboard = () => {
        CommonUtils.copyTextToClipboard(editorInstance.doc.getValue())
            .then(() => {
            setCopyToClipboardIcon(CheckIcon);
            setTimeout(() => {
                setCopyToClipboardIcon(CopyIcon);
            }, 1000);
        });
    };
    /**
     * Handles Full Screen mode toggle event event.
     */
    const handleFullScreenToggle = () => {
        if (!isEditorFullScreen) {
            setFullScreenToggleIcon(MinimizeIcon);
        }
        else {
            setFullScreenToggleIcon(MaximizeIcon);
        }
        onFullScreenToggle(!isEditorFullScreen);
        setIsEditorFullScreen(!isEditorFullScreen);
    };
    const classes = classNames("code-editor", {
        "dark": resolveTheme() === "material",
        "light": resolveTheme() === "default",
        "one-liner": oneLiner,
        "with-actions": withClipboardCopy || allowFullScreen
    }, className);
    const fullScreenWrapperClasses = classNames("code-editor-fullscreen-wrapper", { theme });
    /**
     * Render inner content.
     *
     * @return {React.ReactElement}
     */
    const renderContent = () => {
        // Renders the full screen toggle.
        const renderFullScreenToggle = () => (React.createElement("div", { className: "editor-action", onClick: handleFullScreenToggle },
            React.createElement(Tooltip, { compact: true, trigger: (React.createElement("div", null,
                    React.createElement(GenericIcon, { hoverable: true, size: "mini", transparent: true, icon: fullScreenToggleIcon }))), content: isEditorFullScreen
                    ? (translations === null || translations === void 0 ? void 0 : translations.exitFullScreen) || "Exit full-Screen"
                    : (translations === null || translations === void 0 ? void 0 : translations.goFullScreen) || "Go full-Screen", size: "mini" })));
        return (React.createElement("div", { className: classes },
            React.createElement("div", { className: "editor-actions" },
                allowFullScreen && (
                // If the full screen mode trigger is handled from outside, we don't need to show the
                // embedded `Go full-screen` button.
                !controlledFullScreenMode
                    ? isEditorFullScreen
                        ? renderFullScreenToggle()
                        : null
                    : renderFullScreenToggle()),
                withClipboardCopy && (React.createElement("div", { className: "editor-action", onClick: handleCopyToClipboard },
                    React.createElement(Tooltip, { compact: true, trigger: (React.createElement("div", null,
                            React.createElement(GenericIcon, { hoverable: true, size: "mini", transparent: true, icon: copyToClipboardIcon }))), content: (translations === null || translations === void 0 ? void 0 : translations.copyCode) || "Copy to clipboard", size: "mini" })))),
            React.createElement(CodeMirror, Object.assign({}, rest, { value: beautify ? beautifyCode() : sourceCode, editorDidMount: (editor, ...args) => {
                    if (height) {
                        editor.setSize("", height);
                    }
                    if (oneLiner) {
                        editor.setSize("", "100%");
                    }
                    setEditorInstance(editor);
                    rest.editorDidMount && rest.editorDidMount(editor, ...args);
                }, options: Object.assign({ autoCloseBrackets: smart, autoCloseTags: smart, extraKeys: smart ? { "Ctrl-Space": "autocomplete" } : {}, gutters: ["note-gutter", "CodeMirror-linenumbers", "CodeMirror-lint-markers"], indentUnit: tabSize, lineNumbers: !oneLiner
                        ? showLineNumbers
                        : false, lineWrapping,
                    lint, matchBrackets: smart, matchTags: smart, mode: (options === null || options === void 0 ? void 0 : options.mode) ? options.mode : resolveMode(language), readOnly,
                    tabSize, theme: resolveTheme() }, options), "data-testid": testId }))));
    };
    return ((allowFullScreen && isEditorFullScreen)
        ? (React.createElement(Modal, { open: true, size: "fullscreen", className: fullScreenWrapperClasses },
            React.createElement(Modal.Content, { className: "editor-content", scrolling: true }, renderContent())))
        : renderContent());
};
/**
 * Default props for the code editor component.
 */
CodeEditor.defaultProps = {
    allowFullScreen: false,
    controlledFullScreenMode: true,
    "data-testid": "code-editor",
    language: "javascript",
    lineWrapping: true,
    lint: false,
    readOnly: false,
    showLineNumbers: true,
    smart: false,
    tabSize: 4,
    theme: "dark"
};
//# sourceMappingURL=code-editor.js.map