/**
 * Copyright (c) 2021, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the 'License'); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * 'AS IS' BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import React, { useEffect, useRef, useState } from "react";
import { Button, Divider, Icon, Message, Segment, Tab, TextArea } from "semantic-ui-react";
import { GenericIcon } from "../icon";
import { KJUR, X509 } from "jsrsasign";
import * as forge from "node-forge";
import { CertificateManagementUtils } from "@wso2is/core/utils";
// TODO: Move polyfills to a generalized module.
// This is a polyfill to support `File.arrayBuffer()` in Safari and IE.
if ("File" in self)
    File.prototype.arrayBuffer = File.prototype.arrayBuffer || poly;
Blob.prototype.arrayBuffer = Blob.prototype.arrayBuffer || poly;
function poly() {
    // this: File or Blob
    return new Promise((resolve) => {
        const fr = new FileReader();
        fr.onload = () => {
            resolve(fr.result);
        };
        fr.readAsArrayBuffer(this);
    });
}
// Component constants
const FIRST_FILE_INDEX = 0;
const FIRST_TAB_INDEX = 0;
const SECOND_TAB_INDEX = 1;
const EMPTY_STRING = "";
export const FilePicker = (props) => {
    const { fileStrategy, onChange, dropzoneText, pasteAreaPlaceholderText, uploadButtonText, icon, placeholderIcon, file: initialFile, pastedContent: initialPastedContent, normalizeStateOnRemoveOperations, emptyFileError, emptyFileErrorMsg, hidePasteOption } = props;
    // Document queries
    const hiddenFileUploadInput = useRef(null);
    // Functional State
    const [dark, setDark] = useState(false);
    const [activeIndex, setActiveIndex] = useState(FIRST_TAB_INDEX);
    const [dragOver, setDragOver] = useState(false);
    // Behavioural State
    const [file, setFile] = useState(initialFile);
    const [pastedContent, setPastedContent] = useState(initialPastedContent);
    const [serializedData, setSerializedData] = useState(null);
    const [hasError, setHasError] = useState(false);
    const [errorMessage, setErrorMessage] = useState(null);
    const [pasteFieldTouched, setPasteFieldTouched] = useState(false);
    const [fileFieldTouched, setFileFieldTouched] = useState(false);
    // Hooks
    useEffect(() => {
        if (emptyFileError) {
            setHasError(true);
            setErrorMessage(emptyFileErrorMsg ? emptyFileErrorMsg : "Please add a file");
        }
    }, [emptyFileError]);
    useEffect(() => {
        if (initialFile) {
            addFileToState(initialFile);
            setActiveIndex(FIRST_TAB_INDEX);
            return;
        }
        if (initialPastedContent) {
            addPastedDataToState(initialPastedContent);
            setActiveIndex(SECOND_TAB_INDEX);
            return;
        }
    }, [initialFile, initialPastedContent]);
    useEffect(() => {
        // Query the preferred color scheme.
        const mql = window.matchMedia("(prefers-color-scheme:dark)");
        // Check and set the dark mode initially.
        if (mql === null || mql === void 0 ? void 0 : mql.matches)
            setDark(true);
        // Callback for triggering the same for change events.
        const triggerColorScheme = (event) => {
            var _a;
            setDark((_a = event.matches) !== null && _a !== void 0 ? _a : false);
        };
        // Check to see if match media API is available with the browser.
        if (mql === null || mql === void 0 ? void 0 : mql.addEventListener) {
            mql.addEventListener("change", triggerColorScheme);
        }
        // Cleanup logic.
        return () => {
            if (mql === null || mql === void 0 ? void 0 : mql.addEventListener) {
                mql.removeEventListener("change", triggerColorScheme);
            }
        };
    }, []);
    useEffect(() => {
        if (onChange) {
            onChange({
                file: file,
                pastedContent: pastedContent,
                serialized: serializedData,
                /**
                 * In order result to be valid. Form should not contain errors,
                 * there should be some serialized content, and one of the used
                 * input methods. !! is equivalent to val !== null or Boolean(val)
                 */
                valid: !hasError && !!serializedData && (!!file || !!pastedContent)
            });
        }
    }, [serializedData, errorMessage, file, pastedContent]);
    // Functional logic that used by event handlers to
    // update the state of the picker.
    const validate = (data) => __awaiter(void 0, void 0, void 0, function* () {
        try {
            yield fileStrategy.validate(data);
            setHasError(false);
            setErrorMessage(null);
            return true;
        }
        catch (error) {
            // Ideally the validation result must be type
            // ValidationResult. However, if for some reason
            // developer overrides it and just send a string
            // or a Error object we need to gracefully handle
            // the result.
            readDynamicErrorAndSetToState(error);
            return false;
        }
    });
    const readDynamicErrorAndSetToState = (error) => {
        var _a;
        setHasError(true);
        if (error) {
            if (typeof error === "string") {
                setErrorMessage(error);
            }
            else if (error instanceof Error) {
                setErrorMessage(error === null || error === void 0 ? void 0 : error.message);
            }
            else if (isTypeValidationResult(error)) {
                setErrorMessage((_a = error.errorMessage) !== null && _a !== void 0 ? _a : EMPTY_STRING);
            }
            else {
                setErrorMessage("Your input has unknown errors.");
            }
        }
    };
    // TODO: As a improvement add multiple tabs option. The implementation
    //       of that is a little tricky. We can achieve the behaviour using
    //       the same strategy pattern but it MUST not complex the concrete
    //       implementation.
    // TODO: As a improvement implement the context component (this)
    //       to support multiple file upload/attach strategy. If attaching
    //       multiple then tabs Paste option should be removed dynamically.
    // TODO: As a improvement compare newly added file or the
    //       pasted content with the previous state and set it
    //       to the state only if its a new change.
    const addFileToState = (file) => __awaiter(void 0, void 0, void 0, function* () {
        if (yield validate(file)) {
            setFile(file);
            try {
                setSerializedData(yield fileStrategy.serialize(file));
            }
            catch (error) {
                readDynamicErrorAndSetToState(error);
            }
        }
    });
    const addPastedDataToState = (text) => __awaiter(void 0, void 0, void 0, function* () {
        if (yield validate(text)) {
            try {
                setSerializedData(yield fileStrategy.serialize(text));
            }
            catch (error) {
                readDynamicErrorAndSetToState(error);
            }
        }
        setPastedContent(text);
    });
    // Events that takes care of the drag drop input. User drops
    // a file onto the drop area.
    const handleOnDrop = (event) => {
        if (event) {
            event.preventDefault();
            event.stopPropagation();
        }
        setDragOver(false);
        if (event.dataTransfer.files[FIRST_FILE_INDEX]) {
            const file = event.dataTransfer.files[FIRST_FILE_INDEX];
            if (file) {
                addFileToState(file);
                setFileFieldTouched(true);
                setPasteFieldTouched(false);
            }
            else {
                setFile(null);
            }
        }
    };
    const handleOnDragOver = (event) => {
        if (event) {
            event.preventDefault();
            event.stopPropagation();
        }
        setDragOver(true);
    };
    const handleOnDragLeave = (event) => {
        if (event) {
            event.preventDefault();
            event.stopPropagation();
        }
        setDragOver(false);
    };
    // Events that takes care of the manual input. Where the user
    // clicks the upload button and related file input field changes.
    const handleOnFileInputChange = (event) => {
        const file = event.target.files[FIRST_FILE_INDEX];
        event.target.value = null;
        addFileToState(file);
        setFileFieldTouched(true);
        setPasteFieldTouched(false);
    };
    const handleOnUploadButtonClick = (event) => {
        if (event) {
            event.preventDefault();
            event.stopPropagation();
        }
        hiddenFileUploadInput.current.click();
    };
    /**
     * Once the selected file gets removed. We need to re-evaluate
     * the pasted content properly to send the serialized value to
     * the parent component. This is because the values can differ
     * in both the inputs and can lead to data inconsistencies if
     * not handled properly.
     */
    const normalizeSerializationOnFileRemoval = () => {
        if (pastedContent) {
            addPastedDataToState(pastedContent);
            setActiveIndex(SECOND_TAB_INDEX);
        }
        else {
            setSerializedData(null);
        }
    };
    /**
     * This should be used when the user removed the entire string
     * from the textarea. Same as above {@link normalizeSerializationOnFileRemoval}
     * scenario we need to re-evaluate the selected file properly and set
     * the serialized value to the parent component since two inputs can
     * have different values simultaneously.
     */
    const normalizeSerializationOnPastedTextClear = () => {
        if (file) {
            addFileToState(file);
            setActiveIndex(FIRST_TAB_INDEX);
        }
        else {
            setSerializedData(null);
        }
    };
    // UI elements and render()
    const dragOption = {
        menuItem: "Upload",
        render: () => {
            const previewPlaceholder = (React.createElement(Segment, { placeholder: true },
                React.createElement(Segment, { textAlign: "center", basic: true },
                    React.createElement(GenericIcon, { inline: true, transparent: true, size: "auto", icon: placeholderIcon }),
                    React.createElement("p", null,
                        "You have selected ",
                        React.createElement("em", { className: "file-name" }, file === null || file === void 0 ? void 0 : file.name),
                        " file"),
                    React.createElement("p", null, "Not this file?"),
                    React.createElement(Button, { inverted: true, color: "red", onClick: () => {
                            setFile(null);
                            setFileFieldTouched(false);
                            setErrorMessage(null);
                            setHasError(false);
                            if (normalizeStateOnRemoveOperations) {
                                normalizeSerializationOnFileRemoval();
                            }
                        } },
                        React.createElement(Icon, { name: 'trash alternate' }),
                        " Remove"))));
            const dragDropArea = (React.createElement("div", { onDrop: handleOnDrop, onDragOver: handleOnDragOver, onDragLeave: handleOnDragLeave, "data-testid": `generic-file-upload-dropzone` },
                React.createElement(Segment, { placeholder: true, className: `drop-zone ${dragOver && "drag-over"}` },
                    React.createElement("div", { className: "certificate-upload-placeholder" },
                        React.createElement(GenericIcon, { inline: true, transparent: true, size: "mini", icon: icon }),
                        React.createElement("p", { className: "description" }, dropzoneText),
                        React.createElement("p", { className: "description" }, "\u2013 or \u2013")),
                    React.createElement(Button, { basic: true, primary: true, onClick: handleOnUploadButtonClick }, uploadButtonText))));
            return (file) ? previewPlaceholder : dragDropArea;
        }
    };
    const pasteOption = {
        menuItem: "Paste",
        render: () => {
            return (React.createElement(TextArea, { rows: 10, placeholder: pasteAreaPlaceholderText !== null && pasteAreaPlaceholderText !== void 0 ? pasteAreaPlaceholderText : "Paste your content in this area...", value: pastedContent, onChange: (event) => {
                    var _a;
                    if (event) {
                        event.preventDefault();
                        event.stopPropagation();
                    }
                    if ((_a = event === null || event === void 0 ? void 0 : event.target) === null || _a === void 0 ? void 0 : _a.value) {
                        addPastedDataToState(event.target.value);
                    }
                    else {
                        /**
                         * CLEAR OPERATION:
                         * This block executes when the user have cleared all the content
                         * from the textarea and we will evaluate it as a empty string.
                         */
                        setPastedContent(EMPTY_STRING);
                        /**
                         * If the string is empty we can't show error messages below the
                         * fields. Because, it's like the initial state.
                         */
                        setErrorMessage(null);
                        setHasError(false);
                        if (normalizeStateOnRemoveOperations) {
                            normalizeSerializationOnPastedTextClear();
                        }
                    }
                    setPasteFieldTouched(true);
                    setFileFieldTouched(false);
                }, spellCheck: false, className: `certificate-editor ${dark ? "dark" : "light"}` }));
        }
    };
    return (React.createElement(React.Fragment, null,
        React.createElement("input", { hidden: true, ref: hiddenFileUploadInput, type: "file", accept: fileStrategy.mimeTypes.join(","), onChange: handleOnFileInputChange }),
        !hidePasteOption ?
            React.createElement(Tab, { className: "tabs resource-tabs", menu: { pointing: true, secondary: true }, panes: [dragOption, pasteOption], activeIndex: activeIndex, onTabChange: (event, { activeIndex }) => {
                    const index = parseInt(activeIndex.toString());
                    setActiveIndex(index);
                    if (index === FIRST_TAB_INDEX && fileFieldTouched && !pastedContent) {
                        validate(file);
                    }
                    else if (index === SECOND_TAB_INDEX && pasteFieldTouched && !file) {
                        validate(pastedContent);
                    }
                } }) :
            (React.createElement(React.Fragment, null,
                React.createElement(Divider, { hidden: true }),
                dragOption.render())),
        React.createElement(Message, { error: true, visible: hasError, "data-testid": `file-picker-error-message` },
            React.createElement(Icon, { name: 'file' }),
            errorMessage)));
};
FilePicker.defaultProps = {
    hidePasteOption: false
};
function isTypeValidationResult(obj) {
    return obj && ('valid' in obj || 'errorMessage' in obj);
}
// Concrete strategies implementations.
export class DefaultFileStrategy {
    constructor() {
        this.mimeTypes = ["*"];
    }
    serialize(data) {
        return __awaiter(this, void 0, void 0, function* () {
            return Promise.resolve("");
        });
    }
    validate(data) {
        return __awaiter(this, void 0, void 0, function* () {
            return Promise.resolve({ valid: true });
        });
    }
}
export class XMLFileStrategy {
    constructor(mimeTypes) {
        if (!mimeTypes || mimeTypes.length === 0)
            this.mimeTypes = XMLFileStrategy.DEFAULT_MIMES;
        else
            this.mimeTypes = mimeTypes;
    }
    serialize(data) {
        return __awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => {
                if (!data) {
                    reject({ valid: false });
                    return;
                }
                if (data instanceof File) {
                    const reader = new FileReader();
                    reader.readAsText(data, XMLFileStrategy.ENCODING);
                    reader.onload = () => {
                        this.parseXML(reader.result).then((rawXML) => {
                            resolve(btoa(rawXML));
                        }).catch((error) => {
                            reject({
                                valid: false,
                                errorMessage: error !== null && error !== void 0 ? error : "XML file content is invalid"
                            });
                        });
                    };
                }
                else {
                    this.parseXML(data).then((rawXML) => {
                        resolve(btoa(rawXML));
                    }).catch((error) => {
                        reject({
                            valid: false,
                            errorMessage: error !== null && error !== void 0 ? error : "XML string is invalid"
                        });
                    });
                }
            });
        });
    }
    validate(data) {
        return __awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => {
                if (data instanceof File) {
                    const expected = XMLFileStrategy.MAX_FILE_SIZE * XMLFileStrategy.MEGABYTE;
                    if (data.size > expected) {
                        reject({
                            valid: false,
                            errorMessage: `File exceeds max size of ${XMLFileStrategy.MAX_FILE_SIZE} MB`
                        });
                    }
                    const reader = new FileReader();
                    reader.readAsText(data, XMLFileStrategy.ENCODING);
                    reader.onload = () => {
                        this.parseXML(reader.result).then(() => {
                            resolve({ valid: true });
                        }).catch((error) => {
                            reject({
                                valid: false,
                                errorMessage: error !== null && error !== void 0 ? error : "XML file has errors"
                            });
                        });
                    };
                }
                else {
                    this.parseXML(data).then(() => {
                        resolve({ valid: true });
                    }).catch((error) => {
                        reject({
                            valid: false,
                            errorMessage: error !== null && error !== void 0 ? error : "XML string has errors"
                        });
                    });
                }
            });
        });
    }
    parseXML(xml) {
        return __awaiter(this, void 0, void 0, function* () {
            const domParser = new DOMParser();
            // If the xml is a instance of ArrayBuffer then first
            // convert it to a primitive string.
            if (xml instanceof ArrayBuffer) {
                // TODO: Add the polyfills for IE and older browsers.
                // https://github.com/inexorabletash/text-encoding
                const enc = new TextDecoder(XMLFileStrategy.ENCODING);
                const arr = new Uint8Array(xml);
                xml = enc.decode(arr);
            }
            // Below this point we can ensure that xml is
            // a string type and proceed to parse.
            const dom = domParser.parseFromString(xml, 'text/xml');
            if (dom.getElementsByTagName("parsererror").length > 0) {
                throw "Error while parsing XML file";
            }
            return xml;
        });
    }
}
XMLFileStrategy.ENCODING = "UTF-8";
XMLFileStrategy.DEFAULT_MIMES = [
    "text/xml",
    "application/xml"
];
XMLFileStrategy.MEGABYTE = 1e+6;
XMLFileStrategy.MAX_FILE_SIZE = 3 * XMLFileStrategy.MEGABYTE;
export class CertFileStrategy {
    constructor(mimeTypes) {
        if (!mimeTypes || mimeTypes.length === 0)
            this.mimeTypes = CertFileStrategy.DEFAULT_MIMES;
        else
            this.mimeTypes = mimeTypes;
    }
    serialize(data) {
        if (data instanceof File) {
            return this.convertFromFile(data);
        }
        else {
            return this.convertFromString(data);
        }
    }
    validate(data) {
        return new Promise((resolve, reject) => {
            if (data instanceof File) {
                this.convertFromFile(data).then(() => {
                    resolve({ valid: true });
                }).catch(() => {
                    reject({
                        valid: false,
                        errorMessage: "Invalid certificate file. " +
                            "Please use one of the following formats " +
                            this.mimeTypes.join(",")
                    });
                });
            }
            else {
                this.convertFromString(data).then(() => {
                    resolve({ valid: true });
                }).catch(() => {
                    reject({
                        valid: false,
                        errorMessage: "Invalid certificate pem string."
                    });
                });
            }
        });
    }
    convertFromString(text) {
        return new Promise((resolve, reject) => {
            try {
                const certificateForge = new X509().readCertFromPEM(text);
                resolve({
                    forgeObject: certificateForge,
                    pemStripped: CertificateManagementUtils.stripPem(text),
                    pem: text
                });
            }
            catch (_a) {
                try {
                    const pemValue = CertificateManagementUtils.enclosePem(text);
                    const certificate = forge.pki.certificateFromPem(pemValue);
                    const pem = forge.pki.certificateToPem(certificate);
                    const certificateForge = new X509();
                    certificateForge.readCertPEM(pem);
                    resolve({
                        forgeObject: certificateForge,
                        pemStripped: CertificateManagementUtils.stripPem(text),
                        pem: text
                    });
                }
                catch (error) {
                    reject("Failed to decode pem certificate data.");
                }
            }
        });
    }
    convertFromFile(file) {
        return new Promise((resolve, reject) => {
            file.arrayBuffer().then((buf) => {
                try {
                    const hex = Array.prototype.map.call(new Uint8Array(buf), (x) => ("00" + x.toString(16)).slice(-2)).join("");
                    const cert = new X509();
                    cert.readCertHex(hex);
                    const certificate = new KJUR.asn1.x509.Certificate(cert.getParam());
                    const pem = certificate.getPEM();
                    const pemStripped = CertificateManagementUtils.stripPem(pem);
                    resolve({
                        forgeObject: cert,
                        pem: pem,
                        pemStripped: pemStripped
                    });
                }
                catch (_a) {
                    const byteString = forge.util.createBuffer(buf);
                    try {
                        const asn1 = forge.asn1.fromDer(byteString);
                        const certificate = forge.pki.certificateFromAsn1(asn1);
                        const pem = forge.pki.certificateToPem(certificate);
                        const cert = new X509();
                        cert.readCertPEM(pem);
                        const pemStripped = CertificateManagementUtils.stripPem(pem);
                        resolve({
                            forgeObject: cert,
                            pem: pem,
                            pemStripped: pemStripped
                        });
                    }
                    catch (_b) {
                        try {
                            const cert = new X509();
                            cert.readCertPEM(byteString.data);
                            const certificate = new KJUR.asn1.x509.Certificate(cert.getParam());
                            const pem = certificate.getPEM();
                            const pemStripped = CertificateManagementUtils.stripPem(pem);
                            resolve({
                                forgeObject: cert,
                                pem: pem,
                                pemStripped: pemStripped
                            });
                        }
                        catch (_c) {
                            try {
                                const certificate = forge.pki.certificateFromPem(byteString.data);
                                const pem = forge.pki.certificateToPem(certificate);
                                const cert = new X509();
                                cert.readCertPEM(pem);
                                const pemStripped = CertificateManagementUtils.stripPem(pem);
                                resolve({
                                    forgeObject: cert,
                                    pem: pem,
                                    pemStripped: pemStripped
                                });
                            }
                            catch (_d) {
                                reject({
                                    valid: false,
                                    errorMessage: "Certificate file has errors."
                                });
                            }
                        }
                    }
                }
            }).catch((error) => {
                reject(error);
            });
        });
    }
}
CertFileStrategy.DEFAULT_MIMES = [
    ".pem", ".cer", ".crt", ".cert"
];
//# sourceMappingURL=file-picker.js.map