"use strict";

require("core-js/modules/es.symbol");

require("core-js/modules/es.symbol.description");

require("core-js/modules/es.array.map");

require("core-js/modules/es.function.name");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getPropDefFactory = exports.unknownFactory = exports.flowFactory = exports.tsFactory = exports.javaScriptFactory = void 0;

var _lodash = require("lodash");

var _types = require("./types");

var _utils = require("../utils");

var _createPropDef = require("./flow/createPropDef");

var _defaultValue = require("./utils/defaultValue");

var _createPropDef2 = require("./typeScript/createPropDef");

function createType(type) {
  // A type could be null if a defaultProp has been provided without a type definition.
  return !(0, _lodash.isNil)(type) ? (0, _utils.createSummaryValue)(type.name) : null;
}

function createDefaultValue(defaultValue) {
  if (!(0, _lodash.isNil)(defaultValue)) {
    var value = defaultValue.value;

    if (!(0, _defaultValue.isDefaultValueBlacklisted)(value)) {
      return (0, _utils.createSummaryValue)(value);
    }
  }

  return null;
}

function createBasicPropDef(name, type, docgenInfo) {
  var description = docgenInfo.description,
      required = docgenInfo.required,
      defaultValue = docgenInfo.defaultValue;
  return {
    name: name,
    type: createType(type),
    required: required,
    description: description,
    defaultValue: createDefaultValue(defaultValue)
  };
}

function applyJsDocResult(propDef, jsDocParsingResult) {
  if (jsDocParsingResult.includesJsDoc) {
    var description = jsDocParsingResult.description,
        extractedTags = jsDocParsingResult.extractedTags;

    if (!(0, _lodash.isNil)(description)) {
      // eslint-disable-next-line no-param-reassign
      propDef.description = jsDocParsingResult.description;
    }

    var hasParams = !(0, _lodash.isNil)(extractedTags.params);
    var hasReturns = !(0, _lodash.isNil)(extractedTags.returns) && !(0, _lodash.isNil)(extractedTags.returns.type);

    if (hasParams || hasReturns) {
      // eslint-disable-next-line no-param-reassign
      propDef.jsDocTags = {
        params: hasParams && extractedTags.params.map(function (x) {
          return {
            name: x.getPrettyName(),
            description: x.description
          };
        }),
        returns: hasReturns && {
          description: extractedTags.returns.description
        }
      };
    }
  }

  return propDef;
}

var javaScriptFactory = function javaScriptFactory(propName, docgenInfo, jsDocParsingResult) {
  var propDef = createBasicPropDef(propName, docgenInfo.type, docgenInfo);
  return applyJsDocResult(propDef, jsDocParsingResult);
};

exports.javaScriptFactory = javaScriptFactory;

var tsFactory = function tsFactory(propName, docgenInfo, jsDocParsingResult) {
  var propDef = (0, _createPropDef2.createTsPropDef)(propName, docgenInfo);
  return applyJsDocResult(propDef, jsDocParsingResult);
};

exports.tsFactory = tsFactory;

var flowFactory = function flowFactory(propName, docgenInfo, jsDocParsingResult) {
  var propDef = (0, _createPropDef.createFlowPropDef)(propName, docgenInfo);
  return applyJsDocResult(propDef, jsDocParsingResult);
};

exports.flowFactory = flowFactory;

var unknownFactory = function unknownFactory(propName, docgenInfo, jsDocParsingResult) {
  var propDef = createBasicPropDef(propName, {
    name: 'unknown'
  }, docgenInfo);
  return applyJsDocResult(propDef, jsDocParsingResult);
};

exports.unknownFactory = unknownFactory;

var getPropDefFactory = function getPropDefFactory(typeSystem) {
  switch (typeSystem) {
    case _types.TypeSystem.JAVASCRIPT:
      return javaScriptFactory;

    case _types.TypeSystem.TYPESCRIPT:
      return tsFactory;

    case _types.TypeSystem.FLOW:
      return flowFactory;

    default:
      return unknownFactory;
  }
};

exports.getPropDefFactory = getPropDefFactory;