/**
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import WorkerFile from "web-worker:./oidc.worker.ts";
import { API_CALL, API_CALL_ALL, AUTHORIZATION_CODE, AUTH_REQUIRED, CUSTOM_GRANT, END_USER_SESSION, GET_DECODED_ID_TOKEN, GET_SERVICE_ENDPOINTS, GET_USER_INFO, INIT, LOGOUT_URL, PKCE_CODE_VERIFIER, REQUEST_ERROR, REQUEST_FINISH, REQUEST_START, REQUEST_SUCCESS, ResponseMode, SESSION_STATE, SIGNED_IN, SIGN_IN } from "../constants";
import { getAuthorizationCode } from "../utils";
/**
 * This is a singleton class that allows authentication using the OAuth 2.0 protocol.
 *
 * - To get an instance of this class, use the `getInstance()` method.
 * - To initiate the authentication flow, follow the following procedure:
 * 	1. Initialize the object by calling the `initialize(config)` method. You will have to
 * 		pass a config object as an argument. To know more, checkout the `initialize()` method.
 * 	2. To get *the authorization code* from the callback URL and continue the authentication flow,
 * 		call the `listenForAuthCode()` method. In an SPA, this should be called in the page rendered by the
 * 		callback URL.
 * 	2. Kick off the authentication flow by calling the `signIn()` method.
 *
 * @example
 * Example:
 *
 * ```
 * 	var oAuth = Wso2OAuth.OAuth.getInstance();
 * 		oAuth.initialize({
 * 			clientHost: "https://localhost:9443/",
 * 			clientID: "70gph7I55ioGi5FqhLPz8JvxZCEa",
 * 			serverOrigin: "https://localhost:9443",
 * 			baseUrls: ["https://localhost:9443"],
 * 			origin: origin,
 * 			callbackURL: "https://localhost:9443/worker",
 * 			enablePKCE: true,
 * 			scope: ["SYSTEM", "openid"],
 * 		}).then(response=>{
 * 			console.log(response);
 *
 *			oAuth.listenForAuthCode().then(response=>{
 * 				console.log(response);
 * 			}).catch(error=>{
 * 				console.error(error)
 * 			});
 * 		}).catch(error=>{
 * 			console.error(error)
 * 		});
 *
 *
 * ```
 */
export const WebWorkerClient = (() => {
    /**
     * The private member variable that holds the reference to the web worker.
     */
    let worker;
    /**
     * The private member variable that holds the instance of this class.
     */
    let instance;
    /**
     * The private boolean member variable that specifies if the `initialize()` method has been called or not.
     */
    let initialized = false;
    /**
     * The private boolean member variable that specifies if the user is signed in or not.
     */
    let signedIn = false;
    /**
     * HttpClient handlers
     */
    let httpClientHandlers;
    /**
     * API request time out.
     */
    let requestTimeout;
    /**
     * @private
     *
     * @returns {string} Removes the path parameters and returns the URL.
     *
     * @example
     * `https://localhost:9443?code=g43dhkj243wghdgwedew65&session=34khkg2g`
     * will be stripped to `https://localhost:9443`
     */
    const removeAuthorizationCode = () => {
        const url = location.href;
        return url.replace(/\?code=.*$/, "");
    };
    /**
     * @private
     *
     * Checks if the authorization code is present in the URL or not.
     *
     * @returns {boolean} Authorization code presence status.
     */
    const hasAuthorizationCode = () => {
        return !!getAuthorizationCode();
    };
    /**
     * @private
     *
     * Sends a message to the web worker and returns the response.
     *
     * T - Request data type.
     *
     * R - response data type.
     *
     * @param {Message} message - The message object
     * @param {number} timeout The number of seconds to wait before timing the request out. - optional
     *
     * @returns {Promise<R>} A promise that resolves with the obtained data.
     */
    const communicate = (message) => {
        const channel = new MessageChannel();
        worker.postMessage(message, [channel.port2]);
        return new Promise((resolve, reject) => {
            const timer = setTimeout(() => {
                reject("Operation timed out");
            }, requestTimeout);
            return (channel.port1.onmessage = ({ data }) => {
                clearTimeout(timer);
                if (data === null || data === void 0 ? void 0 : data.success) {
                    const responseData = JSON.parse(data === null || data === void 0 ? void 0 : data.data);
                    if (data === null || data === void 0 ? void 0 : data.blob) {
                        responseData.data = data === null || data === void 0 ? void 0 : data.blob;
                    }
                    resolve(responseData);
                }
                else {
                    reject(JSON.parse(data.error));
                }
            });
        });
    };
    /**
     * Allows using custom grant types.
     *
     * @param {CustomGrantRequestParams} requestParams Request Parameters.
     *
     * @returns {Promise<AxiosResponse|boolean>} A promise that resolves with a boolean value or the request
     * response if the the `returnResponse` attribute in the `requestParams` object is set to `true`.
     */
    const customGrant = (requestParams) => {
        if (!initialized) {
            return Promise.reject("The object has not been initialized yet");
        }
        if (!signedIn && requestParams.signInRequired) {
            return Promise.reject("You have not signed in yet");
        }
        const message = {
            data: requestParams,
            type: CUSTOM_GRANT
        };
        return communicate(message)
            .then((response) => {
            return Promise.resolve(response);
        })
            .catch((error) => {
            return Promise.reject(error);
        });
    };
    /**
     *
     * Send the API request to the web worker and returns the response.
     *
     * @param {AxiosRequestConfig} config The Axios Request Config object
     *
     * @returns {Promise<AxiosResponse>} A promise that resolves with the response data.
     */
    const httpRequest = (config) => {
        if (!initialized) {
            return Promise.reject("The object has not been initialized yet");
        }
        if (!signedIn) {
            return Promise.reject("You have not signed in yet");
        }
        const message = {
            data: config,
            type: API_CALL
        };
        return communicate(message)
            .then((response) => {
            return Promise.resolve(response);
        })
            .catch((error) => {
            return Promise.reject(error);
        });
    };
    /**
     *
     * Send multiple API requests to the web worker and returns the response.
     * Similar `axios.spread` in functionality.
     *
     * @param {AxiosRequestConfig[]} configs - The Axios Request Config object
     *
     * @returns {Promise<AxiosResponse<T>[]>} A promise that resolves with the response data.
     */
    const httpRequestAll = (configs) => {
        if (!initialized) {
            return Promise.reject("The object has not been initialized yet");
        }
        if (!signedIn) {
            return Promise.reject("You have not signed in yet");
        }
        const message = {
            data: configs,
            type: API_CALL_ALL
        };
        return communicate(message)
            .then((response) => {
            return Promise.resolve(response);
        })
            .catch((error) => {
            return Promise.reject(error);
        });
    };
    /**
     * Initializes the object with authentication parameters.
     *
     * @param {ConfigInterface} config The configuration object.
     *
     * @returns {Promise<boolean>} Promise that resolves when initialization is successful.
     *
     * The `config` object has the following attributes:
     * ```
     *  var config = {
     *    authorizationType?: string //optional
     *    clientHost: string
     *    clientID: string
     *    clientSecret?: string //optional
     *    consentDenied?: boolean //optional
     *    enablePKCE?: boolean //optional
     *    prompt?: string //optional
     *    responseMode?: "query" | "form-post" //optional
     *    scope?: string[] //optional
     *    serverOrigin: string
     *    tenant?: string //optional
     *    tenantPath?: string //optional
     *    baseUrls: string[]
     *    callbackURL: string
     *    requestTimeout: number //optional
     *  }
     * ```
     */
    const initialize = (config) => {
        var _a;
        if (config.authorizationType && typeof config.authorizationType !== "string") {
            return Promise.reject("The authorizationType must be a string");
        }
        if (!(config.baseUrls instanceof Array)) {
            return Promise.reject("baseUrls must be an array");
        }
        if (config.baseUrls.find((baseUrl) => typeof baseUrl !== "string")) {
            return Promise.reject("Array elements of baseUrls must all be string values");
        }
        if (typeof config.signInRedirectURL !== "string") {
            return Promise.reject("The sign-in redirect URL must be a string");
        }
        if (typeof config.signOutRedirectURL !== "string") {
            return Promise.reject("The sign-out redirect URL must be a string");
        }
        if (typeof config.clientHost !== "string") {
            return Promise.reject("The clientHost must be a string");
        }
        if (typeof config.clientID !== "string") {
            return Promise.reject("The clientID must be a string");
        }
        if (config.clientSecret && typeof config.clientSecret !== "string") {
            return Promise.reject("The clientString must be a string");
        }
        if (config.consentDenied && typeof config.consentDenied !== "boolean") {
            return Promise.reject("consentDenied must be a boolean");
        }
        if (config.enablePKCE && typeof config.enablePKCE !== "boolean") {
            return Promise.reject("enablePKCE must be a boolean");
        }
        if (config.prompt && typeof config.prompt !== "string") {
            return Promise.reject("The prompt must be a string");
        }
        if (config.responseMode && typeof config.responseMode !== "string") {
            return Promise.reject("The responseMode must be a string");
        }
        if (config.responseMode
            && config.responseMode !== ResponseMode.formPost
            && config.responseMode !== ResponseMode.query) {
            return Promise.reject("The responseMode is invalid");
        }
        if (config.scope && !(config.scope instanceof Array)) {
            return Promise.reject("scope must be an array");
        }
        if (config.scope && config.scope.find((aScope) => typeof aScope !== "string")) {
            return Promise.reject("Array elements of scope must all be string values");
        }
        if (typeof config.serverOrigin !== "string") {
            return Promise.reject("serverOrigin must be a string");
        }
        httpClientHandlers = {
            isHandlerEnabled: true,
            requestErrorCallback: null,
            requestFinishCallback: null,
            requestStartCallback: null,
            requestSuccessCallback: null
        };
        requestTimeout = (_a = config === null || config === void 0 ? void 0 : config.requestTimeout) !== null && _a !== void 0 ? _a : 60000;
        worker.onmessage = ({ data }) => {
            var _a, _b;
            switch (data.type) {
                case REQUEST_ERROR:
                    (httpClientHandlers === null || httpClientHandlers === void 0 ? void 0 : httpClientHandlers.requestErrorCallback) && (httpClientHandlers === null || httpClientHandlers === void 0 ? void 0 : httpClientHandlers.requestErrorCallback(JSON.parse((_a = data.data) !== null && _a !== void 0 ? _a : "")));
                    break;
                case REQUEST_FINISH:
                    (httpClientHandlers === null || httpClientHandlers === void 0 ? void 0 : httpClientHandlers.requestFinishCallback) && (httpClientHandlers === null || httpClientHandlers === void 0 ? void 0 : httpClientHandlers.requestFinishCallback());
                    break;
                case REQUEST_START:
                    (httpClientHandlers === null || httpClientHandlers === void 0 ? void 0 : httpClientHandlers.requestStartCallback) && (httpClientHandlers === null || httpClientHandlers === void 0 ? void 0 : httpClientHandlers.requestStartCallback());
                    break;
                case REQUEST_SUCCESS:
                    (httpClientHandlers === null || httpClientHandlers === void 0 ? void 0 : httpClientHandlers.requestSuccessCallback) && (httpClientHandlers === null || httpClientHandlers === void 0 ? void 0 : httpClientHandlers.requestSuccessCallback(JSON.parse((_b = data.data) !== null && _b !== void 0 ? _b : "")));
                    break;
            }
        };
        const message = {
            data: config,
            type: INIT
        };
        return communicate(message)
            .then(() => {
            initialized = true;
            return Promise.resolve(true);
        })
            .catch((error) => {
            return Promise.reject(error);
        });
    };
    /**
     * @private
     *
     * Sends the authorization code and authenticates the user.
     *
     * @returns {Promise<UserInfo>} Promise that resolves on successful authentication.
     */
    const sendAuthorizationCode = () => {
        const authCode = getAuthorizationCode();
        const sessionState = new URL(window.location.href).searchParams.get(SESSION_STATE);
        const message = {
            data: {
                code: authCode,
                pkce: sessionStorage.getItem(PKCE_CODE_VERIFIER),
                sessionState: sessionState
            },
            type: SIGN_IN
        };
        history.pushState({}, document.title, removeAuthorizationCode());
        sessionStorage.removeItem(PKCE_CODE_VERIFIER);
        sessionStorage.removeItem(AUTHORIZATION_CODE);
        return communicate(message)
            .then((response) => {
            if (response.type === SIGNED_IN) {
                signedIn = true;
                sessionStorage.setItem(LOGOUT_URL, response.data.logoutUrl);
                const data = response.data;
                delete data.logoutUrl;
                return Promise.resolve(data);
            }
            else if (response.type === AUTH_REQUIRED && response.code) {
                if (response.pkce) {
                    sessionStorage.setItem(PKCE_CODE_VERIFIER, response.pkce);
                }
                location.href = response.code;
                return Promise.resolve({
                    allowedScopes: "",
                    displayName: "",
                    email: "",
                    sessionState: "",
                    tenantDomain: "",
                    username: ""
                });
            }
            else if (response.type === AUTH_REQUIRED && !response.code) {
                return Promise.reject("Something went wrong during authentication after obtaining the authorization code." +
                    " Re-authentication failed. No authorization url was received." +
                    JSON.stringify(response));
            }
            return Promise.reject("Something went wrong during authentication. " +
                "Failed during signing in after getting the authorization code.");
        })
            .catch((error) => {
            return Promise.reject(error);
        });
    };
    /**
     * Initiates the authentication flow.
     *
     * @returns {Promise<UserInfo>} A promise that resolves when authentication is successful.
     */
    const signIn = (fidp) => {
        if (initialized) {
            if (hasAuthorizationCode() || sessionStorage.getItem(PKCE_CODE_VERIFIER)) {
                return sendAuthorizationCode();
            }
            else {
                const message = {
                    data: fidp,
                    type: SIGN_IN
                };
                return communicate(message)
                    .then((response) => {
                    if (response.type === SIGNED_IN) {
                        signedIn = true;
                        sessionStorage.setItem(LOGOUT_URL, response.data.logoutUrl);
                        const data = response.data;
                        delete data.logoutUrl;
                        return Promise.resolve(data);
                    }
                    else if (response.type === AUTH_REQUIRED && response.code) {
                        if (response.pkce) {
                            sessionStorage.setItem(PKCE_CODE_VERIFIER, response.pkce);
                        }
                        location.href = response.code;
                        return Promise.resolve({
                            allowedScopes: "",
                            displayName: "",
                            email: "",
                            sessionState: "",
                            tenantDomain: "",
                            username: ""
                        });
                    }
                    else {
                        if (response.type === AUTH_REQUIRED && !response.code) {
                            return Promise.reject("Something went wrong during authentication." +
                                " No authorization url was received." +
                                JSON.stringify(response));
                        }
                        else {
                            return Promise.reject("Something went wrong during authentication." +
                                "Unknown response received. " +
                                JSON.stringify(response));
                        }
                    }
                })
                    .catch((error) => {
                    return Promise.reject(error);
                });
            }
        }
        else {
            return Promise.reject("Error while signing in. The object has not been initialized yet.");
        }
    };
    /**
     * Initiates the sign out flow.
     *
     * @returns {Promise<boolean>} A promise that resolves when sign out is completed.
     */
    const signOut = () => {
        if (sessionStorage.getItem(LOGOUT_URL)) {
            const logoutUrl = sessionStorage.getItem(LOGOUT_URL);
            sessionStorage.removeItem(LOGOUT_URL);
            window.location.href = logoutUrl;
            return Promise.resolve(true);
        }
        else {
            return Promise.reject("No login sessions.");
        }
    };
    /**
     * Revokes token.
     *
     * @returns {Promise<boolean>} A promise that resolves when revoking is completed.
     */
    const endUserSession = () => {
        if (!signedIn) {
            return Promise.reject("You have not signed in yet");
        }
        const message = {
            type: END_USER_SESSION
        };
        return communicate(message)
            .then((response) => {
            return Promise.resolve(response);
        })
            .catch((error) => {
            return Promise.reject(error);
        });
    };
    const getServiceEndpoints = () => {
        const message = {
            type: GET_SERVICE_ENDPOINTS
        };
        return communicate(message)
            .then((response) => {
            return Promise.resolve(response);
        })
            .catch((error) => {
            return Promise.reject(error);
        });
    };
    const getUserInfo = () => {
        const message = {
            type: GET_USER_INFO
        };
        return communicate(message)
            .then((response) => {
            return Promise.resolve(response);
        })
            .catch((error) => {
            return Promise.reject(error);
        });
    };
    const getDecodedIDToken = () => {
        const message = {
            type: GET_DECODED_ID_TOKEN
        };
        return communicate(message)
            .then((response) => {
            return Promise.resolve(response);
        })
            .catch((error) => {
            return Promise.reject(error);
        });
    };
    const onHttpRequestSuccess = (callback) => {
        if (callback && typeof callback === "function") {
            httpClientHandlers.requestSuccessCallback = callback;
        }
    };
    const onHttpRequestError = (callback) => {
        if (callback && typeof callback === "function") {
            httpClientHandlers.requestErrorCallback = callback;
        }
    };
    const onHttpRequestStart = (callback) => {
        if (callback && typeof callback === "function") {
            httpClientHandlers.requestStartCallback = callback;
        }
    };
    const onHttpRequestFinish = (callback) => {
        if (callback && typeof callback === "function") {
            httpClientHandlers.requestFinishCallback = callback;
        }
    };
    /**
     * @constructor
     *
     * This returns the object containing the public methods.
     *
     * @returns {WebWorkerClientInterface} OAuthInterface object
     */
    function Constructor() {
        worker = new WorkerFile();
        return {
            customGrant,
            endUserSession,
            getDecodedIDToken,
            getServiceEndpoints,
            getUserInfo,
            httpRequest,
            httpRequestAll,
            initialize,
            onHttpRequestError,
            onHttpRequestFinish,
            onHttpRequestStart,
            onHttpRequestSuccess,
            signIn,
            signOut
        };
    }
    return {
        getInstance: () => {
            if (instance) {
                return instance;
            }
            else {
                instance = Constructor();
                return instance;
            }
        }
    };
})();
//# sourceMappingURL=web-worker-client.js.map