/**
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the 'License'); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * 'AS IS' BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { saveAs } from "file-saver";
import * as forge from "node-forge";
import { CertificateManagementConstants } from "../constants";
/**
 * Utility class for certificate management operations.
 */
export class CertificateManagementUtils {
    /**
     * Private constructor to avoid object instantiation from outside
     * the class.
     *
     * @hideconstructor
     */
    // eslint-disable-next-line @typescript-eslint/no-empty-function
    constructor() { }
    /**
     * Converts a PEM encoded string to a Forge certificate object.
     *
     * @param {string} pem The PEM encoded certificate content.
     *
     * @returns {forge.pki.Certificate} The Forge Certificate object.
     */
    static decodeCertificate(pem) {
        // Check if the certificate content is base 64 encoded.
        const base64DecodedCert = this.getBase64DecodedCertificate(pem);
        if (base64DecodedCert !== null) {
            if (base64DecodedCert.startsWith(CertificateManagementConstants.CERTIFICATE_BEGIN)) {
                return forge.pki.certificateFromPem(base64DecodedCert);
            }
        }
        const pemValue = pem === null || pem === void 0 ? void 0 : pem.split("\n");
        // appends -----END CERTIFICATE-----.
        pemValue === null || pemValue === void 0 ? void 0 : pemValue.push(CertificateManagementConstants.CERTIFICATE_END);
        // appends a new line.
        pemValue === null || pemValue === void 0 ? void 0 : pemValue.push(CertificateManagementConstants.END_LINE);
        // pushes -----BEGIN CERTIFICATE----- to the top.
        pemValue === null || pemValue === void 0 ? void 0 : pemValue.unshift(CertificateManagementConstants.CERTIFICATE_BEGIN);
        const pemCert = pemValue === null || pemValue === void 0 ? void 0 : pemValue.join("\n");
        try {
            return forge.pki.certificateFromPem(pemCert);
        }
        catch (e) {
            return null;
        }
    }
    /**
     * This serializes the certificate content to a displayable format.
     *
     * @param {Certificate} certificate The Certificate object returned by teh API endpoints.
     * @param {string} pem The PEM encoded certificate content.
     * @return {DisplayCertificate}
     */
    static displayCertificate(certificate = null, pem) {
        const certificateForge = CertificateManagementUtils.decodeCertificate(pem);
        if (certificateForge) {
            return {
                alias: certificate ? certificate.alias : null,
                issuerDN: certificateForge.issuer.attributes
                    .map(attribute => {
                    return {
                        [attribute.shortName]: attribute.value
                    };
                }),
                serialNumber: certificateForge.serialNumber,
                subjectDN: certificateForge.subject.attributes
                    .map(attribute => {
                    return {
                        [attribute.shortName]: attribute.value
                    };
                }),
                validFrom: certificateForge.validity.notBefore,
                validTill: certificateForge.validity.notAfter,
                version: certificateForge.version
            };
        }
        return null;
    }
    /**
     * This serializes the certificate object.
     *
     * @param {Certificate} data The data object containing the alias and the PEM string.
     * @param {forge.pki.Certificate} certificateForge The Forge Certificate object.
     */
    static decodeForgeCertificate(data, certificateForge) {
        return {
            alias: data.alias,
            issuerDN: certificateForge.issuer.attributes
                .map(attribute => {
                return {
                    [attribute.shortName]: attribute.value
                };
            }),
            serialNumber: certificateForge.serialNumber,
            subjectDN: certificateForge.subject.attributes
                .map(attribute => {
                return {
                    [attribute.shortName]: attribute.value
                };
            }),
            validFrom: certificateForge.validity.notBefore,
            validTill: certificateForge.validity.notAfter,
            version: certificateForge.version
        };
    }
    /**
     * The following function search the issuerDN array and return a alias for the issuer.
     *
     * @param issuerDN
     */
    static searchIssuerDNAlias(issuerDN) {
        let issuerAlias = "";
        issuerDN.map((issuer) => {
            if (Object.prototype.hasOwnProperty.call(issuer, "CN")) {
                issuerAlias = issuer["CN"];
                return;
            }
            else if (Object.prototype.hasOwnProperty.call(issuer, "O")) {
                issuerAlias = issuer["O"];
                return;
            }
        });
        return issuerAlias;
    }
    /**
     * This converts a PEM-encoded certificate to a
     * DER encoded ASN.1 certificate and saves it to the disk.
     *
     * ```
     * const intArray = der.data.split("").map(char => {
            return char.charCodeAt(0);
        });
     * ```
     * The `ByteStringBuffer` that holds the DER encoded
     * string actually has `UTF-16` encoded string values.
     *
     * The above code snippet is used to decode the `UTF-16` string.
     *
     * @param {string} name The alias of the certificate.
     * @param {string} pem The PEM encoded certificate content.
     */
    static exportCertificate(name, pem) {
        const certificate = CertificateManagementUtils.decodeCertificate(pem);
        const der = forge.asn1.toDer(forge.pki.certificateToAsn1(certificate));
        const intArray = der.data.split("").map(char => {
            return char.charCodeAt(0);
        });
        const buffer = new Uint8Array(intArray).buffer;
        const blob = new Blob([buffer], {
            type: "application/x-x509-ca-cert"
        });
        saveAs(blob, name + ".cer");
    }
    /**
     * This strips **BEGIN CERTIFICATE** and **END CERTIFICATE** parts from
     * the PEM encoded string.
     *
     * @param {string} pemString The PEM-encoded content of a certificate.
     *
     * @returns {string} The PEM string without the **BEGIN CERTIFICATE** and **END CERTIFICATE** parts.
     */
    static stripPem(pemString) {
        var _a, _b;
        const pemValue = pemString.split("\n");
        // removes -----BEGIN CERTIFICATE----- if present.
        ((_a = pemValue[0]) === null || _a === void 0 ? void 0 : _a.includes(CertificateManagementConstants.CERTIFICATE_BEGIN)) && pemValue.shift();
        // removes "\n" if present.
        pemValue[pemValue.length - 1] === CertificateManagementConstants.END_LINE
            && pemValue.pop();
        // removes -----END CERTIFICATE----- if present.
        ((_b = pemValue[pemValue.length - 1]) === null || _b === void 0 ? void 0 : _b.includes(CertificateManagementConstants.CERTIFICATE_END))
            && pemValue.pop();
        return pemValue.join("\n");
    }
    ;
    /**
     * This encloses a stripped PEM string with **BEGIN CERTIFICATE** and **END CERTIFICATE**.
     *
     * @param {string} pemString The stripped PEM string (usually received as from an API call)
     *
     * @returns {string} A full PEM string.
     */
    static enclosePem(pemString) {
        var _a, _b;
        const pemValue = pemString.split("\n");
        // adds -----BEGIN CERTIFICATE----- if not present.
        !((_a = pemValue[0]) === null || _a === void 0 ? void 0 : _a.includes(CertificateManagementConstants.CERTIFICATE_BEGIN))
            && pemValue.unshift(CertificateManagementConstants.CERTIFICATE_BEGIN);
        // adds "\n" if not present.
        !(pemValue[pemValue.length - 1] === CertificateManagementConstants.END_LINE)
            && pemValue.push(CertificateManagementConstants.END_LINE);
        // adds -----END CERTIFICATE----- if not present.
        if (!((_b = pemValue[pemValue.length - 2]) === null || _b === void 0 ? void 0 : _b.includes(CertificateManagementConstants.CERTIFICATE_END))) {
            const lastLine = pemValue.pop();
            pemValue.push(CertificateManagementConstants.CERTIFICATE_END);
            pemValue.push(lastLine);
        }
        return pemValue.join("\n");
    }
    ;
    /**
     * Base 64 decodes the certificate content. If not a valid base 64 content, returns null.
     *
     * @param {string} certContent to be base 64 decoded.
     */
    static getBase64DecodedCertificate(certContent) {
        try {
            return atob(certContent);
        }
        catch (e) {
            return null;
        }
    }
}
//# sourceMappingURL=certificate-management-utils.js.map