/**
 * Copyright (c) 2021, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the 'License'); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * 'AS IS' BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { FC } from "react";
import { Icon, SemanticICONS } from "semantic-ui-react";
export interface FilePickerProps {
    /**
     * The target strategy of this file picker. This
     * can take many forms depending on the use-case.
     */
    fileStrategy: PickerStrategy<any>;
    /**
     * The delegated event handler for the parent
     * component. This is called when each time file
     * drag & drop, file input, or content changes.
     * This has no debounce time and will be called
     * immediately.
     *
     * @param result {PickerResult} data.
     */
    onChange: (result: PickerResult<any>) => void;
    /**
     * In the dropzone we can explain what types of file
     * and some descriptive info about the required file.
     */
    dropzoneText?: string;
    /**
     * This is the placeholder text for paste tab content
     * area. By default this has a value like the following:-
     * "Paste your content in this area..."
     */
    pasteAreaPlaceholderText?: string;
    /**
     * The manual upload button text. This button is
     * placed beneath the description.
     */
    uploadButtonText?: string;
    /**
     * Please refer {@link GenericIcon} for specific sub
     * attributes. Usually it can be SVG, Semantic UI icon
     * class, or a <Icon> component. The {@link icon} and
     * {@link placeholderIcon} are two different icons.
     *
     * 1) {@link icon} will render once a file has been selected
     *    or dragged on to the area.
     *
     * 2) {@link placeholderIcon} will render when the placeholder
     *    area is active. Usually this can be a plain icon.
     */
    icon?: SemanticICONS | Icon | SVGElement | string | any;
    placeholderIcon?: SemanticICONS | Icon | SVGElement | string | any;
    /**
     * A prop that accepts a file by default. This file will
     * be the initial state for the picker. Behaviour is same
     * as {@link pastedContent}.
     */
    file?: File | null;
    /**
     * A prop that accepts a raw string by default. The raw value
     * should be bind to the form and the api consumer can specify
     * whether to fire the initial values via {@link onChange}
     * callback.
     */
    pastedContent?: string | null;
    /**
     * By enabling this you can always guarantee that a field value
     * remove operation (either file or pasted content) will provide
     * the subsequent serialized value of the next available value
     * or none at all.
     *
     * Say for example: A user can upload a certificate or paste the PEM
     * string in the textarea. First, user will switch to "Paste" tab
     * and paste the value. Now component will propagate the change
     * to parent with {@link PickerResult}. However, then the user switch
     * to "Upload" tab and uploads a different file. Once again component
     * will propagate those change to parent. However, if the user remove
     * a certain input value we need a way to show what input is being
     * used in the form.
     *
     * This property ensures the value that gets saved to the state,
     * propagate, and also visually represent in the form which input
     * is being used.
     */
    normalizeStateOnRemoveOperations?: boolean;
    /**
     * Hide selection taps & paste section.
     */
    hidePasteOption?: boolean;
    /**
     * Trigger empty file error.
     */
    emptyFileError?: boolean;
    /**
     * Empty file message.
     */
    emptyFileErrorMsg?: string;
}
export declare const FilePicker: FC<FilePickerProps>;
/**
 * Re-usable and pluggable file strategies for the component. The context of these
 * different strategies is the FilePicker component itself. It makes the FilePicker
 * dynamic on handling the file specific logic. You can implement different
 * strategies for each use-case or use the available ones.
 */
export interface PickerResult<T> {
    file?: File;
    pastedContent?: string;
    serialized?: T | any;
    valid?: boolean;
}
export interface PickerStrategy<T> {
    /**
     * Should contain the logic to validate a attached file or
     * the pasted text into the text area. For example, you can
     * validate the type, size, name, and the expected content
     * if applicable.
     *
     * @param data {File | string} uploaded file or the pasted text.
     */
    validate: (data: File | string) => Promise<ValidationResult>;
    /**
     * The associated mime-types for the target file.
     * This will be used in the {@link HTMLInputElement.accept}
     * attribute to specify the allowed ones in the interface.
     *
     * file_extension   [.gif, .jpg, .png, .doc .pem, .cert]
     * audio/*          All audio file types.
     * video/*          All video file types.
     * image/*          All image file types.
     *
     * Or just specify the exact mime type:
     * {@link https://www.iana.org/assignments/media-types/media-types.xhtml}
     */
    mimeTypes: string[];
    /**
     * Serialize the uploaded file / pasted text to a more primitive
     * representation. You can return an empty string if there's
     * no post-processing logic to your use-case.
     *
     * @param data {File | string} uploaded file or pasted text.
     */
    serialize: (data: File | string) => Promise<T>;
}
interface ValidationResult {
    valid: boolean;
    errorMessage?: string | undefined | null;
}
export declare class DefaultFileStrategy implements PickerStrategy<string> {
    mimeTypes: string[];
    constructor();
    serialize(data: File | string): Promise<string>;
    validate(data: File | string): Promise<ValidationResult>;
}
export declare class XMLFileStrategy implements PickerStrategy<string> {
    static readonly ENCODING: string;
    static readonly DEFAULT_MIMES: string[];
    static readonly MEGABYTE: number;
    static readonly MAX_FILE_SIZE: number;
    mimeTypes: string[];
    constructor(mimeTypes?: string[]);
    serialize(data: File | string): Promise<string>;
    validate(data: File | string): Promise<ValidationResult>;
    parseXML(xml: string | ArrayBuffer): Promise<string>;
}
export interface CertificateDecodeResult {
    forgeObject: any;
    pem: string;
    pemStripped: string;
}
export declare class CertFileStrategy implements PickerStrategy<CertificateDecodeResult> {
    static readonly DEFAULT_MIMES: string[];
    mimeTypes: string[];
    constructor(mimeTypes?: string[]);
    serialize(data: File | string): Promise<CertificateDecodeResult>;
    validate(data: File | string): Promise<ValidationResult>;
    convertFromString(text: string): Promise<CertificateDecodeResult>;
    convertFromFile(file: File): Promise<CertificateDecodeResult>;
}
export {};
